#!/usr/bin/env python3
"""
Test script for the Report Generator module
Tests basic functionality and generates sample reports
"""
import sys
import os
import tempfile
import logging
from datetime import datetime, timedelta

# Add the project root to the Python path
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))

def setup_test_environment():
    """Set up test environment"""
    logging.basicConfig(
        level=logging.INFO,
        format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
    )
    logger = logging.getLogger(__name__)
    return logger

def test_basic_imports():
    """Test that all modules import correctly"""
    logger = logging.getLogger(__name__)
    
    try:
        from report_generator import (
            OrchidReportGenerator, ReportConfig, ReportType, OutputFormat,
            generate_data_summary_report, generate_collection_analysis_report
        )
        logger.info("✅ All imports successful")
        return True
    except ImportError as e:
        logger.error(f"❌ Import failed: {e}")
        return False

def test_config_creation():
    """Test ReportConfig creation with various options"""
    logger = logging.getLogger(__name__)
    
    try:
        from report_generator import ReportConfig, ReportType, OutputFormat
        
        # Test basic config
        config = ReportConfig(
            report_type=ReportType.DATA_SUMMARY,
            output_formats=[OutputFormat.PDF, OutputFormat.CSV],
            title="Test Report",
            subtitle="Generated by test script",
            author="Test System"
        )
        
        logger.info(f"✅ Config created: {config.title}")
        logger.info(f"   Report Type: {config.report_type.value}")
        logger.info(f"   Formats: {[f.value for f in config.output_formats]}")
        logger.info(f"   Color Scheme: {config.color_scheme}")
        
        return True
    except Exception as e:
        logger.error(f"❌ Config creation failed: {e}")
        return False

def test_mock_data_generation():
    """Test report generation with mock data (without database)"""
    logger = logging.getLogger(__name__)
    
    try:
        from report_generator import OrchidReportGenerator, ReportConfig, ReportType, OutputFormat
        import tempfile
        
        # Create generator
        generator = OrchidReportGenerator()
        
        # Create test config
        config = ReportConfig(
            report_type=ReportType.DATA_SUMMARY,
            output_formats=[OutputFormat.JSON],  # Start with JSON as it's simplest
            title="Test Data Summary Report",
            subtitle="Generated for testing purposes",
            author="Test System",
            include_charts=False,  # Skip charts for basic test
            include_images=False
        )
        
        # Mock the data gathering method for testing
        def mock_gather_data(config):
            return {
                'records': [],
                'summary': {
                    'total_orchids': 1000,
                    'total_users': 25,
                    'recent_orchids': 50,
                    'ai_analyzed_count': 750,
                    'avg_ai_confidence': 0.85
                },
                'analytics': {
                    'genus_distribution': [
                        {'genus': 'Cattleya', 'count': 150},
                        {'genus': 'Dendrobium', 'count': 120},
                        {'genus': 'Phalaenopsis', 'count': 100}
                    ],
                    'country_distribution': [
                        {'country': 'Thailand', 'count': 200},
                        {'country': 'Philippines', 'count': 150}
                    ]
                }
            }
        
        # Replace the data gathering method temporarily
        original_method = generator._gather_report_data
        generator._gather_report_data = mock_gather_data
        
        logger.info("🚀 Testing report generation with mock data...")
        
        # Generate report
        result = generator.generate_report(config)
        
        # Restore original method
        generator._gather_report_data = original_method
        
        if result['status'] == 'success':
            logger.info("✅ Mock report generation successful!")
            logger.info(f"   Report ID: {result['report_id']}")
            logger.info(f"   Files generated: {list(result['files'].keys())}")
            
            # Check if JSON file exists and is readable
            if 'json' in result['files']:
                json_path = result['files']['json']
                if os.path.exists(json_path):
                    file_size = os.path.getsize(json_path)
                    logger.info(f"   JSON file size: {file_size} bytes")
                    
                    # Try to read the JSON content
                    import json
                    with open(json_path, 'r') as f:
                        data = json.load(f)
                    logger.info(f"   JSON structure: {list(data.keys())}")
                else:
                    logger.warning("⚠️ JSON file not found")
            
            return True
        else:
            logger.error(f"❌ Report generation failed: {result.get('error', 'Unknown error')}")
            return False
            
    except Exception as e:
        logger.error(f"❌ Mock data generation test failed: {e}")
        import traceback
        logger.error(traceback.format_exc())
        return False

def test_visualization_components():
    """Test visualization components separately"""
    logger = logging.getLogger(__name__)
    
    try:
        import matplotlib
        matplotlib.use('Agg')  # Use non-interactive backend
        import matplotlib.pyplot as plt
        import seaborn as sns
        import numpy as np
        
        logger.info("✅ Visualization libraries imported successfully")
        
        # Test basic chart generation
        fig, ax = plt.subplots(figsize=(8, 6))
        
        # Create sample data
        data = {
            'Cattleya': 150,
            'Dendrobium': 120,
            'Phalaenopsis': 100,
            'Oncidium': 80,
            'Vanda': 60
        }
        
        # Create bar chart
        bars = ax.bar(data.keys(), data.values(), color=sns.color_palette("husl", len(data)))
        ax.set_title('Test Genus Distribution', fontsize=14, fontweight='bold')
        ax.set_xlabel('Genus')
        ax.set_ylabel('Count')
        
        # Add value labels
        for bar in bars:
            height = bar.get_height()
            ax.text(bar.get_x() + bar.get_width()/2., height,
                   f'{int(height)}', ha='center', va='bottom')
        
        plt.xticks(rotation=45)
        plt.tight_layout()
        
        # Save test chart
        with tempfile.NamedTemporaryFile(suffix='.png', delete=False) as tmp:
            plt.savefig(tmp.name, dpi=300, bbox_inches='tight')
            plt.close()
            
            if os.path.exists(tmp.name):
                file_size = os.path.getsize(tmp.name)
                logger.info(f"✅ Test chart generated: {file_size} bytes")
                os.unlink(tmp.name)  # Clean up
                return True
            else:
                logger.error("❌ Chart file not created")
                return False
        
    except Exception as e:
        logger.error(f"❌ Visualization test failed: {e}")
        import traceback
        logger.error(traceback.format_exc())
        return False

def test_pdf_components():
    """Test PDF generation components"""
    logger = logging.getLogger(__name__)
    
    try:
        from reportlab.lib.pagesizes import A4
        from reportlab.platypus import SimpleDocTemplate, Paragraph, Spacer
        from reportlab.lib.styles import getSampleStyleSheet
        from reportlab.lib.colors import Color
        import tempfile
        
        logger.info("✅ ReportLab imported successfully")
        
        # Create a simple test PDF
        with tempfile.NamedTemporaryFile(suffix='.pdf', delete=False) as tmp:
            doc = SimpleDocTemplate(tmp.name, pagesize=A4)
            
            # Get styles
            styles = getSampleStyleSheet()
            
            # Create content
            story = []
            story.append(Paragraph("Test PDF Report", styles['Title']))
            story.append(Spacer(1, 20))
            story.append(Paragraph("This is a test PDF generated by the Report Generator system.", styles['Normal']))
            story.append(Spacer(1, 20))
            story.append(Paragraph("Key Features:", styles['Heading2']))
            story.append(Paragraph("• Professional PDF formatting", styles['Normal']))
            story.append(Paragraph("• Chart integration capability", styles['Normal']))
            story.append(Paragraph("• Custom styling and colors", styles['Normal']))
            
            # Build PDF
            doc.build(story)
            
            if os.path.exists(tmp.name):
                file_size = os.path.getsize(tmp.name)
                logger.info(f"✅ Test PDF generated: {file_size} bytes")
                os.unlink(tmp.name)  # Clean up
                return True
            else:
                logger.error("❌ PDF file not created")
                return False
        
    except Exception as e:
        logger.error(f"❌ PDF test failed: {e}")
        import traceback
        logger.error(traceback.format_exc())
        return False

def test_excel_components():
    """Test Excel generation components"""
    logger = logging.getLogger(__name__)
    
    try:
        import openpyxl
        from openpyxl.styles import Font, PatternFill
        import tempfile
        
        logger.info("✅ OpenPyXL imported successfully")
        
        # Create test workbook
        wb = openpyxl.Workbook()
        ws = wb.active
        ws.title = "Test Report"
        
        # Add headers
        headers = ['ID', 'Genus', 'Species', 'Count', 'Percentage']
        for col, header in enumerate(headers, 1):
            cell = ws.cell(row=1, column=col, value=header)
            cell.font = Font(bold=True, color='FFFFFF')
            cell.fill = PatternFill(start_color='366092', end_color='366092', fill_type='solid')
        
        # Add test data
        test_data = [
            [1, 'Cattleya', 'labiata', 50, 25.0],
            [2, 'Dendrobium', 'nobile', 40, 20.0],
            [3, 'Phalaenopsis', 'amabilis', 60, 30.0]
        ]
        
        for row_idx, row_data in enumerate(test_data, 2):
            for col_idx, value in enumerate(row_data, 1):
                ws.cell(row=row_idx, column=col_idx, value=value)
        
        # Save test file
        with tempfile.NamedTemporaryFile(suffix='.xlsx', delete=False) as tmp:
            wb.save(tmp.name)
            
            if os.path.exists(tmp.name):
                file_size = os.path.getsize(tmp.name)
                logger.info(f"✅ Test Excel file generated: {file_size} bytes")
                os.unlink(tmp.name)  # Clean up
                return True
            else:
                logger.error("❌ Excel file not created")
                return False
        
    except Exception as e:
        logger.error(f"❌ Excel test failed: {e}")
        import traceback
        logger.error(traceback.format_exc())
        return False

def run_comprehensive_tests():
    """Run all tests and report results"""
    logger = setup_test_environment()
    
    logger.info("🚀 Starting comprehensive Report Generator tests...")
    
    tests = [
        ("Basic Imports", test_basic_imports),
        ("Config Creation", test_config_creation),
        ("Visualization Components", test_visualization_components),
        ("PDF Components", test_pdf_components),
        ("Excel Components", test_excel_components),
        ("Mock Data Generation", test_mock_data_generation)
    ]
    
    results = {}
    
    for test_name, test_func in tests:
        logger.info(f"\n📋 Running test: {test_name}")
        try:
            success = test_func()
            results[test_name] = success
            if success:
                logger.info(f"✅ {test_name}: PASSED")
            else:
                logger.error(f"❌ {test_name}: FAILED")
        except Exception as e:
            logger.error(f"💥 {test_name}: ERROR - {e}")
            results[test_name] = False
    
    # Print summary
    logger.info("\n" + "="*60)
    logger.info("📊 TEST SUMMARY")
    logger.info("="*60)
    
    passed = sum(results.values())
    total = len(results)
    
    for test_name, success in results.items():
        status = "✅ PASSED" if success else "❌ FAILED"
        logger.info(f"{test_name:30} {status}")
    
    logger.info("-"*60)
    logger.info(f"TOTAL: {passed}/{total} tests passed ({passed/total*100:.1f}%)")
    
    if passed == total:
        logger.info("🎉 All tests passed! Report Generator is ready for use.")
    elif passed > total * 0.5:
        logger.info("⚠️ Most tests passed. Some components may need attention.")
    else:
        logger.error("🚨 Multiple test failures. System needs debugging.")
    
    return passed, total

if __name__ == "__main__":
    print("Orchid Continuum Report Generator Test Suite")
    print("=" * 50)
    
    try:
        passed, total = run_comprehensive_tests()
        
        print("\n" + "=" * 50)
        if passed == total:
            print("🎉 SUCCESS: All components working correctly!")
            sys.exit(0)
        else:
            print(f"⚠️ PARTIAL: {passed}/{total} components working")
            sys.exit(1)
            
    except KeyboardInterrupt:
        print("\n⚡ Tests interrupted by user")
        sys.exit(130)
    except Exception as e:
        print(f"\n💥 Critical error during testing: {e}")
        import traceback
        traceback.print_exc()
        sys.exit(1)