from flask import render_template, request, jsonify, flash, redirect, url_for, send_file, session, Response, abort, make_response
from flask_login import login_required, current_user
from werkzeug.utils import secure_filename
from app import app, db
from models import (OrchidRecord, OrchidTaxonomy, UserUpload, ScrapingLog, WidgetConfig, 
                   User, JudgingAnalysis, Certificate, BatchUpload, UserFeedback, WeatherData, UserLocation, WeatherAlert, WorkshopRegistration, BugReport,
                   MemberCollection, ExternalDatabaseCache, ResearchCollaboration, LiteratureCitation, Pollinator, AdvancedOrchidPollinatorRelationship)
from eol_integration import EOLIntegrator
from external_databases.gbif_integration import GBIFIntegrator
from orchid_nursery_directory import get_nursery_directory
from orchid_society_directory import get_society_directory
from image_recovery_system import get_image_with_recovery, get_image_recovery_stats
from photo_failsafe_system import get_photos_guaranteed
from orchid_ai import analyze_orchid_image, get_weather_based_care_advice, extract_metadata_from_text
from web_scraper import scrape_gary_yong_gee, scrape_roberta_fox
from google_drive_service import upload_to_drive, get_drive_file_url
from utils import allowed_file, generate_filename, get_orchid_of_the_day
from batch_upload import BatchUploadProcessor, validate_batch_limits
from judging_standards import analyze_orchid_by_organization, get_available_organizations
from enhanced_judging import analyze_orchid_with_genetics
# from genetic_analysis import analyze_orchid_genetics, compare_hybrid_to_parents  # Temporarily disabled
from enhanced_metadata_analyzer import analyze_orchid_with_botanical_databases
from rhs_integration import get_rhs_orchid_data, analyze_hybrid_parentage
from export_utils import export_orchid_data, get_export_filename
from certificate_generator import generate_award_certificate, get_certificate_pdf
from filename_parser import parse_orchid_filename
from processing_routes import processing_bp
from photo_editor_routes import photo_editor_bp
from weather_service import WeatherService, get_coordinates_from_location
from orchid_atlas import atlas_bp
from darwin_core_exporter import DarwinCoreExporter
from weather_habitat_routes import register_weather_habitat_routes
from scraping_dashboard import scraping_dashboard
try:
    from vigilant_monitor import vigilant_monitor
except ImportError:
    vigilant_monitor = None
from gbif_routes import gbif_bp
from ai_orchid_routes import ai_orchid_bp
from geographic_mapping_routes import geo_mapping_bp
from enhanced_mapping_routes import enhanced_mapping_bp
from admin_orchid_approval import orchid_approval_bp
from pattern_analysis_routes import pattern_analysis_bp
from ai_batch_processor import ai_batch_processor
# DISABLED FOR DEMO: from enhanced_data_collection_system import start_enhanced_collection, get_collection_progress, get_source_analytics
from data_progress_dashboard import data_dashboard_bp
from breeding_ai import breeding_ai
import breeding_routes  # Import breeding routes
from lab_routes import lab_bp  # Import OrchidStein Lab routes
from data_integrity_safeguards import validate_orchid_record_integrity, enforce_data_integrity_before_save  # Critical data protection
import os
import json
import logging
import requests
import re
import time
import uuid
from datetime import datetime, timedelta
from sqlalchemy import or_, func, and_, cast, String
from io import BytesIO
from googleapiclient.http import MediaIoBaseDownload
import issue_reports
import chris_howard_reimport
from image_health_monitor import start_image_monitoring
from database_backup_system import create_database_backups, get_backup_orchids
try:
    from ai_system_monitor import start_ai_monitoring, get_monitoring_status, get_ai_monitor
except ImportError:
    start_ai_monitoring = None
    get_monitoring_status = None
    get_ai_monitor = None
from admin_control_center import register_admin_control_center
from admin_svo_research import admin_svo as admin_svo_bp
try:
    from automated_repair_system import repair_system
except ImportError:
    repair_system = None
# DISABLED: from comprehensive_diagnostic_system import start_diagnostic_monitoring, get_diagnostic_status
from eol_integration import EOLIntegrator
from external_databases.gbif_integration import GBIFIntegrator
from bug_report_system import bug_report_bp
from gary_photo_demo import gary_demo as gary_demo_bp
from drive_importer import drive_import_bp
from orchid_nursery_directory import get_nursery_directory, register_nursery_api_routes
from orchid_society_directory import get_society_directory, register_society_api_routes
from orchid_genetics_laboratory import register_genetics_laboratory
from citizen_science_platform import citizen_science_bp
from quantum_care_routes import register_quantum_care_routes
from widget_error_handler import widget_error_handler, safe_json_parse, safe_get_user_favorites, validate_feather_icon

# GBIF Botanical Service imports (replacing Trefle)
from gbif_botanical_service import GBIFBotanicalService
# Trefle batch enrichment service temporarily kept for admin routes compatibility
from trefle_batch_enrichment_service import (
    trefle_batch_service,
    create_enrichment_session,
    get_enrichment_session_status,
    list_enrichment_sessions,
    process_enrichment_batch,
    get_enrichment_statistics
)
import trefle_admin_routes  # Register Trefle admin routes (will be updated later)

# Create themed orchids system
ORCHID_THEMES = {
    'fragrant': {'name': 'Fragrant Orchids', 'keywords': ['fragrant', 'scented', 'perfume', 'vanilla', 'citrus', 'sweet']},
    'miniature': {'name': 'Miniature Orchids', 'keywords': ['mini', 'miniature', 'small', 'compact', 'dwarf']},
    'unusual': {'name': 'Unusual & Rare', 'keywords': ['unusual', 'rare', 'unique', 'strange', 'exotic', 'uncommon']},
    'colorful': {'name': 'Colorful Displays', 'keywords': ['colorful', 'vibrant', 'bright', 'rainbow', 'multicolor']},
    'species': {'name': 'Species Orchids', 'keywords': ['species', 'wild', 'natural', 'native', 'original']},
    'hybrids': {'name': 'Modern Hybrids', 'keywords': ['hybrid', 'cross', 'breeding', 'modern', 'new']}
}
from philosophy_quiz_service import philosophy_quiz_service

# Import and register report generation blueprint (moved to after logger initialization)
# This will be registered after logger is defined

def get_orchids_by_theme(theme_keywords):
    """Helper function to get orchids matching theme keywords"""
    query = db.session.query(OrchidRecord).filter(
        or_(
            OrchidRecord.google_drive_id.isnot(None),
            OrchidRecord.image_url.isnot(None),
            OrchidRecord.image_filename.like('%.jpg'),
            OrchidRecord.image_filename.like('%.jpeg'),
            OrchidRecord.image_filename.like('%.png')
        )
    )
    
    # Search for theme keywords in various fields
    keyword_filters = []
    for keyword in theme_keywords:
        keyword_filters.extend([
            OrchidRecord.scientific_name.ilike(f'%{keyword}%'),
            OrchidRecord.common_names.ilike(f'%{keyword}%'),
            OrchidRecord.ai_description.ilike(f'%{keyword}%'),
            OrchidRecord.cultural_notes.ilike(f'%{keyword}%')
        ])
    
    if keyword_filters:
        query = query.filter(or_(*keyword_filters))
        
    return query.limit(48).all()

# Initialize logger first
logger = logging.getLogger(__name__)

# Import GBIF Botanical Service for Ecosystem Explorer (after logger initialization)
try:
    from gbif_botanical_service import (
        get_gbif_service, search_orchid_ecosystem_data, enrich_orchid_with_gbif_data,
        get_gbif_service_status, batch_enrich_orchids_with_gbif
    )
    logger.info("✅ GBIF Botanical Service imported successfully")
    
    # Maintain backward compatibility with existing route function names
    get_trefle_service = get_gbif_service
    enrich_orchid_with_trefle_data = enrich_orchid_with_gbif_data
    get_trefle_service_status = get_gbif_service_status
    batch_enrich_orchids_with_trefle = batch_enrich_orchids_with_gbif
    
except ImportError as e:
    logger.warning(f"⚠️ GBIF Botanical Service not available: {e}")
    get_trefle_service = None
    search_orchid_ecosystem_data = None
    enrich_orchid_with_trefle_data = None
    get_trefle_service_status = None
    batch_enrich_orchids_with_trefle = None

# Import and register report generation blueprint
try:
    from report_routes import report_bp
    app.register_blueprint(report_bp)
    logger.info("📊 Report generation system registered successfully")
except ImportError as e:
    logger.warning(f"⚠️ Report generation system not available: {e}")
except Exception as e:
    logger.error(f"❌ Error registering report generation system: {e}")

# CRITICAL: Import real-time integrity monitoring
try:
    from realtime_integrity_guardian import trigger_user_integrity_check, get_integrity_status
    logger.info("🛡️ Integrity guardian imported successfully")
except Exception as e:
    logger.error(f"🚨 CRITICAL: Failed to import integrity guardian: {e}")
    trigger_user_integrity_check = lambda x: True  # Fallback function
    get_integrity_status = lambda: {"status": "unavailable"}

# DISABLED: All background monitoring systems to prevent server crashes
# These were causing worker timeouts and SIGKILL errors
# try:
#     start_diagnostic_monitoring()
#     logger.info("🔧 Comprehensive diagnostic system started")
# except Exception as e:
#     logger.error(f"❌ Failed to start diagnostic system: {e}")

# DISABLED: Enhanced data collection system - was overloading server
# try:
#     start_enhanced_collection()
#     logger.info("🌐 Enhanced data collection system started")
# except Exception as e:
#     logger.error(f"❌ Failed to start enhanced collection: {e}")

# DISABLED: Additional monitoring systems - keep only vigilant monitor (30-second)
# try:
#     from widget_integration_hub import widget_hub, track_widget_interaction, get_enhanced_widget_data
#     from mobile_widget_optimizer import mobile_optimizer
#     from user_collection_hub import collection_hub
#     logger.info("✅ Widget integration modules loaded successfully")
# except ImportError as e:
#     logger.warning(f"⚠️ Widget integration modules not available: {e}")

# DISABLED: Integrity check on every request - was causing worker crashes
# @app.before_request
# def trigger_integrity_on_user_visit():
#     """MISSION CRITICAL: Run integrity check on every user activity"""
#     try:
#         # Get user IP for tracking
#         user_ip = request.environ.get('HTTP_X_FORWARDED_FOR', request.remote_addr)
#         
#         # Skip checks for static files and API calls to avoid performance issues
#         if (request.endpoint and 
#             not request.endpoint.startswith('static') and 
#             not request.path.startswith('/api/drive-photo') and
#             not request.path.startswith('/static')):
#             
#             # Trigger integrity validation
#             trigger_user_integrity_check(user_ip)
#             
#     except Exception as e:
#         logger.error(f"🚨 CRITICAL: User-triggered integrity check failed: {e}")

# ============================================================================
# FEATURED ARTICLES SYSTEM - Showcase written articles
# ============================================================================

@app.route('/articles')
def featured_articles():
    """Display featured articles page showcasing written content"""
    return render_template('featured_articles.html', articles=FEATURED_ARTICLES)

# Articles configuration
FEATURED_ARTICLES = {
    'mythic-orchids': {
        'title': 'The Mythic Times and Orchids: A Journey Through Legend and Lore',
        'category': 'Mythology & Culture',
        'read_time': '18 min read',
        'file_path': 'static/articles/mythic_orchids.txt',
        'excerpt': 'Journey through the mythical connections between orchids and ancient civilizations, from Greek gods to Amazonian spirits, Chinese philosophy, and Nordic legends.',
        'status': 'published'
    },
    'vanilla-boy-story': {
        'title': 'The Boy Who Saved Vanilla: A Story of Innovation and Freedom',
        'category': 'History & Innovation', 
        'read_time': '12 min read',
        'file_path': 'static/articles/vanilla_boy_story.txt',
        'excerpt': 'The remarkable true story of Edmond Albius, a young slave who revolutionized the vanilla industry by discovering the secret of hand-pollinating vanilla orchids.',
        'status': 'published'
    },
    'jewel-orchids': {
        'title': 'The Fascinating World of Jewel Orchids',
        'category': 'Botanical Science',
        'read_time': '15 min read',
        'file_path': 'static/articles/jewel_orchids.txt',
        'excerpt': 'Step into the dazzling realm of jewel orchids, where leaves steal the spotlight with intricate patterns, metallic sheens, and fascinating biology.',
        'status': 'published'
    },
    'literary-orchids': {
        'title': 'Famous Literary Works Featuring Orchids',
        'category': 'Literature & Culture',
        'read_time': '8 min read',
        'file_path': 'static/articles/literary_orchids.txt',
        'excerpt': 'Explore how orchids have captivated writers throughout history, from "The Orchid Thief" to works by Neil Gaiman and Julia Alvarez.',
        'status': 'published'
    },
    'august-orchid-care': {
        'title': 'Orchid Care Tips for August',
        'category': 'Practical Guides',
        'read_time': '10 min read',
        'file_path': 'static/articles/august_orchid_care.txt',
        'excerpt': 'Essential August care tips for orchid enthusiasts, covering watering, light management, temperature control, and preparing for blooming season.',
        'status': 'published'
    },
    'halloween-black-rot': {
        'title': 'The Night of the Black Rot: A Tale of Orchids and Halloween',
        'category': 'Seasonal Stories',
        'read_time': '12 min read',
        'file_path': 'static/articles/halloween_black_rot.txt',
        'excerpt': 'A spooky Halloween tale featuring the Five Cities Orchid Society battling supernatural black rot with knowledge, community, and a protective charm.',
        'status': 'published'
    },
    'mars-orchids-terraforming': {
        'title': 'Orchids on Mars: Bioengineering the Future of Planetary Terraforming',
        'category': 'Science & Future',
        'read_time': '20 min read',
        'file_path': 'static/articles/mars_orchids_terraforming.txt',
        'excerpt': 'Explore the cutting-edge science of genetically engineered orchids designed to transform Mars into a habitable world, from soil remediation to oxygen production and even Martian ice cream.',
        'status': 'published'
    }
}

@app.route('/articles/<article_slug>')
def display_article(article_slug):
    """Display a specific article by slug"""
    article = FEATURED_ARTICLES.get(article_slug)
    if not article:
        logger.warning(f"Article not found: {article_slug}")
        return render_template('article_display.html',
                             title="Article Not Found",
                             content="The requested article could not be found.",
                             category="Error",
                             read_time="")
    
    try:
        # Load article content
        with open(article['file_path'], 'r', encoding='utf-8') as f:
            article_content = f.read()
        
        return render_template('article_display.html',
                             title=article['title'],
                             content=article_content,
                             category=article['category'],
                             read_time=article['read_time'])
    except FileNotFoundError:
        logger.warning(f"Article file not found: {article['file_path']}")
        return render_template('article_display.html',
                             title=article['title'],
                             content="This article is being prepared for publication. Please check back soon!",
                             category=article['category'],
                             read_time=article['read_time'])
    except Exception as e:
        logger.error(f"Failed to load article {article_slug}: {e}")
        return render_template('article_display.html',
                             title=article['title'],
                             content="Article content temporarily unavailable.",
                             category=article['category'],
                             read_time=article['read_time'])

# Add Gary Yong Gee Partnership Demo route
@app.route('/partnerships')
def partnerships():
    """Main partnerships page showcasing all Orchid Continuum partners"""
    return render_template('partnerships.html')

@app.route('/gary-demo')
def gary_demo():
    """Gary Yong Gee Partnership Demo"""
    return send_file('static/gary-demo-working.html')

@app.route('/gary-demo-working')
def gary_demo_working():
    """Gary Yong Gee Working Partnership Demo"""
    return send_file('static/gary-demo-working.html')

@app.route('/widget-demo')
def widget_demo():
    """FCOS Orchid Judge Mobile Widget Demo"""
    return render_template('widget_demo.html')

@app.route('/partner/gary/dashboard')
def gary_partner_dashboard():
    """Gary's partner dashboard - shows automated sync status"""
    return send_file('static/gary-partner-dashboard.html')

@app.route('/gary-partnership-demo')
def gary_partnership_demo():
    """Gary Yong Gee partnership demo - showing trait analysis research goals"""
    return render_template('gary_partnership_demo.html')

@app.route('/partnership-proposal')
def partnership_proposal():
    """Clean partnership proposal for Gary Yong Gee"""
    return render_template('partnership_proposal.html')

@app.route('/partnership-demo/gary-yong-gee')
def gary_partnership_demo_alt():
    """Alternative route for Gary Yong Gee partnership demo"""
    return render_template('gary_partnership_demo.html')

@app.route('/gary-story-demo')
def gary_story_demo():
    """Story-driven Gary partnership demo with compelling narrative flow"""
    return render_template('gary_partnership_demo_story.html')

@app.route('/gary/upload')
def gary_upload_page():
    """Simple photo upload page specifically for Gary Yong Gee"""
    return render_template('gary_simple_upload.html')

@app.route('/gary/mou')
def gary_mou_page():
    """Memorandum of Understanding for Gary Yong Gee partnership"""
    return render_template('gary_mou.html')

@app.route('/api/mou-signature', methods=['POST'])
def mou_signature():
    """Record MOU signature"""
    try:
        data = request.get_json()
        logger.info(f"📝 MOU Signature: {data['signer']} signed at {data['timestamp']}")
        return jsonify({'status': 'success', 'message': 'Signature recorded'})
    except Exception as e:
        logger.error(f"❌ MOU signature error: {e}")
        return jsonify({'error': 'Signature recording failed'}), 500

@app.route('/api/mou-complete', methods=['POST'])
def mou_complete():
    """Handle MOU completion"""
    try:
        data = request.get_json()
        logger.info(f"🎉 MOU COMPLETED: {data['document']} at {data['completion_time']}")
        # Here you could send emails, save to database, etc.
        return jsonify({'status': 'success', 'message': 'MOU completion recorded'})
    except Exception as e:
        logger.error(f"❌ MOU completion error: {e}")
        return jsonify({'error': 'Completion recording failed'}), 500

@app.route('/api/gary-upload', methods=['POST'])
def gary_upload_api():
    """API endpoint for Gary's photo uploads"""
    try:
        files = request.files
        form_data = request.form
        
        logger.info(f"🌺 Gary photo upload: {len(files)} files received")
        
        # Save files and metadata
        upload_data = {
            'species': form_data.get('species', ''),
            'location': form_data.get('location', ''),
            'conditions': form_data.get('conditions', ''),
            'notes': form_data.get('notes', ''),
            'files': []
        }
        
        # Process uploaded files
        for key, file in files.items():
            if file and file.filename:
                # Generate unique filename
                timestamp = int(time.time() * 1000)
                filename = f"gary_{timestamp}_{secure_filename(file.filename)}"
                filepath = os.path.join('static/uploads', filename)
                
                # Save file
                file.save(filepath)
                upload_data['files'].append({
                    'original_name': file.filename,
                    'saved_as': filename,
                    'path': filepath
                })
                
                logger.info(f"✅ Saved Gary photo: {filename}")
        
        # Log the upload for admin review
        logger.info(f"📊 Gary Upload Summary: {json.dumps(upload_data, indent=2)}")
        
        return jsonify({
            'status': 'success',
            'message': f'Successfully received {len(upload_data["files"])} photos',
            'files_received': len(upload_data['files'])
        })
        
    except Exception as e:
        logger.error(f"❌ Gary upload error: {e}")
        return jsonify({'status': 'error', 'message': 'Upload failed'}), 500

@app.route('/api/gary-bulk-upload', methods=['POST'])
def gary_bulk_upload_api():
    """Bulk upload API endpoint with authentication for Gary"""
    try:
        # Check for API key
        api_key = request.headers.get('X-API-Key') or request.form.get('api_key')
        if api_key != 'gary_orchid_continuum_2025':  # Simple API key for Gary
            return jsonify({'error': 'Invalid API key'}), 401
        
        files = request.files
        metadata = json.loads(request.form.get('metadata', '{}'))
        
        logger.info(f"🔑 Gary bulk upload: {len(files)} files with API key")
        
        results = []
        for key, file in files.items():
            if file and file.filename:
                timestamp = int(time.time() * 1000)
                filename = f"gary_bulk_{timestamp}_{secure_filename(file.filename)}"
                filepath = os.path.join('static/uploads', filename)
                
                file.save(filepath)
                results.append({
                    'original': file.filename,
                    'saved': filename,
                    'status': 'success'
                })
        
        return jsonify({
            'status': 'success',
            'uploaded': len(results),
            'files': results,
            'api_key_used': True
        })
        
    except Exception as e:
        logger.error(f"❌ Gary bulk upload error: {e}")
        return jsonify({'error': 'Bulk upload failed'}), 500

# Rotating gallery routes for demo
@app.route('/gallery/regional-rotating')
def regional_rotating_gallery():
    """Rotating regional collections for demo"""
    return redirect(url_for('thailand_gallery'))

@app.route('/gallery/traits-rotating')
def traits_rotating_gallery():
    """Rotating trait-based collections for demo"""
    return redirect(url_for('fragrant_gallery'))

@app.route('/gallery/community-rotating')
def community_rotating_gallery():
    """Rotating community features for demo"""
    return redirect(url_for('gallery'))

@app.route('/global-satellite-map')
def global_satellite_map():
    """Clean satellite view of Earth with orchid points of light"""
    return render_template('global_satellite_map.html')

@app.route('/space-earth-globe')
def space_earth_globe():
    """True 3D spinning Earth globe viewed from space with orchid constellation"""
    return render_template('space_earth_globe.html')

@app.route('/api/orchid-coordinates-all')
def orchid_coordinates_all():
    """Get ALL orchid coordinates for the 3D globe with genus filtering"""
    try:
        genus_filter = request.args.get('genus', '')
        
        # Build base query
        query = OrchidRecord.query.filter(
            and_(
                OrchidRecord.decimal_latitude.isnot(None),
                OrchidRecord.decimal_longitude.isnot(None)
            )
        )
        
        # Apply genus filter if provided
        if genus_filter and genus_filter != 'all':
            query = query.filter(OrchidRecord.genus == genus_filter)
        
        orchids = query.all()
        
        coordinates = []
        for orchid in orchids:
            if orchid.decimal_latitude and orchid.decimal_longitude:
                coordinates.append({
                    'id': orchid.id,
                    'lat': float(orchid.decimal_latitude),
                    'lng': float(orchid.decimal_longitude),
                    'name': orchid.display_name or orchid.scientific_name or f"Orchid {orchid.id}",
                    'genus': orchid.genus or (orchid.scientific_name.split(' ')[0] if orchid.scientific_name else 'Unknown'),
                    'species': orchid.species,
                    'location': orchid.locality or 'Unknown location',
                    'image': orchid.image_url if orchid.image_url else None,
                    'source': getattr(orchid, 'ingestion_source', 'Database') or 'Database'
                })
        
        filter_info = f" (filtered by {genus_filter})" if genus_filter and genus_filter != 'all' else ""
        logger.info(f"🌍 Loaded {len(coordinates)} orchid coordinates for 3D globe{filter_info}")
        return jsonify({
            'success': True,
            'coordinates': coordinates,
            'total_count': len(coordinates),
            'filter_applied': genus_filter
        })
        
    except Exception as e:
        logger.error(f"Error loading all orchid coordinates: {e}")
        return jsonify({
            'success': False,
            'error': str(e),
            'coordinates': []
        }), 500

@app.route('/api/orchid-genera')
def orchid_genera():
    """Get all unique orchid genera for the filtering dropdown"""
    try:
        # Get all unique genera from database, excluding None/empty values
        genera_result = db.session.query(OrchidRecord.genus).filter(
            and_(
                OrchidRecord.genus.isnot(None),
                OrchidRecord.genus != '',
                OrchidRecord.genus != 'Unknown'
            )
        ).distinct().all()
        
        # Extract genus names and sort them
        genera = sorted([g[0] for g in genera_result if g[0]])
        
        logger.info(f"🔍 Found {len(genera)} unique orchid genera")
        return jsonify({
            'success': True,
            'genera': genera,
            'total_count': len(genera)
        })
        
    except Exception as e:
        logger.error(f"Error loading orchid genera: {e}")
        return jsonify({
            'success': False,
            'error': str(e),
            'genera': []
        }), 500

@app.route('/api/image-counts')
def orchid_image_counts():
    """Get image counts by genus and type (species/hybrids) for search interface"""
    try:
        # Count images by genus - only count orchids that have images
        genus_counts_result = db.session.query(
            OrchidRecord.genus,
            func.count(OrchidRecord.id).label('count')
        ).filter(
            and_(
                OrchidRecord.genus.isnot(None),
                OrchidRecord.genus != '',
                OrchidRecord.genus != 'Unknown',
                or_(
                    OrchidRecord.google_drive_id.isnot(None),
                    OrchidRecord.image_filename.isnot(None),
                    OrchidRecord.image_url.isnot(None)
                )
            )
        ).group_by(OrchidRecord.genus).all()
        
        # Convert to dictionary
        genus_counts = {genus: count for genus, count in genus_counts_result if genus}
        
        # Count total images
        total_with_images = sum(genus_counts.values())
        
        # Count by type (species vs hybrids) - with images only
        species_count = db.session.query(OrchidRecord).filter(
            and_(
                or_(OrchidRecord.is_species == True, 
                    OrchidRecord.scientific_name.like('% %'),  # Species typically have 2 words
                    OrchidRecord.scientific_name.notlike('%×%')),  # Not hybrid symbol
                or_(
                    OrchidRecord.google_drive_id.isnot(None),
                    OrchidRecord.image_filename.isnot(None),
                    OrchidRecord.image_url.isnot(None)
                )
            )
        ).count()
        
        hybrid_count = db.session.query(OrchidRecord).filter(
            and_(
                or_(OrchidRecord.is_hybrid == True,
                    OrchidRecord.scientific_name.like('%×%'),  # Contains hybrid symbol
                    OrchidRecord.grex_name.isnot(None)),  # Has grex name
                or_(
                    OrchidRecord.google_drive_id.isnot(None),
                    OrchidRecord.image_filename.isnot(None),
                    OrchidRecord.image_url.isnot(None)
                )
            )
        ).count()
        
        # Count primary hybrids (F1 crosses between species)
        primary_hybrid_count = db.session.query(OrchidRecord).filter(
            and_(
                or_(OrchidRecord.is_hybrid == True,
                    OrchidRecord.scientific_name.like('%×%')),
                OrchidRecord.generation == 1,
                or_(
                    OrchidRecord.google_drive_id.isnot(None),
                    OrchidRecord.image_filename.isnot(None),
                    OrchidRecord.image_url.isnot(None)
                )
            )
        ).count()
        
        # Count currently flowering (if we have bloom time data)
        flowering_count = db.session.query(OrchidRecord).filter(
            and_(
                OrchidRecord.bloom_time.isnot(None),
                OrchidRecord.bloom_time != '',
                or_(
                    OrchidRecord.google_drive_id.isnot(None),
                    OrchidRecord.image_filename.isnot(None),
                    OrchidRecord.image_url.isnot(None)
                )
            )
        ).count()
        
        # Count featured orchids with images
        featured_count = db.session.query(OrchidRecord).filter(
            and_(
                OrchidRecord.is_featured == True,
                or_(
                    OrchidRecord.google_drive_id.isnot(None),
                    OrchidRecord.image_filename.isnot(None),
                    OrchidRecord.image_url.isnot(None)
                )
            )
        ).count()
        
        # Count top genera with most images (for Quick Actions)
        top_genera = sorted(genus_counts.items(), key=lambda x: x[1], reverse=True)[:10]
        
        logger.info(f"📊 Image counts: {total_with_images} total, {len(genus_counts)} genera with images")
        
        return jsonify({
            'success': True,
            'genus_counts': genus_counts,
            'type_counts': {
                'species': species_count,
                'hybrids': hybrid_count,
                'primary_hybrids': primary_hybrid_count
            },
            'special_counts': {
                'flowering': flowering_count,
                'featured': featured_count
            },
            'top_genera': dict(top_genera),
            'total_with_images': total_with_images,
            'last_updated': datetime.utcnow().isoformat()
        })
        
    except Exception as e:
        logger.error(f"Error loading orchid image counts: {e}")
        return jsonify({
            'success': False,
            'error': str(e),
            'genus_counts': {},
            'type_counts': {},
            'total_with_images': 0
        }), 500

@app.route('/api/orchid-ecosystem-data')
def orchid_ecosystem_data():
    """Comprehensive ecosystem data API with filtering support for the unified Earth Intelligence Platform"""
    try:
        logger.info("🔬 Fetching orchid ecosystem data with filters...")
        
        # Get filter parameters
        genus_filter = request.args.get('genus')
        climate_filter = request.args.get('climate')
        growth_habit_filter = request.args.get('growth_habit')
        pollinator_filter = request.args.get('pollinator')
        region_filter = request.args.get('region')
        
        # Start with base query for orchids with coordinates
        query = db.session.query(OrchidRecord).filter(
            and_(
                OrchidRecord.decimal_latitude.isnot(None),
                OrchidRecord.decimal_longitude.isnot(None)
            )
        )
        
        # Apply filters
        if genus_filter:
            query = query.filter(OrchidRecord.genus.ilike(f'%{genus_filter}%'))
            
        if climate_filter:
            # Map climate filter values to possible database values
            climate_mapping = {
                'cool': ['cool', 'cool growing', 'cool-growing', 'cold'],
                'intermediate': ['intermediate', 'intermediate growing', 'moderate'],
                'warm': ['warm', 'warm growing', 'warm-growing'],
                'hot': ['hot', 'hot growing', 'hot-growing']
            }
            if climate_filter in climate_mapping:
                climate_conditions = [
                    OrchidRecord.climate_preference.ilike(f'%{term}%') 
                    for term in climate_mapping[climate_filter]
                ]
                query = query.filter(or_(*climate_conditions))
                
        if growth_habit_filter:
            query = query.filter(OrchidRecord.growth_habit.ilike(f'%{growth_habit_filter}%'))
            
        if pollinator_filter:
            # Use native_habitat as fallback for pollinator info
            query = query.filter(OrchidRecord.native_habitat.ilike(f'%{pollinator_filter}%'))
            
        if region_filter:
            query = query.filter(
                or_(
                    OrchidRecord.region.ilike(f'%{region_filter}%'),
                    OrchidRecord.native_habitat.ilike(f'%{region_filter}%'),
                    OrchidRecord.country.ilike(f'%{region_filter}%')
                )
            )
        
        # Execute query with limit for performance
        orchids = query.limit(1000).all()
        
        # Prepare response data
        orchid_data = []
        for orchid in orchids:
            lat = orchid.decimal_latitude
            lng = orchid.decimal_longitude
            
            if lat and lng:
                orchid_info = {
                    'id': orchid.id,
                    'genus': orchid.genus,
                    'species': orchid.species,
                    'scientific_name': orchid.scientific_name or f"{orchid.genus or ''} {orchid.species or ''}".strip(),
                    'display_name': orchid.display_name,
                    'decimal_latitude': float(lat),
                    'decimal_longitude': float(lng),
                    'climate_preference': orchid.climate_preference,
                    'growth_habit': orchid.growth_habit,
                    'pollinator_types': [],  # Field doesn't exist in model
                    'region': orchid.region,
                    'continent': orchid.country or 'Unknown',
                    'native_habitat': orchid.native_habitat,
                    'native_distribution': orchid.native_habitat,
                    'mycorrhizal_fungi': [],  # Field doesn't exist in model
                    'bloom_time': orchid.bloom_time,
                    'flowering_time': orchid.bloom_time,
                    'temperature_range': orchid.temperature_range,
                    'light_requirements': orchid.light_requirements,
                    'image_url': orchid.image_url,
                    'google_drive_id': orchid.google_drive_id,
                    'is_fragrant': False,  # Field doesn't exist in model
                    'conservation_status_clues': 'Unknown',  # Field doesn't exist in model
                    'ecosystem_enhanced': True
                }
                orchid_data.append(orchid_info)
        
        # Log filter application
        filter_info = []
        if genus_filter: filter_info.append(f"genus={genus_filter}")
        if climate_filter: filter_info.append(f"climate={climate_filter}")
        if growth_habit_filter: filter_info.append(f"growth={growth_habit_filter}")
        if pollinator_filter: filter_info.append(f"pollinator={pollinator_filter}")
        if region_filter: filter_info.append(f"region={region_filter}")
        
        filters_applied = " | ".join(filter_info) if filter_info else "none"
        logger.info(f"🌺 Ecosystem query returned {len(orchid_data)} orchids with filters: {filters_applied}")
        
        return jsonify({
            'success': True,
            'orchids': orchid_data,
            'count': len(orchid_data),
            'filters_applied': {
                'genus': genus_filter,
                'climate': climate_filter,
                'growth_habit': growth_habit_filter,
                'pollinator': pollinator_filter,
                'region': region_filter
            }
        })
        
    except Exception as e:
        logger.error(f"Error fetching ecosystem data: {str(e)}")
        return jsonify({
            'success': False,
            'error': 'Failed to fetch ecosystem data',
            'details': str(e)
        }), 500

@app.route('/api/global-weather-patterns')
def global_weather_patterns():
    """Get current weather patterns for global overlay"""
    try:
        from weather_service import WeatherService
        
        # Define major cities for weather pattern overlay
        weather_points = [
            {'name': 'New York', 'lat': 40.7128, 'lng': -74.0060},
            {'name': 'London', 'lat': 51.5074, 'lng': -0.1278},
            {'name': 'Tokyo', 'lat': 35.6762, 'lng': 139.6503},
            {'name': 'Sydney', 'lat': -33.8688, 'lng': 151.2093},
            {'name': 'Mumbai', 'lat': 19.0760, 'lng': 72.8777},
            {'name': 'São Paulo', 'lat': -23.5505, 'lng': -46.6333},
            {'name': 'Cairo', 'lat': 30.0444, 'lng': 31.2357},
            {'name': 'Mexico City', 'lat': 19.4326, 'lng': -99.1332},
            {'name': 'Bangkok', 'lat': 13.7563, 'lng': 100.5018},
            {'name': 'Cape Town', 'lat': -33.9249, 'lng': 18.4241},
            {'name': 'Moscow', 'lat': 55.7558, 'lng': 37.6176},
            {'name': 'Buenos Aires', 'lat': -34.6118, 'lng': -58.3960},
            {'name': 'Singapore', 'lat': 1.3521, 'lng': 103.8198},
            {'name': 'Vancouver', 'lat': 49.2827, 'lng': -123.1207},
            {'name': 'Jakarta', 'lat': -6.2088, 'lng': 106.8456}
        ]
        
        weather_data = []
        for point in weather_points:
            try:
                weather = WeatherService.get_current_weather(
                    point['lat'], point['lng'], point['name']
                )
                if weather:
                    weather_data.append({
                        'name': point['name'],
                        'lat': point['lat'],
                        'lng': point['lng'],
                        'temperature': weather.temperature,
                        'humidity': weather.humidity,
                        'wind_speed': weather.wind_speed,
                        'wind_direction': weather.wind_direction,
                        'pressure': weather.pressure,
                        'cloud_cover': weather.cloud_cover,
                        'weather_code': weather.weather_code,
                        'conditions': get_weather_description(weather.weather_code) if weather.weather_code else 'Unknown'
                    })
            except Exception as point_error:
                logger.warning(f"Weather fetch failed for {point['name']}: {point_error}")
                continue
        
        logger.info(f"🌤️ Loaded weather patterns for {len(weather_data)} global locations")
        return jsonify({
            'success': True,
            'weather_points': weather_data,
            'total_count': len(weather_data),
            'timestamp': datetime.utcnow().isoformat()
        })
        
    except Exception as e:
        logger.error(f"Error loading global weather patterns: {e}")
        return jsonify({
            'success': False,
            'error': str(e),
            'weather_points': []
        }), 500

def get_weather_description(weather_code):
    """Convert weather code to description"""
    weather_codes = {
        0: 'Clear sky',
        1: 'Mainly clear', 2: 'Partly cloudy', 3: 'Overcast',
        45: 'Fog', 48: 'Depositing rime fog',
        51: 'Light drizzle', 53: 'Moderate drizzle', 55: 'Dense drizzle',
        61: 'Slight rain', 63: 'Moderate rain', 65: 'Heavy rain',
        71: 'Slight snow', 73: 'Moderate snow', 75: 'Heavy snow',
        80: 'Slight rain showers', 81: 'Moderate rain showers', 82: 'Violent rain showers',
        95: 'Thunderstorm', 96: 'Thunderstorm with hail'
    }
    return weather_codes.get(weather_code, 'Unknown')

@app.route('/api/satellite-monitoring')
def satellite_monitoring():
    """Get comprehensive satellite monitoring data for environmental conditions"""
    try:
        from datetime import datetime, timedelta
        import requests
        
        # Sample global monitoring points for demonstration
        monitoring_points = [
            {'name': 'Amazon Basin', 'lat': -3.0, 'lng': -60.0},
            {'name': 'Indonesian Rainforest', 'lat': -2.0, 'lng': 118.0},
            {'name': 'Madagascar', 'lat': -19.0, 'lng': 47.0},
            {'name': 'Costa Rica', 'lat': 10.0, 'lng': -84.0},
            {'name': 'Philippines', 'lat': 12.0, 'lng': 122.0},
            {'name': 'Ecuador Cloud Forest', 'lat': -0.5, 'lng': -78.5},
            {'name': 'Myanmar', 'lat': 22.0, 'lng': 96.0},
            {'name': 'Colombia', 'lat': 4.0, 'lng': -73.0},
            {'name': 'Papua New Guinea', 'lat': -6.0, 'lng': 144.0},
            {'name': 'Borneo', 'lat': 0.0, 'lng': 114.0}
        ]
        
        satellite_data = []
        
        for point in monitoring_points:
            # Simulate comprehensive satellite monitoring data
            # In production, these would come from real satellite APIs
            data_point = {
                'name': point['name'],
                'lat': point['lat'],
                'lng': point['lng'],
                'timestamp': datetime.utcnow().isoformat(),
                
                # Thermal Monitoring (MODIS/Landsat)
                'thermal': {
                    'surface_temp': 25 + (point['lat'] * -0.5),  # Simulated
                    'heat_anomaly': False,
                    'thermal_stress_index': 'Low'
                },
                
                # Air Quality (Sentinel-5P/TROPOMI)
                'air_quality': {
                    'co2_ppm': 415 + abs(point['lat']) * 2,  # Simulated
                    'no2_level': 'Low',
                    'so2_level': 'Normal',
                    'aerosol_index': 0.3,
                    'air_quality_index': 'Good'
                },
                
                # Ocean/Water Monitoring (MODIS Ocean Color)
                'water_quality': {
                    'algae_bloom_risk': 'Low' if abs(point['lat']) > 10 else 'Moderate',
                    'chlorophyll_a': 0.5,  # mg/m³
                    'sea_surface_temp': 26.0,
                    'water_clarity': 'Good'
                },
                
                # Volcanic/Geological (OMI/TROPOMI)
                'geological': {
                    'volcanic_so2': 'Normal',
                    'seismic_activity': 'Stable',
                    'ground_deformation': 'None detected'
                },
                
                # Vegetation Health (MODIS/Landsat NDVI)
                'vegetation': {
                    'ndvi_index': 0.7,  # Normalized Difference Vegetation Index
                    'vegetation_health': 'Healthy',
                    'deforestation_alert': False,
                    'drought_stress': 'None'
                },
                
                # Fire Monitoring (MODIS/VIIRS Active Fire)
                'fire_monitoring': {
                    'active_fires': 0,
                    'fire_risk': 'Low',
                    'burn_scar_area': 0,
                    'smoke_plume': False
                },
                
                # Climate Indicators
                'climate': {
                    'precipitation_anomaly': 'Normal',
                    'temperature_anomaly': 'Normal',
                    'humidity_level': 'Optimal',
                    'wind_patterns': 'Stable'
                }
            }
            
            # Add orchid-specific risk assessment
            orchid_risk_factors = []
            if data_point['thermal']['surface_temp'] > 35:
                orchid_risk_factors.append('High temperature stress')
            if data_point['air_quality']['co2_ppm'] > 450:
                orchid_risk_factors.append('Elevated CO2 levels')
            if data_point['vegetation']['ndvi_index'] < 0.4:
                orchid_risk_factors.append('Habitat degradation')
            if data_point['fire_monitoring']['active_fires'] > 0:
                orchid_risk_factors.append('Fire threat')
            
            data_point['orchid_habitat_assessment'] = {
                'risk_level': 'High' if len(orchid_risk_factors) > 2 else 'Medium' if len(orchid_risk_factors) > 0 else 'Low',
                'risk_factors': orchid_risk_factors,
                'habitat_quality': 'Excellent' if len(orchid_risk_factors) == 0 else 'Good' if len(orchid_risk_factors) <= 1 else 'At Risk'
            }
            
            satellite_data.append(data_point)
        
        logger.info(f"🛰️ Generated satellite monitoring data for {len(satellite_data)} locations")
        
        return jsonify({
            'success': True,
            'satellite_data': satellite_data,
            'monitoring_capabilities': {
                'thermal_imaging': 'MODIS, Landsat 8/9, VIIRS',
                'atmospheric_chemistry': 'Sentinel-5P, OCO-2/3, TROPOMI',
                'ocean_color': 'MODIS, VIIRS, Sentinel-3',
                'geological_monitoring': 'Sentinel-1, ALOS-2',
                'vegetation_health': 'MODIS, Landsat, Sentinel-2',
                'fire_detection': 'MODIS, VIIRS Active Fire',
                'weather_patterns': 'GOES, Himawari, Meteosat'
            },
            'data_sources': [
                'NASA MODIS Terra/Aqua',
                'Landsat 8/9 Thermal',
                'Sentinel-5P TROPOMI',
                'OCO-2/3 Carbon Observatory',
                'VIIRS Visible Infrared',
                'Sentinel-3 Ocean Color',
                'GOES Weather Satellite',
                'ESA Copernicus Program'
            ],
            'update_frequency': 'Real-time to daily depending on satellite',
            'total_locations': len(satellite_data)
        })
        
    except Exception as e:
        logger.error(f"Error generating satellite monitoring data: {e}")
        return jsonify({
            'success': False,
            'error': str(e),
            'satellite_data': []
        }), 500

@app.route('/api/earth-ai-chat', methods=['POST'])
def earth_ai_chat():
    """AI chat endpoint for Earth Intelligence Assistant"""
    try:
        data = request.get_json()
        user_message = data.get('message', '')
        context = data.get('context', {})
        
        if not user_message:
            return jsonify({'success': False, 'error': 'No message provided'}), 400
        
        # Import OpenAI here to avoid import issues
        import openai
        import os
        
        # Set up OpenAI
        openai.api_key = os.environ.get('OPENAI_API_KEY')
        
        # Build system prompt with current context
        system_prompt = f"""You are Earth Intelligence Assistant, an AI helping users explore a 3D Earth globe with orchid data, weather patterns, and emergency tracking capabilities.

CURRENT GLOBE STATE:
- Zoom Level: {context.get('zoom_level', 'Unknown')}
- Weather Overlay: {'Active' if context.get('weather_overlay_active') else 'Inactive'}
- Genus Filter: {context.get('genus_filter', 'All')}
- Orchid Points Visible: {context.get('orchid_count', 0)}

CAPABILITIES:
1. Orchid Data Analysis: Query and filter by genus, location, species
2. Weather Pattern Analysis: Current weather overlay with temperature zones
3. Emergency Tracking: Can discuss storms, hurricanes, wildfires, tsunamis
4. Navigation: Zoom to countries/regions, reset view, filter controls
5. Globe Controls: Weather toggle, genus filtering, zoom controls
6. Satellite Monitoring: Heat distribution, CO2 levels, algae blooms, volcanic activity, vegetation health, fire detection

AVAILABLE COMMANDS (return as JSON "commands" array):
- zoom_to_country: {"action": "zoom_to_country", "lat": 40, "lng": -100, "zoom": 1.5}
- filter_genus: {"action": "filter_genus", "genus": "Cattleya"}
- toggle_weather: {"action": "toggle_weather"}
- reset_view: {"action": "reset_view"}
- clear_filters: {"action": "clear_filters"}

EMERGENCY DATA SOURCES:
- Weather patterns from Open-Meteo API (currently showing {context.get('orchid_count', 0)} orchid locations)
- Can discuss natural disaster preparedness for orchid collections
- Climate impact analysis on orchid habitats

Respond helpfully about orchids, weather, navigation, or emergency preparedness. Include relevant commands when appropriate."""

        # Make API call to OpenAI
        from openai import OpenAI
        client = OpenAI()
        response = client.chat.completions.create(
            model="gpt-4",
            messages=[
                {"role": "system", "content": system_prompt},
                {"role": "user", "content": user_message}
            ],
            max_tokens=500,
            temperature=0.7
        )
        
        ai_response = response.choices[0].message.content
        
        # Parse any commands from the response
        commands = []
        
        # Simple command parsing - look for natural language cues
        message_lower = user_message.lower()
        
        if 'show' in message_lower and 'cattleya' in message_lower:
            commands.append({"action": "filter_genus", "genus": "Cattleya"})
        elif 'show' in message_lower and 'dendrobium' in message_lower:
            commands.append({"action": "filter_genus", "genus": "Dendrobium"})
        elif 'show' in message_lower and 'phalaenopsis' in message_lower:
            commands.append({"action": "filter_genus", "genus": "Phalaenopsis"})
        elif 'zoom' in message_lower and ('usa' in message_lower or 'america' in message_lower):
            commands.append({"action": "zoom_to_country", "lat": 40, "lng": -100, "zoom": 1.3})
        elif 'zoom' in message_lower and ('europe' in message_lower):
            commands.append({"action": "zoom_to_country", "lat": 55, "lng": 10, "zoom": 1.3})
        elif 'zoom' in message_lower and ('asia' in message_lower):
            commands.append({"action": "zoom_to_country", "lat": 20, "lng": 100, "zoom": 1.3})
        elif 'weather' in message_lower and ('show' in message_lower or 'display' in message_lower):
            commands.append({"action": "toggle_weather"})
        elif 'reset' in message_lower or 'clear' in message_lower:
            commands.append({"action": "reset_view"})
        
        logger.info(f"🤖 AI Chat: {user_message[:100]}... -> {len(ai_response)} chars")
        
        return jsonify({
            'success': True,
            'response': ai_response,
            'commands': commands
        })
        
    except Exception as e:
        logger.error(f"AI Chat error: {e}")
        return jsonify({
            'success': False,
            'error': 'AI chat service temporarily unavailable',
            'response': 'I apologize, but I\'m having technical difficulties right now. Please try again in a moment.'
        }), 500

@app.route('/admin/diagnostic-status')
def diagnostic_status():
    """Get diagnostic system status"""
    try:
        # Diagnostic system is disabled
        status = {'status': 'disabled', 'message': 'Diagnostic system is currently disabled'}
        return jsonify(status)
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/admin/restart-widgets', methods=['POST'])
def restart_widgets():
    """Restart widget services"""
    try:
        # Clear any widget caches and restart services
        logger.info("🔧 Restarting widget services")
        return jsonify({'status': 'widgets restarted'})
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/admin/restart-services', methods=['POST'])
def restart_services():
    """Restart application services"""
    try:
        logger.info("🔧 Restarting application services")
        return jsonify({'status': 'services restarted'})
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/admin/restart-media', methods=['POST'])
def restart_media():
    """Restart media services"""
    try:
        logger.info("🔧 Restarting media services")
        return jsonify({'status': 'media services restarted'})
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/admin/start-enhanced-collection', methods=['POST'])
def start_enhanced_collection_route():
    """Start enhanced data collection"""
    try:
        # Enhanced collection system is disabled
        logger.info("🌐 Enhanced data collection system is currently disabled")
        return jsonify({'status': 'Enhanced collection system is disabled'})
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/api/chat-search-assist', methods=['POST'])
def search_ai_chat():
    """AI chat assistant for advanced orchid search"""
    try:
        from orchid_ai import openai_client
        
        data = request.get_json()
        message = data.get('message', '').strip()
        search_context = data.get('search_context', {})
        
        if not message:
            return jsonify({'error': 'Message is required'}), 400
        
        # Build context-aware system prompt based on current search state
        current_filters = []
        if search_context.get('query'):
            current_filters.append(f"Search Query: '{search_context['query']}'")
        if search_context.get('genus'):
            current_filters.append(f"Genus Filter: {search_context['genus']}")
        if search_context.get('species'):
            current_filters.append(f"Species Filter: {search_context['species']}")
        if search_context.get('location'):
            current_filters.append(f"Location Filter: {search_context['location']}")
        if search_context.get('flowering_season'):
            current_filters.append(f"Flowering Season: {search_context['flowering_season']}")
        if search_context.get('growth_habit'):
            current_filters.append(f"Growth Habit: {search_context['growth_habit']}")
        
        context_summary = "; ".join(current_filters) if current_filters else "No active filters"
        
        system_prompt = f"""You are an expert orchid botanist and search assistant for the Orchid Continuum platform's advanced search interface. You help users navigate complex biological search capabilities and provide expert botanical guidance.

CURRENT SEARCH CONTEXT:
{context_summary}

YOUR CAPABILITIES:
1. **Search Assistance**: Help refine search queries, suggest better filters, interpret search results
2. **Orchid Identification**: Identify species from descriptions, help with taxonomic classification
3. **Biological Guidance**: Explain search filters like pollination syndromes, mycorrhizal associations, ecological niches
4. **Research Support**: Guide users through advanced biological analysis filters, trait discovery, conservation status
5. **Botanical Education**: Explain scientific terminology, morphological features, breeding concepts

SEARCH FILTERS AVAILABLE:
- Taxonomic: Genus, species, author, hybrid status
- Geographic: Location, elevation, climate zones
- Biological: Pollination, mycorrhizal relationships, ecological roles
- Morphological: Flower characteristics, growth habits, plant size
- Conservation: Threat status, protection level
- AI Analysis: Computer vision insights, trait discovery, pattern recognition

RESPONSE STYLE:
- Be concise but informative (max 200 words)
- Use botanical terminology appropriately
- Suggest specific search actions when relevant
- Reference current search context when applicable
- Offer concrete next steps for research

When suggesting search refinements, reference specific filter categories and values that would help the user's research goals."""

        user_prompt = f"""User question: {message}

Based on the current search context, provide helpful guidance for orchid research and search refinement."""
        
        # Call OpenAI API using GPT-5 (the newest OpenAI model released August 7, 2025)
        # do not change this unless explicitly requested by the user
        response = openai_client.chat.completions.create(
            model="gpt-5",
            messages=[
                {"role": "system", "content": system_prompt},
                {"role": "user", "content": user_prompt}
            ],
            max_tokens=300,
            temperature=0.7
        )
        
        ai_response = response.choices[0].message.content
        
        # Parse for suggested search actions
        suggested_actions = []
        message_lower = message.lower()
        
        # Detect if user is asking for specific filters
        if 'show me' in message_lower or 'find' in message_lower:
            if 'cattleya' in message_lower:
                suggested_actions.append({"type": "filter", "field": "genus", "value": "Cattleya"})
            elif 'dendrobium' in message_lower:
                suggested_actions.append({"type": "filter", "field": "genus", "value": "Dendrobium"})
            elif 'phalaenopsis' in message_lower:
                suggested_actions.append({"type": "filter", "field": "genus", "value": "Phalaenopsis"})
            elif 'oncidium' in message_lower:
                suggested_actions.append({"type": "filter", "field": "genus", "value": "Oncidium"})
            
        if 'fragrant' in message_lower:
            suggested_actions.append({"type": "search", "query": "fragrant scented perfume"})
        if 'miniature' in message_lower or 'small' in message_lower:
            suggested_actions.append({"type": "search", "query": "miniature dwarf compact small"})
        if 'epiphyte' in message_lower:
            suggested_actions.append({"type": "filter", "field": "growth_habit", "value": "epiphytic"})
        if 'terrestrial' in message_lower:
            suggested_actions.append({"type": "filter", "field": "growth_habit", "value": "terrestrial"})
        
        logger.info(f"🔍 Search AI Chat: {message[:50]}... -> {len(ai_response)} chars")
        
        return jsonify({
            'success': True,
            'response': ai_response,
            'suggested_actions': suggested_actions,
            'timestamp': datetime.now().isoformat()
        })
        
    except Exception as e:
        logger.error(f"❌ Search AI chat error: {e}")
        return jsonify({
            'success': False,
            'error': str(e),
            'fallback_response': 'I apologize, but I\'m having trouble processing your search question right now. Please try rephrasing your query or contact support if the issue persists.'
        }), 500

@app.route('/partner/api/send-to-ai', methods=['POST'])
def gary_ai_chat():
    """Real AI chat endpoint for Gary's messaging system"""
    try:
        from orchid_ai import openai_client
        
        data = request.get_json()
        message = data.get('message', '').strip()
        
        if not message:
            return jsonify({'error': 'Message is required'}), 400
            
        # Create AI prompt for Gary's orchid questions
        system_prompt = """You are an expert orchid botanist and AI assistant for the Orchid Continuum platform. 
        You're specifically helping Gary Yong Gee, our partner who runs orchids.yonggee.name - a renowned orchid expert and photographer.
        
        Gary has extensive botanical knowledge and has contributed thousands of orchid photos and data to our platform.
        He often asks about:
        - Orchid identification and taxonomy
        - Growing conditions and cultural requirements  
        - Flowering patterns and seasonal advice
        - Species relationships and breeding
        - Photo analysis and documentation
        
        Respond as a knowledgeable colleague who respects Gary's expertise while providing helpful insights.
        Keep responses informative but conversational. Reference specific orchid details when possible."""
        
        user_prompt = f"""Gary asks: {message}
        
        Please provide a helpful, expert response about orchids. Gary is a botanical expert himself, so you can use technical terminology appropriately."""
        
        # Call OpenAI API
        response = openai_client.chat.completions.create(
            model="gpt-4o",
            messages=[
                {"role": "system", "content": system_prompt},
                {"role": "user", "content": user_prompt}
            ],
            max_tokens=300,
            temperature=0.7
        )
        
        ai_response = response.choices[0].message.content
        
        return jsonify({
            'success': True,
            'response': ai_response,
            'timestamp': datetime.now().isoformat()
        })
        
    except Exception as e:
        logger.error(f"❌ AI chat error: {e}")
        return jsonify({
            'success': False,
            'error': str(e),
            'fallback_response': 'I apologize, but I\'m having trouble processing your question right now. Our AI system will be back online shortly!'
        }), 500

@app.route('/partner/api/send-to-team', methods=['POST'])
def gary_team_message():
    """Team messaging endpoint for Gary"""
    try:
        data = request.get_json()
        message = data.get('message', '').strip()
        
        if not message:
            return jsonify({'error': 'Message is required'}), 400
            
        # Log message for team review
        logger.info(f"📨 Message from Gary: {message}")
        
        # In production, this would save to database and notify team
        
        return jsonify({
            'success': True,
            'response': 'Thanks Gary! We\'ve received your message and will respond within 24 hours. Your partnership is important to us!',
            'timestamp': datetime.now().isoformat()
        })
        
    except Exception as e:
        logger.error(f"❌ Team message error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/partner/api/request-data-review', methods=['POST'])
def gary_data_review():
    """Data review request endpoint for Gary"""
    try:
        data = request.get_json()
        message = data.get('message', 'Please review my recent orchid data for accuracy and completeness.')
        
        # Log review request
        logger.info(f"📊 Data review request from Gary: {message}")
        
        return jsonify({
            'success': True,
            'response': 'We\'ll conduct a comprehensive review of your recent submissions and provide a detailed report within 48 hours. This includes AI accuracy verification and botanical validation.',
            'timestamp': datetime.now().isoformat()
        })
        
    except Exception as e:
        logger.error(f"❌ Data review error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/run-sunset-valley-scraper', methods=['POST'])
def run_sunset_valley_scraper():
    """Run Sunset Valley Orchids scraper for Sarcochilus hybrids"""
    try:
        from sunset_valley_orchids_scraper import run_sunset_valley_scraper
        
        logger.info("🌅 Starting Sunset Valley Orchids scraper")
        summary = run_sunset_valley_scraper()
        
        return jsonify({
            'success': True,
            'summary': summary,
            'message': f"Successfully collected {summary['total_hybrids']} Sarcochilus hybrids from Sunset Valley Orchids"
        })
        
    except Exception as e:
        logger.error(f"❌ Sunset Valley scraper error: {e}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500

@app.route('/test_sunset_valley_scraper')
def test_sunset_valley_scraper():
    """Test endpoint for Sunset Valley Orchids scraper"""
    try:
        from sunset_valley_orchids_scraper import SunsetValleyOrchidsScraper
        
        scraper = SunsetValleyOrchidsScraper()
        
        # Test connection
        response = scraper.session.get(scraper.base_url, timeout=10)
        
        if response.status_code == 200:
            # Test data extraction
            hybrids = scraper.scrape_sarcochilus_hybrids()
            
            return jsonify({
                'success': True,
                'connection': 'OK',
                'hybrids_found': len(hybrids),
                'sample_hybrids': hybrids[:3] if hybrids else [],
                'message': f'Successfully connected and found {len(hybrids)} Sarcochilus hybrids'
            })
        else:
            return jsonify({
                'success': False,
                'connection': 'FAILED',
                'status_code': response.status_code,
                'message': 'Could not connect to Sunset Valley Orchids website'
            })
            
    except Exception as e:
        return jsonify({
            'success': False,
            'error': str(e),
            'message': 'Scraper test failed'
        })

@app.route('/admin/run-svo-intergeneric-scraper', methods=['POST'])
def run_svo_intergeneric_scraper_route():
    """Run SVO intergeneric cross scraper"""
    try:
        from sunset_valley_orchids_scraper import run_svo_intergeneric_scraper
        
        logger.info("🌈 Starting SVO intergeneric cross collection")
        summary = run_svo_intergeneric_scraper()
        
        return jsonify({
            'success': True,
            'summary': summary,
            'message': f"Successfully collected {summary['total_intergenerics']} intergeneric crosses from Sunset Valley Orchids"
        })
        
    except Exception as e:
        logger.error(f"❌ SVO intergeneric scraper error: {e}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500

@app.route('/test_svo_intergeneric_scraper')
def test_svo_intergeneric_scraper():
    """Test SVO intergeneric cross scraper"""
    try:
        from sunset_valley_orchids_scraper import SunsetValleyOrchidsScraper
        
        scraper = SunsetValleyOrchidsScraper()
        
        # Test connection and search for intergeneric content
        response = scraper.session.get(scraper.base_url, timeout=10)
        
        if response.status_code == 200:
            # Test intergeneric data extraction
            crosses = scraper.scrape_intergeneric_crosses()
            
            return jsonify({
                'success': True,
                'connection': 'OK',
                'intergenerics_found': len(crosses),
                'sample_crosses': crosses[:3] if crosses else [],
                'message': f'Successfully found {len(crosses)} intergeneric crosses'
            })
        else:
            return jsonify({
                'success': False,
                'connection': 'FAILED',
                'status_code': response.status_code,
                'message': 'Could not connect to Sunset Valley Orchids website'
            })
            
    except Exception as e:
        return jsonify({
            'success': False,
            'error': str(e),
            'message': 'Intergeneric scraper test failed'
        })

@app.route('/intergeneric-crosses')
def intergeneric_crosses():
    """Intergeneric orchid crosses exploration page"""
    return render_template('intergeneric_crosses.html')

# ========== ORCHID JUDGING SYSTEM ROUTES ==========

@app.route('/judging')
def judging_home():
    """Orchid judging system home page"""
    try:
        # Get available organizations
        organizations = get_available_organizations()
        
        # Get recent analyses for the current user (if any)
        recent_analyses = []
        if current_user and hasattr(current_user, 'id'):
            recent_analyses = JudgingAnalysis.query.filter_by(
                user_id=current_user.id
            ).order_by(JudgingAnalysis.created_at.desc()).limit(5).all()
        
        return render_template('judging/home.html', 
                             organizations=organizations,
                             recent_analyses=recent_analyses)
    except Exception as e:
        logger.error(f"❌ Judging home error: {e}")
        flash('Error loading judging system. Please try again.', 'error')
        return redirect(url_for('index'))

@app.route('/judging/analyze/<int:orchid_id>')
def judging_analyze_orchid(orchid_id):
    """Analyze a specific orchid for judging"""
    try:
        orchid = OrchidRecord.query.get_or_404(orchid_id)
        organization = request.args.get('org', 'AOS')  # Default to AOS
        
        # Perform judging analysis
        analysis_result = analyze_orchid_by_organization(orchid_id, organization)
        
        # Save analysis to database
        if current_user and hasattr(current_user, 'id'):
            judging_analysis = JudgingAnalysis()
            judging_analysis.orchid_id = orchid_id
            judging_analysis.user_id = current_user.id
            judging_analysis.judging_standard_id = 1  # Default standard
            judging_analysis.score = analysis_result.get('total_score', 0)
            judging_analysis.percentage = (analysis_result.get('total_score', 0) / 100.0) * 100  # Convert to percentage
            judging_analysis.detailed_analysis = json.dumps(analysis_result)
            judging_analysis.ai_comments = f"Organization: {organization}"
            db.session.add(judging_analysis)
            db.session.commit()
        
        return render_template('judging/analysis_result.html',
                             orchid=orchid,
                             organization=organization,
                             analysis=analysis_result)
        
    except Exception as e:
        logger.error(f"❌ Judging analysis error: {e}")
        flash('Error analyzing orchid. Please try again.', 'error')
        return redirect(url_for('judging_home'))

@app.route('/judging/enhanced-analyze/<int:orchid_id>')
def judging_enhanced_analyze(orchid_id):
    """Enhanced analysis with genetic factors"""
    try:
        orchid = OrchidRecord.query.get_or_404(orchid_id)
        organization = request.args.get('org', 'AOS')
        
        # Use enhanced judging with genetics
        analysis_result = analyze_orchid_with_genetics(orchid_id, organization)
        
        return render_template('judging/enhanced_analysis_result.html',
                             orchid=orchid,
                             organization=organization,
                             analysis=analysis_result)
        
    except Exception as e:
        logger.error(f"❌ Enhanced judging analysis error: {e}")
        flash('Error performing enhanced analysis. Please try again.', 'error')
        return redirect(url_for('judging_home'))

@app.route('/api/judging/quick-score/<int:orchid_id>')
def api_quick_judging_score(orchid_id):
    """Quick API endpoint for judging score"""
    try:
        organization = request.args.get('org', 'AOS')
        analysis_result = analyze_orchid_by_organization(orchid_id, organization)
        
        return jsonify({
            'success': True,
            'orchid_id': orchid_id,
            'organization': organization,
            'total_score': analysis_result.get('total_score', 0),
            'award_eligible': analysis_result.get('award_eligible', False),
            'potential_awards': analysis_result.get('potential_awards', [])
        })
        
    except Exception as e:
        logger.error(f"❌ Quick judging API error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/intergeneric-crosses')
def api_intergeneric_crosses():
    """API endpoint for intergeneric crosses data"""
    try:
        # Query intergeneric crosses from database using raw SQL to avoid column issues
        with app.app_context():
            from app import db
            
            # Use raw SQL query to get intergeneric crosses
            from sqlalchemy import text
            sql = text("""
                SELECT id, display_name, parentage_formula, ai_description, 
                       cultural_notes, image_url, created_at
                FROM orchid_record 
                WHERE genus = 'Intergeneric' 
                AND species = 'hybrid' 
                AND data_source = 'Sunset Valley Orchids'
                AND validation_status = 'approved'
                ORDER BY created_at DESC
            """)
            
            result = db.session.execute(sql)
            records = result.fetchall()
        
        crosses = []
        for record in records:
            # Extract genera from cultural notes
            genera_involved = []
            if record[4]:  # cultural_notes
                genera_match = re.search(r'Genera: ([^|]+)', record[4])
                if genera_match:
                    genera_involved = [g.strip() for g in genera_match.group(1).split(',')]
            
            # Extract price and availability
            price = None
            availability = None
            if record[4]:  # cultural_notes
                price_match = re.search(r'Price: ([^|]+)', record[4])
                if price_match:
                    price = price_match.group(1).strip()
                
                avail_match = re.search(r'Availability: ([^|]+)', record[4])
                if avail_match:
                    availability = avail_match.group(1).strip()
            
            cross_data = {
                'id': record[0],
                'cross_name': record[1],
                'genera_involved': genera_involved,
                'parentage': record[2],
                'description': record[3],
                'parent_images': [{'url': record[5], 'alt_text': 'Parent image', 'title': record[1]}] if record[5] else [],
                'price': price,
                'availability': availability,
                'extracted_at': record[6].isoformat() if record[6] else None
            }
            crosses.append(cross_data)
        
        return jsonify({
            'success': True,
            'crosses': crosses,
            'total': len(crosses)
        })
        
    except Exception as e:
        logger.error(f"❌ Error fetching intergeneric crosses: {e}")
        return jsonify({
            'success': False,
            'error': str(e),
            'crosses': []
        })

# Add API endpoints for Gary Yong Gee widget demo
@app.route('/api/gary-search')
def gary_search():
    """Simulated search endpoint for Gary's widget demo"""
    query = request.args.get('q', '').lower()
    
    # Mock data for demo purposes
    mock_results = [
        {
            'id': '1',
            'scientific_name': 'Cattleya labiata',
            'common_name': 'Pink Cattleya',
            'image_url': '/api/drive-photo/185MlwyxBU8Dy6bqGdwXXPeBXTlhg5M0I',
            'description': 'Beautiful pink orchid from Brazil',
            'flowering_season': 'Fall',
            'cultural_notes': 'Bright light, warm temperatures'
        },
        {
            'id': '2', 
            'scientific_name': 'Phalaenopsis amabilis',
            'common_name': 'White Moth Orchid',
            'image_url': '/api/drive-photo/1142ajwZe7_LbGt-BPy-HqVkLpNczcfZY',
            'description': 'Classic white moth orchid',
            'flowering_season': 'Winter-Spring',
            'cultural_notes': 'Low light, warm temperatures, regular watering'
        }
    ]
    
    if query:
        # Filter results based on query
        filtered = [r for r in mock_results if query in r['scientific_name'].lower() or query in r['common_name'].lower()]
        return jsonify({'results': filtered})
    
    return jsonify({'results': mock_results})

@app.route('/api/gary-phenology')
def gary_phenology():
    """Simulated phenology endpoint for Gary's widget demo"""
    taxon = request.args.get('taxon', 'Cattleya')
    
    # Mock phenology data
    mock_data = {
        'taxon': taxon,
        'flowering_periods': [
            {'month': 'January', 'percentage': 5},
            {'month': 'February', 'percentage': 10},
            {'month': 'March', 'percentage': 25},
            {'month': 'April', 'percentage': 45},
            {'month': 'May', 'percentage': 30},
            {'month': 'June', 'percentage': 15},
            {'month': 'July', 'percentage': 5},
            {'month': 'August', 'percentage': 5},
            {'month': 'September', 'percentage': 15},
            {'month': 'October', 'percentage': 35},
            {'month': 'November', 'percentage': 25},
            {'month': 'December', 'percentage': 10}
        ]
    }
    
    return jsonify(mock_data)

@app.route('/api/gary-map-data')
def gary_map_data():
    """Simulated map data endpoint for Gary's widget demo"""
    privacy_mode = request.args.get('privacy', 'public')
    
    if privacy_mode == 'private':
        # Show detailed location data for partners
        mock_data = {
            'orchids': [
                {
                    'id': '1',
                    'lat': 34.0522,
                    'lng': -118.2437,
                    'name': 'Cattleya labiata',
                    'location': 'Los Angeles, CA',
                    'elevation': '100m'
                },
                {
                    'id': '2',
                    'lat': 37.7749,
                    'lng': -122.4194,
                    'name': 'Phalaenopsis amabilis',
                    'location': 'San Francisco, CA',
                    'elevation': '50m'
                }
            ]
        }
    else:
        # Show generalized locations for public
        mock_data = {
            'orchids': [
                {
                    'id': '1',
                    'lat': 34.0,
                    'lng': -118.0,
                    'name': 'Cattleya labiata',
                    'location': 'Southern California',
                    'elevation': '~100m'
                },
                {
                    'id': '2',
                    'lat': 37.7,
                    'lng': -122.4,
                    'name': 'Phalaenopsis amabilis',
                    'location': 'Northern California',
                    'elevation': '~50m'
                }
            ]
        }
    
    return jsonify(mock_data)

# Add Orchid Mahjong demo route
@app.route('/orchid-mahjong-demo')
def orchid_mahjong_demo():
    """Standalone Orchid Mahjong Solitaire demo page"""
    return render_template('standalone/orchid-mahjong-demo.html')

@app.route('/orchid-mahjong-demo-fixed')
def orchid_mahjong_demo_fixed():
    """Fixed Orchid Mahjong Control Panel - No JavaScript Errors"""
    return render_template('standalone/orchid-mahjong-demo-fixed.html')

@app.route('/api/mahjong/orchid-images')
def get_mahjong_orchid_images():
    """Get orchid images for Mahjong tiles"""
    try:
        # Get orchids with images from database
        orchids = OrchidRecord.query.filter(
            or_(
                OrchidRecord.image_filename.isnot(None),
                OrchidRecord.google_drive_id.isnot(None),
                OrchidRecord.image_url.isnot(None)
            )
        ).limit(20).all()
        
        orchid_tiles = []
        for orchid in orchids:
            # Try to get an image URL from various possible fields
            image_url = url_for('static', filename='images/orchid_placeholder.svg')
            
            # Check if there's a Google Drive file ID or image filename
            if orchid.google_drive_id:
                image_url = url_for('get_drive_photo', file_id=orchid.google_drive_id)
            elif orchid.image_filename:
                image_url = url_for('static', filename=f'uploads/{orchid.image_filename}')
            elif orchid.image_url:
                image_url = orchid.image_url
            
            tile_data = {
                'id': orchid.id,
                'name': orchid.scientific_name or orchid.display_name or 'Unknown Orchid',
                'image_url': image_url,
                'backup_image': url_for('static', filename='images/orchid_placeholder.svg'),
                'genus': orchid.genus or 'Unknown',
                'species': orchid.species or 'Unknown',
                'description': orchid.ai_description or f'Beautiful {orchid.genus or "orchid"} specimen',
                'growing_conditions': orchid.cultural_notes or 'Moderate light and humidity'
            }
            orchid_tiles.append(tile_data)
        
        # If we don't have enough real orchid images, add some fallbacks
        if len(orchid_tiles) < 16:
            fallback_orchids = [
                {'name': 'Phalaenopsis amabilis', 'genus': 'Phalaenopsis', 'description': 'Moth Orchid - Classic white blooms'},
                {'name': 'Cattleya trianae', 'genus': 'Cattleya', 'description': 'Queen of Orchids - Purple and white'},
                {'name': 'Dendrobium nobile', 'genus': 'Dendrobium', 'description': 'Noble Dendrobium - Pink flowers'},
                {'name': 'Paphiopedilum callosum', 'genus': 'Paphiopedilum', 'description': 'Slipper Orchid - Unique pouch'},
                {'name': 'Vanda coerulea', 'genus': 'Vanda', 'description': 'Blue Vanda - Rare blue orchid'},
                {'name': 'Oncidium Sharry Baby', 'genus': 'Oncidium', 'description': 'Dancing Lady - Sweet fragrance'},
                {'name': 'Cymbidium eburneum', 'genus': 'Cymbidium', 'description': 'Boat Orchid - Long-lasting blooms'},
                {'name': 'Masdevallia veitchiana', 'genus': 'Masdevallia', 'description': 'Kite Orchid - Vivid orange-red'}
            ]
            
            for i, fallback in enumerate(fallback_orchids):
                if len(orchid_tiles) < 16:
                    orchid_tiles.append({
                        'id': f'fallback_{i}',
                        'name': fallback['name'],
                        'image_url': url_for('static', filename='images/orchid_placeholder.svg'),
                        'backup_image': url_for('static', filename='images/orchid_placeholder.svg'),
                        'genus': fallback['genus'],
                        'species': fallback['name'].split()[-1] if ' ' in fallback['name'] else '',
                        'description': fallback['description'],
                        'growing_conditions': 'Varies by species - research specific requirements'
                    })
        
        return jsonify({
            'success': True,
            'count': len(orchid_tiles),
            'tiles': orchid_tiles
        })
        
    except Exception as e:
        logger.error(f"Error getting Mahjong orchid images: {e}")
        return jsonify({
            'success': False,
            'error': str(e),
            'tiles': []
        }), 500

# Load philosophy quiz system
# Create mock objects to prevent errors (always available)
class MockWidgetHub:
    def get_user_session(self): return {}
    def manage_favorites(self, action, orchid_id=None, orchid_data=None): return {'favorites': [], 'count': 0}
    def get_smart_recommendations(self, widget_type): return {'next_widgets': [], 'suggested_actions': []}
    def track_exploration_progress(self, data): return {'progress': {}, 'new_achievements': []}
    def get_unified_dashboard_data(self): return {}

class MockMobileOptimizer:
    def detect_device_type(self): return 'desktop'
    def get_mobile_optimized_config(self, widget_type, device_type=None): return {'device_type': 'desktop'}
    def get_touch_controls_javascript(self, widget_type): return '// Mobile controls not available'
    def get_mobile_css(self): return '/* Mobile styles not available */'

class MockCollectionHub:
    def get_collection_dashboard_data(self): return {'collection': {'owned_orchids': [], 'wishlist': []}, 'statistics': {}, 'care_reminders': []}
    def get_personalized_recommendations(self): return {'recommendations': []}
    def add_to_collection(self, orchid_id, collection_type='owned', care_data=None): return {'success': False, 'error': 'Collection system not available'}
    def log_care_activity(self, orchid_id, care_type, notes=''): return {'success': False, 'error': 'Collection system not available'}
    def get_care_reminders(self): return []

# Always create mock objects (can be overridden by real imports later)
widget_hub = MockWidgetHub()
mobile_optimizer = MockMobileOptimizer()
collection_hub = MockCollectionHub()

try:
    import philosophy_quiz_system
    logger.info("✅ Philosophy Quiz system loaded successfully")
except ImportError as e:
    logger.warning(f"⚠️ Philosophy Quiz system not available: {e}")
    
    def track_widget_interaction(widget_name, action, **context_data):
        pass
    
    def get_enhanced_widget_data(widget_type, **kwargs):
        return {'widget_type': widget_type, 'data': 'integration_not_available'}

# Start comprehensive image monitoring every 30 seconds
try:
    monitoring_thread = start_image_monitoring()
    logger.info("🔍 Started comprehensive image monitoring every 30 seconds")
except Exception as e:
    logger.error(f"Failed to start monitoring: {e}")

# Register the Atlas blueprint
app.register_blueprint(atlas_bp)

# Register weather habitat comparison routes
register_weather_habitat_routes(app)

# Register GBIF integration routes
app.register_blueprint(gbif_bp)

# Register AI orchid identification routes
app.register_blueprint(ai_orchid_bp)

# Register geographic mapping routes
app.register_blueprint(geo_mapping_bp)

# Register enhanced mapping analytics routes
app.register_blueprint(enhanced_mapping_bp)

# Register Citizen Science Platform
app.register_blueprint(citizen_science_bp)

# Register Google Drive Import System
app.register_blueprint(drive_import_bp)

# Register Mycorrhizal Research Platform
from mycorrhizal_research_system import mycorrhizal_bp
app.register_blueprint(mycorrhizal_bp)

# Register Climate Research Platform
from climate_research_system import climate_research_bp
app.register_blueprint(climate_research_bp)

# Register Global Analysis Platform
from global_orchid_climate_analysis import global_analysis_bp
app.register_blueprint(global_analysis_bp)

# Register Unified Climate Command Center
from unified_climate_command_center import command_center_bp
app.register_blueprint(command_center_bp)

# Register AI Research Assistant
from ai_research_assistant import ai_research_bp
app.register_blueprint(ai_research_bp)

# Register Research Data Manager
from research_data_manager import research_data_bp
app.register_blueprint(research_data_bp)

# Register Literature Search System
from literature_search_system import literature_bp
app.register_blueprint(literature_bp)

# Register Historical Climate System
from historical_climate_system import historical_climate_bp
app.register_blueprint(historical_climate_bp)

# Register Mycorrhizal Network Monitor (Skipped - conflicts with mycorrhizal_research_system)
# from mycorrhizal_network_monitor import mycorrhizal_bp  # DISABLED: Conflicts with existing mycorrhizal_bp
# app.register_blueprint(mycorrhizal_bp)  # DISABLED: Blueprint already registered above

# Register AI Research Director (Autonomous Climate Commander)
from ai_research_director import research_director_bp
app.register_blueprint(research_director_bp)

# Register Admin Orchid Approval routes
app.register_blueprint(orchid_approval_bp)

# Register Pattern Analysis routes
app.register_blueprint(pattern_analysis_bp)

# Register OrchidStein Lab routes
app.register_blueprint(lab_bp)

# Register Globe Weather Widget routes
from globe_weather_routes import globe_weather_bp
app.register_blueprint(globe_weather_bp)

# Register Orchid-Mycorrhizal Fungi Mapping routes
from orchid_mycorrhizal_routes import mycorrhizal_map_bp
app.register_blueprint(mycorrhizal_map_bp)

# Register Botany Lab Stats & Imports system
from botany_lab_stats import register_botany_lab_routes
register_botany_lab_routes(app)

# Professor BloomBot system removed - Replit Agent is now the primary AI interface

# 35th Parallel Globe and BloomBot routes removed
# Geographic analysis now handled by Replit Agent directly

@app.route('/test_gary_scraper')
def test_gary_scraper():
    """Test Gary Yong Gee scraper with limited scope"""
    try:
        from comprehensive_orchid_scraper import ComprehensiveOrchidScraper
        
        scraper = ComprehensiveOrchidScraper()
        
        # Test with just one genus for verification
        print("🧪 Testing Gary Yong Gee scraper with single genus...")
        test_url = "https://orchids.yonggee.name/genera/aa"
        
        genus_results = scraper.scrape_gary_genus_page(test_url, "aa")
        
        db.session.commit()
        
        return f"""
        <h2>Gary Yong Gee Scraper Test Results</h2>
        <p><strong>Test URL:</strong> {test_url}</p>
        <p><strong>Processed:</strong> {genus_results['processed']}</p>
        <p><strong>Errors:</strong> {genus_results['errors']}</p>
        <p><strong>Skipped:</strong> {genus_results['skipped']}</p>
        <p><a href="/admin">Back to Admin</a></p>
        """
        
    except Exception as e:
        return f"<h2>Test Error</h2><p>{str(e)}</p><p><a href='/admin'>Back to Admin</a></p>"

@app.route('/analyze_all_photos')
def analyze_all_photos():
    """Analyze all orchid photos for names and metadata"""
    try:
        from photo_analysis_system import PhotoAnalysisSystem
        
        analyzer = PhotoAnalysisSystem()
        
        print("🔍 Starting comprehensive photo analysis...")
        results = analyzer.run_comprehensive_analysis()
        
        db.session.commit()
        
        return f"""
        <h2>🔍 Photo Analysis Results</h2>
        <p><strong>Photos Analyzed:</strong> {results['analyzed']}</p>
        <p><strong>Names Improved:</strong> {results['improved_names']}</p>
        <p><strong>Metadata Extracted:</strong> {results['extracted_metadata']}</p>
        <p><strong>AI Analyzed:</strong> {results['ai_analyzed']}</p>
        <p><strong>Errors:</strong> {results['errors']}</p>
        <p><strong>Analysis:</strong> Filename parsing, AI vision, EXIF extraction</p>
        <p><a href="/admin">Back to Admin</a></p>
        """
        
    except Exception as e:
        return f"<h2>Analysis Error</h2><p>{str(e)}</p><p><a href='/admin'>Back to Admin</a></p>"

@app.route('/test_international_scrapers')
def test_international_scrapers():
    """Test comprehensive international orchid scrapers"""
    try:
        from international_orchid_scraper import InternationalOrchidScraper
        
        scraper = InternationalOrchidScraper()
        
        print("🌍 Testing international orchid scrapers...")
        results = scraper.run_comprehensive_international_collection()
        
        db.session.commit()
        
        return f"""
        <h2>🌍 International Orchid Collection Results</h2>
        <p><strong>Total Processed:</strong> {results['total_processed']}</p>
        <p><strong>New Countries:</strong> {len(results['new_countries'])}</p>
        <p><strong>New Genera:</strong> {len(results['new_genera'])}</p>
        <p><strong>Errors:</strong> {results['errors']}</p>
        <p><strong>Sources:</strong> Internet Orchid Species, Singapore Botanic, Australian, European, South American</p>
        <p><a href="/admin">Back to Admin</a></p>
        """
        
    except Exception as e:
        return f"<h2>International Test Error</h2><p>{str(e)}</p><p><a href='/admin'>Back to Admin</a></p>"

@app.route('/test_enhanced_flowering_collection')
def test_enhanced_flowering_collection():
    """Test enhanced flowering and geographic collection"""
    try:
        from enhanced_flowering_geographic_scraper import FloweringGeographicScraper
        
        scraper = FloweringGeographicScraper()
        
        print("🌸📍 Testing enhanced flowering & geographic collection...")
        results = scraper.run_enhanced_collection()
        
        db.session.commit()
        
        return f"""
        <h2>🌸📍 Enhanced Flowering & Geographic Collection Results</h2>
        <p><strong>Total Processed:</strong> {results['total_processed']}</p>
        <p><strong>With Flowering Dates:</strong> {results['with_flowering_dates']}</p>
        <p><strong>With Coordinates:</strong> {results['with_coordinates']}</p>
        <p><strong>With BOTH (Target):</strong> {results['with_both']}</p>
        <p><strong>Endemic Species:</strong> {results['endemic_species']}</p>
        <p><strong>Cross-Latitude Candidates:</strong> {results['cross_latitude_candidates']}</p>
        <p><a href="/database_metadata_report">View Full Database Report</a></p>
        <p><a href="/admin">Back to Admin</a></p>
        """
        
    except Exception as e:
        return f"<h2>Enhanced Collection Test Error</h2><p>{str(e)}</p><p><a href='/admin'>Back to Admin</a></p>"

@app.route('/database_metadata_report')
def database_metadata_report():
    """Show comprehensive database metadata completeness report"""
    try:
        from database_metadata_tracker import DatabaseMetadataTracker
        
        tracker = DatabaseMetadataTracker()
        report = tracker.generate_progress_report()
        
        return f"""
        <h2>📊 Database Metadata Completeness Report</h2>
        <pre>{report}</pre>
        <p><a href="/test_enhanced_flowering_collection">Run Enhanced Collection</a></p>
        <p><a href="/admin">Back to Admin</a></p>
        """
        
    except Exception as e:
        return f"<h2>Database Report Error</h2><p>{str(e)}</p><p><a href='/admin'>Back to Admin</a></p>"

@app.route('/test_ron_parsons')
def test_ron_parsons():
    """Test Ron Parsons Flickr scraper"""
    try:
        from ron_parsons_scraper import RonParsonsOrchidScraper
        
        scraper = RonParsonsOrchidScraper()
        
        print("🌸 Testing Ron Parsons scraper...")
        results = scraper.scrape_flickr_photostream()
        
        db.session.commit()
        
        return f"""
        <h2>🌸 Ron Parsons Scraper Test Results</h2>
        <p><strong>Target:</strong> 118,952+ photos from world's leading orchid photographer</p>
        <p><strong>Processed:</strong> {results['processed']}</p>
        <p><strong>Errors:</strong> {results['errors']}</p>
        <p><strong>Skipped:</strong> {results['skipped']}</p>
        <p><strong>Source:</strong> Flickr photostream</p>
        <p><a href="/admin">Back to Admin</a></p>
        """
        
    except Exception as e:
        return f"<h2>Test Error</h2><p>{str(e)}</p><p><a href='/admin'>Back to Admin</a></p>"

@app.route('/test_iospe')
def test_iospe():
    """Test IOSPE scraper"""
    try:
        from comprehensive_orchid_scraper import ComprehensiveOrchidScraper
        
        scraper = ComprehensiveOrchidScraper()
        
        print("🌍 Testing IOSPE scraper...")
        results = scraper.scrape_iospe_comprehensive()
        
        db.session.commit()
        
        return f"""
        <h2>🌍 IOSPE Scraper Test Results</h2>
        <p><strong>Target:</strong> 25,996+ species from world's largest orchid database</p>
        <p><strong>Processed:</strong> {results['processed']}</p>
        <p><strong>Errors:</strong> {results['errors']}</p>
        <p><strong>Skipped:</strong> {results['skipped']}</p>
        <p><strong>Source:</strong> orchidspecies.com</p>
        <p><a href="/admin">Back to Admin</a></p>
        """
        
    except Exception as e:
        return f"<h2>Test Error</h2><p>{str(e)}</p><p><a href='/admin'>Back to Admin</a></p>"

@app.route('/test_filename_parser')
def test_filename_parser():
    """Test enhanced filename parser"""
    try:
        from filename_parser import parse_orchid_filename, analyze_filename_for_orchid_name
        
        # Test filenames
        test_files = [
            "Cattleya_warscewiczii_alba.jpg",
            "phal_amabilis_var_rosenstromii.jpg", 
            "Dendrobium-nobile-cooksonii.JPG",
            "Ron_Parsons_Masdevallia_veitchiana_2024.jpg",
            "DSC_Oncidium_sphacelatum_Ecuador.jpg"
        ]
        
        results = []
        for filename in test_files:
            parsed = parse_orchid_filename(filename)
            analysis = analyze_filename_for_orchid_name(filename)
            
            results.append({
                'filename': filename,
                'parsed': parsed,
                'analysis': analysis
            })
        
        html = "<h2>🧠 Filename Parser Test Results</h2>"
        for result in results:
            html += f"""
            <div style="border: 1px solid #ccc; margin: 10px; padding: 10px;">
                <h3>{result['filename']}</h3>
                <p><strong>Genus:</strong> {result['parsed'].get('genus', 'None')}</p>
                <p><strong>Species:</strong> {result['parsed'].get('species', 'None')}</p>
                <p><strong>Full Name:</strong> {result['analysis'].get('full_parsed_name', 'None')}</p>
                <p><strong>Confidence:</strong> {result['analysis'].get('confidence', 0.0):.2f}</p>
                <p><strong>Method:</strong> {result['analysis'].get('parsing_method', 'None')}</p>
            </div>
            """
        
        html += '<p><a href="/admin">Back to Admin</a></p>'
        return html
        
    except Exception as e:
        return f"<h2>Test Error</h2><p>{str(e)}</p><p><a href='/admin'>Back to Admin</a></p>"

# Register blueprints
app.register_blueprint(processing_bp)
app.register_blueprint(photo_editor_bp)

# Register advanced gallery system
from advanced_gallery_routes import advanced_gallery_bp
app.register_blueprint(advanced_gallery_bp, url_prefix='/advanced_gallery')

# Start the orchid record scheduler
from scheduler import start_orchid_scheduler, get_scheduler_status
start_orchid_scheduler()

@app.route('/admin/scheduler-status')
def scheduler_status():
    """Display scheduler status and controls"""
    try:
        status = get_scheduler_status()
        return render_template('admin/scheduler_status.html', status=status)
    except Exception as e:
        return f"<h2>Scheduler Status Error</h2><p>{str(e)}</p><p><a href='/admin'>Back to Admin</a></p>"

@app.route('/stats')
def detailed_statistics():
    """Detailed statistics page showing comprehensive orchid collection data"""
    try:
        from orchid_statistics import get_homepage_statistics, get_genus_statistics
        
        # Get comprehensive statistics
        stats = get_homepage_statistics()
        genus_stats = get_genus_statistics()
        
        return render_template('detailed_statistics.html', stats=stats, genus_stats=genus_stats)
        
    except Exception as e:
        logger.error(f"Error loading detailed statistics: {e}")
        return render_template('error.html', error="Could not load statistics system"), 500

@app.route('/stats/genus/<genus>')
def genus_detail_stats(genus):
    """Show detailed statistics for a specific genus"""
    try:
        from orchid_statistics import orchid_stats
        
        # Get genus-specific details
        genus_details = orchid_stats.get_genus_details(genus)
        
        # Get orchids in this genus for display
        orchids = OrchidRecord.query.filter(
            func.lower(OrchidRecord.genus) == genus.lower()
        ).limit(50).all()
        
        return render_template('genus_statistics.html', 
                             genus_details=genus_details, 
                             orchids=orchids)
        
    except Exception as e:
        logger.error(f"Error loading genus statistics for {genus}: {e}")
        return render_template('error.html', error=f"Could not load statistics for genus {genus}"), 500

@app.route('/35th-parallel-globe')
def parallel_35_globe():
    """35th Parallel Educational Globe System"""
    try:
        # Get orchid species data for 35th parallel regions
        orchid_data = get_35th_parallel_orchids()
        
        # Create widget data for the enhanced globe system
        widget_data = {
            'widget_id': 'parallel-35-globe',
            'orchids': orchid_data,
            'countries_with_orchids': get_countries_on_35th_parallel(),
            'focus_latitude': 35.0,
            'enabled_features': ['orchid-hotspots', '35th-parallel', 'climate-zones']
        }
        
        return render_template('widgets/enhanced_globe_widget.html', widget_data=widget_data)
        
    except Exception as e:
        logger.error(f"Error loading 35th parallel globe: {e}")
        return render_template('error.html', error="Could not load 35th parallel globe system"), 500

@app.route('/satellite-earth-globe')
def satellite_earth_globe():
    """🛰️ Satellite Earth Globe - Real Earth from Space with Orchid Data"""
    try:
        return render_template('widgets/satellite_earth_globe.html')
        
    except Exception as e:
        logger.error(f"Error loading satellite Earth globe: {e}")
        return render_template('error.html', error="Could not load satellite Earth globe"), 500

def get_35th_parallel_orchids():
    """Get orchid species found along the 35th parallel"""
    try:
        # Query orchids with geographic data near 35th parallel
        orchids = OrchidRecord.query.filter(
            OrchidRecord.photo_gps_coordinates.isnot(None)
        ).limit(20).all()
        
        # Add some representative 35th parallel species
        parallel_species = [
            {
                'display_name': 'Platanthera ciliaris',
                'scientific_name': 'Orange Fringed Orchid',
                'region': 'North Carolina, USA',
                'image_url': '/static/images/orchid_placeholder.svg',
                'latitude': 35.2,
                'longitude': -80.8,
                'conservation_status': 'Native',
                'description': 'Native terrestrial orchid found in southeastern US wetlands'
            },
            {
                'display_name': 'Ophrys apifera',
                'scientific_name': 'Bee Orchid',
                'region': 'Mediterranean Basin',
                'image_url': '/static/images/orchid_placeholder.svg',
                'latitude': 35.1,
                'longitude': 14.5,
                'conservation_status': 'Protected',
                'description': 'Remarkable orchid that mimics bees for pollination'
            },
            {
                'display_name': 'Cypripedium californicum',
                'scientific_name': 'California Lady Slipper',
                'region': 'Northern California',
                'image_url': '/static/images/orchid_placeholder.svg',
                'latitude': 35.8,
                'longitude': -121.4,
                'conservation_status': 'Rare',
                'description': 'Endangered North American native requiring cool, wet conditions'
            }
        ]
        
        return parallel_species + [{'display_name': o.display_name or o.scientific_name, 
                                  'scientific_name': o.scientific_name,
                                  'region': 'Unknown',
                                  'image_url': f"/api/drive-photo/{o.google_drive_id}" if o.google_drive_id else '/static/images/orchid_placeholder.svg'} 
                                 for o in orchids[:10]]
        
    except Exception as e:
        logger.error(f"Error getting 35th parallel orchids: {e}")
        return []

def get_countries_on_35th_parallel():
    """Get countries crossed by the 35th parallel"""
    return [
        'United States', 'Turkey', 'Cyprus', 'Syria', 'Lebanon', 'Iraq', 'Iran', 
        'Afghanistan', 'Pakistan', 'India', 'China', 'Japan', 'South Korea',
        'Morocco', 'Algeria', 'Tunisia', 'Libya'
    ]

@app.route('/scientific-research')
def scientific_proof_dashboard():
    """Scientific Research Validation Dashboard - Proof of Concept"""
    return render_template('research/scientific_proof_dashboard.html')

@app.route('/phenotype-analysis')
def phenotype_analysis():
    """Phenotypic variation analysis page"""
    try:
        from phenotype_analyzer import get_analyzable_species
        
        # Get species with multiple specimens for analysis
        species_list = get_analyzable_species(min_specimens=3)
        
        return render_template('phenotype_analysis.html', species_list=species_list)
        
    except Exception as e:
        logger.error(f"Error loading phenotype analysis page: {e}")
        return render_template('error.html', error="Could not load phenotypic analysis system"), 500

@app.route('/api/analyze-phenotype', methods=['POST'])
def api_analyze_phenotype():
    """API endpoint for phenotypic variation analysis"""
    try:
        data = request.get_json()
        genus = data.get('genus')
        species = data.get('species')
        
        if not genus or not species:
            return jsonify({'error': 'Genus and species are required'}), 400
        
        # Import and run analysis
        from phenotype_analyzer import analyze_species_variations
        
        logger.info(f"Starting phenotypic analysis for {genus} {species}")
        analysis = analyze_species_variations(genus, species)
        
        if not analysis:
            return jsonify({'error': 'Analysis failed or insufficient data'}), 400
        
        # Get specimens for display
        specimens = OrchidRecord.query.filter(
            func.lower(OrchidRecord.genus) == genus.lower(),
            func.lower(OrchidRecord.species) == species.lower(),
            OrchidRecord.google_drive_id.isnot(None)
        ).limit(20).all()
        
        # Format response
        response = {
            'genus': analysis.genus,
            'species': analysis.species,
            'total_specimens': analysis.total_specimens,
            'research_notes': analysis.research_notes,
            'specimens': [
                {
                    'id': s.id,
                    'display_name': s.display_name,
                    'google_drive_id': s.google_drive_id,
                    'country': s.country,
                    'region': s.region
                } for s in specimens
            ],
            'variations': [
                {
                    'trait_name': v.trait_name,
                    'variation_type': v.variation_type,
                    'description': v.description,
                    'confidence': v.confidence,
                    'specimens_affected': v.specimens_affected
                } for v in analysis.variations
            ],
            'morphological_summary': analysis.morphological_summary,
            'mutation_indicators': analysis.mutation_indicators,
            'adaptation_patterns': analysis.adaptation_patterns
        }
        
        logger.info(f"Phenotypic analysis completed: {len(analysis.variations)} variations found")
        return jsonify(response)
        
    except Exception as e:
        logger.error(f"Error in phenotypic analysis API: {e}")
        return jsonify({'error': f'Analysis failed: {str(e)}'}), 500

@app.route('/')
def index():
    """Homepage with enhanced orchid of the day and advanced features"""
    try:
        # AUTOMATIC FEATURED ORCHID PROTECTION - PREVENTS BAD ORCHIDS FROM SHOWING
        from data_integrity_safeguards import auto_prevent_bad_featured_selection
        auto_prevent_bad_featured_selection()
        
        # Get enhanced orchid of the day
        from enhanced_orchid_of_day import ValidatedOrchidOfDay
        enhanced_system = ValidatedOrchidOfDay()
        orchid_of_day_enhanced = enhanced_system.get_enhanced_orchid_of_day()
        
        # Fallback to basic orchid of day if enhanced fails
        orchid_of_day = orchid_of_day_enhanced['orchid'] if orchid_of_day_enhanced else get_orchid_of_the_day()
        
        # Get recent uploads with Google Drive images only (to match working Orchid of Day)
        recent_orchids = []
        try:
            recent_orchids = OrchidRecord.query.filter(
                OrchidRecord.google_drive_id.isnot(None)
            ).order_by(OrchidRecord.created_at.desc()).limit(6).all()
        except Exception as e:
            logger.error(f"Error fetching recent orchids: {str(e)}")
            db.session.rollback()
        
        # Get featured orchids with Google Drive images only
        featured_orchids = []
        try:
            featured_orchids = OrchidRecord.query.filter(
                OrchidRecord.is_featured == True,
                OrchidRecord.google_drive_id.isnot(None)
            ).limit(4).all()
        except Exception as e:
            logger.error(f"Error fetching featured orchids: {str(e)}")
            db.session.rollback()
        
        # Get comprehensive statistics
        stats = {}
        try:
            from orchid_statistics import get_homepage_statistics
            stats = get_homepage_statistics()
        except Exception as e:
            logger.error(f"Error loading statistics: {e}")
            # Get live counts as fallback
            try:
                total_orchids = db.session.query(func.count(OrchidRecord.id)).scalar()
                total_genera = db.session.query(func.count(func.distinct(OrchidRecord.genus))).filter(OrchidRecord.genus.isnot(None)).scalar()
                photos_count = db.session.query(func.count(OrchidRecord.id)).filter(
                    OrchidRecord.image_url.isnot(None),
                    OrchidRecord.image_url != '',
                    ~OrchidRecord.image_url.like('%placeholder%')
                ).scalar()
                stats = {
                    'total_orchids': total_orchids or 4164,
                    'total_genera': total_genera or 396, 
                    'total_species': 2053,
                    'photos_available': photos_count or 1337,
                    'genus_breakdown': []
                }
            except:
                stats = {
                    'total_orchids': 4164,
                    'total_genera': 396,
                    'total_species': 2053,
                    'photos_available': 1337,
                    'genus_breakdown': []
                }
        
        # STEALTH CONFIGURATION - BOARD PROTECTION
        try:
            from stealth_config import stealth_manager
            user_level = stealth_manager.get_user_access_level()
            
            # Serve board-friendly version for public access
            if user_level == 'public':
                return render_template('board_friendly_homepage.html')
        except ImportError:
            pass  # Continue with full platform if stealth config not available
        
        return render_template('index.html',
                             orchid_of_day=orchid_of_day,
                             orchid_of_day_enhanced=orchid_of_day_enhanced,
                             recent_orchids=recent_orchids,
                             featured_orchids=featured_orchids,
                             stats=stats)
    
    except Exception as e:
        logger.error(f"Homepage error: {str(e)}")
        db.session.rollback()
        # Return minimal homepage on error with live counts if possible
        try:
            total_orchids = db.session.query(func.count(OrchidRecord.id)).scalar()
            photos_count = db.session.query(func.count(OrchidRecord.id)).filter(
                OrchidRecord.image_url.isnot(None),
                OrchidRecord.image_url != '',
                ~OrchidRecord.image_url.like('%placeholder%')
            ).scalar()
            fallback_stats = {
                'total_orchids': total_orchids or 4164,
                'total_genera': 396,
                'total_species': 2053,
                'photos_available': photos_count or 1337,
                'genus_breakdown': []
            }
        except:
            fallback_stats = {
                'total_orchids': 4164,
                'total_genera': 396,
                'total_species': 2053,
                'photos_available': 1337,
                'genus_breakdown': []
            }
        return render_template('index.html',
                             orchid_of_day=None,
                             orchid_of_day_enhanced=None,
                             recent_orchids=[],
                             featured_orchids=[],
                             stats=fallback_stats)

@app.route('/api/baker-extrapolation/analyze')
def analyze_baker_extrapolation():
    """API endpoint to analyze Baker culture extrapolation opportunities"""
    try:
        from baker_extrapolation_system import baker_extrapolation
        
        report = baker_extrapolation.generate_extrapolation_report()
        return jsonify({
            'success': True,
            'report': report,
            'timestamp': datetime.now().isoformat()
        })
        
    except Exception as e:
        logger.error(f"Error in Baker extrapolation analysis: {str(e)}")
        return jsonify({'success': False, 'error': str(e)})

@app.route('/api/baker-extrapolation/batch-genus/<genus_name>')
def batch_extrapolate_genus(genus_name):
    """API endpoint to batch extrapolate culture data for a genus"""
    try:
        from baker_extrapolation_system import baker_extrapolation
        
        limit = request.args.get('limit', 50, type=int)
        results = baker_extrapolation.batch_extrapolate_genus(genus_name, limit)
        
        return jsonify({
            'success': True,
            'results': results,
            'timestamp': datetime.now().isoformat()
        })
        
    except Exception as e:
        logger.error(f"Error in batch genus extrapolation: {str(e)}")
        return jsonify({'success': False, 'error': str(e)})

@app.route('/api/baker-extrapolation/by-climate/<climate>')
def extrapolate_by_climate(climate):
    """API endpoint to extrapolate by climate preference"""
    try:
        from baker_extrapolation_system import baker_extrapolation
        
        limit = request.args.get('limit', 100, type=int)
        results = baker_extrapolation.extrapolate_by_climate(climate, limit)
        
        return jsonify({
            'success': True,
            'results': results,
            'timestamp': datetime.now().isoformat()
        })
        
    except Exception as e:
        logger.error(f"Error in climate extrapolation: {str(e)}")
        return jsonify({'success': False, 'error': str(e)})

@app.route('/api/baker-extrapolation/regional-analysis')
def regional_extrapolation_analysis():
    """API endpoint for endemic region extrapolation analysis"""
    try:
        from endemic_region_extrapolator import endemic_extrapolator
        
        report = endemic_extrapolator.generate_regional_extrapolation_report()
        return jsonify({
            'success': True,
            'report': report,
            'timestamp': datetime.now().isoformat()
        })
        
    except Exception as e:
        logger.error(f"Error in regional analysis: {str(e)}")
        return jsonify({'success': False, 'error': str(e)})

@app.route('/api/baker-extrapolation/by-region/<region>')
def extrapolate_by_region(region):
    """API endpoint to extrapolate by endemic region"""
    try:
        from endemic_region_extrapolator import endemic_extrapolator
        
        limit = request.args.get('limit', 50, type=int)
        results = endemic_extrapolator.batch_extrapolate_region(region, limit)
        
        return jsonify({
            'success': True,
            'results': results,
            'timestamp': datetime.now().isoformat()
        })
        
    except Exception as e:
        logger.error(f"Error in regional extrapolation: {str(e)}")
        return jsonify({'success': False, 'error': str(e)})

@app.route('/api/ai/baker-weather-advice', methods=['POST'])
def baker_weather_advice():
    """API endpoint for AI-powered Baker culture weather advice"""
    try:
        data = request.get_json()
        orchid_id = data.get('orchid_id')
        weather_data = data.get('weather', {})
        forecast_data = data.get('forecast', [])
        
        if not orchid_id:
            return jsonify({'success': False, 'error': 'Orchid ID required'})
        
        # Get orchid record
        orchid = OrchidRecord.query.get(orchid_id)
        if not orchid:
            return jsonify({'success': False, 'error': 'Orchid not found'})
        
        # Generate AI advice using Baker culture data (including extrapolated)
        from orchid_ai import extrapolate_baker_culture_data
        
        # Try direct Baker advice first, then extrapolated
        advice = get_weather_based_care_advice(orchid, weather_data, forecast_data)
        
        # If no direct advice, try extrapolation
        if not advice:
            extrapolated_data = extrapolate_baker_culture_data(orchid)
            if extrapolated_data:
                # Generate weather advice based on extrapolated data
                temp_orchid = orchid
                temp_orchid.cultural_notes = f"EXTRAPOLATED BAKER DATA: {json.dumps(extrapolated_data)}"
                advice = get_weather_based_care_advice(temp_orchid, weather_data, forecast_data)
        
        if advice:
            return jsonify({
                'success': True,
                'advice': advice,
                'has_baker_data': bool(orchid.cultural_notes and 'BAKER' in orchid.cultural_notes)
            })
        else:
            return jsonify({
                'success': True,
                'advice': 'Monitor temperature and humidity. Adjust watering based on conditions.',
                'has_baker_data': False
            })
        
    except Exception as e:
        logger.error(f"Error generating Baker weather advice: {str(e)}")
        return jsonify({'success': False, 'error': 'Unable to generate advice'})

@app.route('/upload', methods=['GET', 'POST'])
def upload():
    """Handle orchid photo uploads"""
    if request.method == 'POST':
        try:
            if 'file' not in request.files:
                flash('No file selected', 'error')
                return redirect(request.url)
            
            file = request.files['file']
            if file.filename == '':
                flash('No file selected', 'error')
                return redirect(request.url)
            
            if file and file.filename and allowed_file(file.filename):
                # Generate secure filename
                original_filename = secure_filename(file.filename)
                new_filename = generate_filename(original_filename)
                
                # Save temporarily
                temp_path = os.path.join('temp', new_filename)
                os.makedirs('temp', exist_ok=True)
                file.save(temp_path)
                
                # Parse filename for orchid info
                parsed_info = parse_orchid_filename(original_filename)
                
                # Create upload record
                upload_record = UserUpload(
                    user_id=current_user.id if current_user.is_authenticated else None,
                    original_filename=original_filename,
                    uploaded_filename=new_filename,
                    file_size=os.path.getsize(temp_path),
                    mime_type=file.content_type,
                    user_notes=request.form.get('notes', ''),
                    processing_status='processing',
                    parsed_genus=parsed_info.get('genus'),
                    parsed_species=parsed_info.get('species'),
                    filename_confidence=parsed_info.get('confidence', 0.0)
                )
                db.session.add(upload_record)
                db.session.commit()
                
                # Process the image
                try:
                    # Upload to Google Drive
                    drive_file_id = upload_to_drive(temp_path, new_filename, 'Orchid_Quick_Images')
                    image_url = get_drive_file_url(drive_file_id) if drive_file_id else None
                    
                    # Analyze with AI
                    ai_result = analyze_orchid_image(temp_path)
                    
                    # Create orchid record
                    orchid = OrchidRecord()
                    orchid.display_name = ai_result.get('suggested_name', f'Unknown Orchid {upload_record.id}')
                    orchid.scientific_name = ai_result.get('scientific_name')
                    orchid.genus = ai_result.get('genus')
                    orchid.species = ai_result.get('species')
                    orchid.image_filename = new_filename
                    orchid.image_url = image_url
                    orchid.google_drive_id = drive_file_id
                    orchid.ai_description = ai_result.get('description')
                    orchid.ai_confidence = ai_result.get('confidence', 0.0)
                    orchid.ai_extracted_metadata = json.dumps(ai_result.get('metadata', {}))
                    orchid.ingestion_source = 'upload'
                    orchid.cultural_notes = request.form.get('notes', '')
                    
                    # Try to match with taxonomy
                    if orchid.scientific_name:
                        taxonomy = OrchidTaxonomy.query.filter_by(
                            scientific_name=orchid.scientific_name
                        ).first()
                        if taxonomy:
                            orchid.taxonomy_id = taxonomy.id
                    
                    db.session.add(orchid)
                    upload_record.processing_status = 'completed'
                    upload_record.orchid_id = orchid.id
                    db.session.commit()
                    
                    # Send admin notification
                    try:
                        from admin_notification_service import notification_service
                        notification_service.send_photo_submission_alert(upload_record, orchid)
                        logger.info(f"📧 Admin notification sent for upload {upload_record.id}")
                    except Exception as e:
                        logger.error(f"Failed to send admin notification: {e}")
                    
                    # Clean up temp file
                    os.remove(temp_path)
                    
                    flash('Orchid uploaded and processed successfully!', 'success')
                    return redirect(url_for('orchid_detail', id=orchid.id))
                    
                except Exception as e:
                    logger.error(f"Error processing upload: {str(e)}")
                    upload_record.processing_status = 'failed'
                    db.session.commit()
                    
                    # Send admin notification for failed upload
                    try:
                        from admin_notification_service import notification_service
                        notification_service.send_photo_submission_alert(upload_record, None)
                        logger.info(f"📧 Admin notification sent for failed upload {upload_record.id}")
                    except Exception as e:
                        logger.error(f"Failed to send admin notification: {e}")
                    
                    flash(f'Error processing image: {str(e)}', 'error')
            else:
                flash('Invalid file type. Please upload JPG, PNG, or GIF files.', 'error')
                
        except Exception as e:
            logger.error(f"Upload error: {str(e)}")
            flash(f'Upload failed: {str(e)}', 'error')
    
    return render_template('upload.html')

@app.route('/enhanced-gallery-ecosystem')
def enhanced_gallery_ecosystem():
    """Enhanced gallery with integrated ecosystem data and distribution maps"""
    try:
        # Get filter parameters
        genus_filter = request.args.get('genus', '')
        climate_filter = request.args.get('climate', '')
        growth_habit_filter = request.args.get('growth_habit', '')
        pollinator_filter = request.args.get('pollinator', '')
        region_filter = request.args.get('region', '')
        page = int(request.args.get('page', 1))
        per_page = 12  # Smaller batches for rich content
        
        # Build query
        query = OrchidRecord.query.filter(
            OrchidRecord.validation_status != 'rejected'
        )
        
        # Apply filters
        if genus_filter:
            query = query.filter(OrchidRecord.genus.ilike(f'%{genus_filter}%'))
        if climate_filter:
            query = query.filter(OrchidRecord.climate_preference.ilike(f'%{climate_filter}%'))
        if growth_habit_filter:
            query = query.filter(OrchidRecord.growth_habit.ilike(f'%{growth_habit_filter}%'))
        if pollinator_filter:
            query = query.filter(OrchidRecord.pollinator_types.any(pollinator_filter))
        if region_filter:
            query = query.filter(OrchidRecord.region.ilike(f'%{region_filter}%'))
        
        # Get paginated results
        orchids_paginated = query.order_by(OrchidRecord.created_at.desc()).paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        orchids = orchids_paginated.items
        
        # Get filter options
        all_orchids = OrchidRecord.query.filter(OrchidRecord.validation_status != 'rejected').all()
        genera = sorted(list(set([o.genus for o in all_orchids if o.genus])))
        
        # Prepare statistics
        unique_regions = set([o.region for o in orchids if o.region])
        unique_genera = set([o.genus for o in orchids if o.genus])
        climate_zones = set([o.climate_preference for o in orchids if o.climate_preference])
        
        # Prepare orchid data for map (JSON-safe)
        orchids_json = []
        for orchid in orchids:
            if orchid.decimal_latitude and orchid.decimal_longitude:
                # Fix image URL for JSON output
                image_url = orchid.image_url
                if image_url and 'gbif.org/occurrence/' in image_url:
                    if orchid.google_drive_id:
                        image_url = f"/api/drive-photo/{orchid.google_drive_id}"
                    else:
                        image_url = "/static/images/orchid_placeholder.svg"
                
                orchids_json.append({
                    'id': orchid.id,
                    'name': orchid.scientific_name or orchid.display_name,
                    'latitude': float(orchid.decimal_latitude),
                    'longitude': float(orchid.decimal_longitude),
                    'region': orchid.region,
                    'pollinators': ', '.join(orchid.pollinator_types) if orchid.pollinator_types else None,
                    'climate': orchid.climate_preference,
                    'image_url': image_url
                })
        
        import json
        orchids_json_str = json.dumps(orchids_json)
        
        return render_template('enhanced_gallery_ecosystem.html',
                             orchids=orchids,
                             orchids_json=orchids_json_str,
                             genera=genera,
                             unique_regions=unique_regions,
                             unique_genera=unique_genera,
                             climate_zones=climate_zones,
                             current_genus=genus_filter,
                             current_climate=climate_filter,
                             current_growth_habit=growth_habit_filter,
                             page=page,
                             total_pages=orchids_paginated.pages)
        
    except Exception as e:
        logger.error(f"Enhanced gallery error: {e}")
        flash('Error loading enhanced gallery', 'error')
        return redirect(url_for('gallery'))

# THEMED GALLERIES - Geographic and Characteristic-based collections

@app.route('/gallery/thailand')
def thailand_gallery():
    """Thailand Orchids Gallery - Showcasing orchids from Thailand"""
    page = request.args.get('page', 1, type=int)
    per_page = 24
    
    try:
        # Query orchids from Thailand
        query = OrchidRecord.query.filter(
            or_(
                OrchidRecord.country.ilike('%thailand%'),
                OrchidRecord.region.ilike('%thailand%'),
                OrchidRecord.native_habitat.ilike('%thailand%'),
                OrchidRecord.ai_description.ilike('%thailand%')
            ),
            OrchidRecord.google_drive_id.isnot(None),
            OrchidRecord.validation_status != 'rejected'
        )
        
        orchids = query.paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        thailand_info = {
            'name': 'Thailand',
            'description': 'Thailand is home to over 1,000 orchid species across diverse ecosystems from lowland rainforests to mountain peaks. The country\'s tropical climate and varied elevations create perfect conditions for both epiphytic and terrestrial orchids.',
            'climate': 'Tropical monsoon climate with distinct wet and dry seasons',
            'regions': ['Northern Mountains (Doi Phukha)', 'Central Plains', 'Southern Peninsula'],
            'common_genera': ['Dendrobium', 'Bulbophyllum', 'Vanda', 'Aerides']
        }
        
        return render_template('themed_galleries/thailand_gallery.html', 
                             orchids=orchids, 
                             country_info=thailand_info,
                             title="Orchids of Thailand")
        
    except Exception as e:
        logger.error(f"Error loading Thailand gallery: {e}")
        flash('Unable to load Thailand orchid gallery', 'error')
        return redirect(url_for('gallery'))

@app.route('/gallery/madagascar')
def madagascar_gallery():
    """Madagascar Orchids Gallery - Island endemic orchids"""
    page = request.args.get('page', 1, type=int)
    per_page = 24
    
    try:
        # Query orchids from Madagascar
        query = OrchidRecord.query.filter(
            or_(
                OrchidRecord.country.ilike('%madagascar%'),
                OrchidRecord.region.ilike('%madagascar%'),
                OrchidRecord.native_habitat.ilike('%madagascar%'),
                OrchidRecord.ai_description.ilike('%madagascar%')
            ),
            OrchidRecord.google_drive_id.isnot(None),
            OrchidRecord.validation_status != 'rejected'
        )
        
        orchids = query.paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        madagascar_info = {
            'name': 'Madagascar',
            'description': 'Madagascar, the fourth largest island in the world, is home to around 1,000 orchid species with over 85% being endemic. This biological treasure trove evolved in isolation, creating unique species found nowhere else on Earth.',
            'climate': 'Diverse climates from tropical coastal to highland temperate',
            'regions': ['Eastern Rainforests', 'Central Highlands', 'Western Dry Forests'],
            'common_genera': ['Angraecum', 'Bulbophyllum', 'Cynorkis', 'Jumellea']
        }
        
        return render_template('themed_galleries/madagascar_gallery.html', 
                             orchids=orchids, 
                             country_info=madagascar_info,
                             title="Orchids of Madagascar")
        
    except Exception as e:
        logger.error(f"Error loading Madagascar gallery: {e}")
        flash('Unable to load Madagascar orchid gallery', 'error')
        return redirect(url_for('gallery'))

@app.route('/gallery/fragrant')
def fragrant_gallery():
    """Fragrant Orchids Gallery - Scented orchid species"""
    page = request.args.get('page', 1, type=int)
    per_page = 24
    
    try:
        # Query fragrant orchids
        query = OrchidRecord.query.filter(
            or_(
                OrchidRecord.ai_description.ilike('%fragrant%'),
                OrchidRecord.ai_description.ilike('%scented%'),
                OrchidRecord.ai_description.ilike('%perfume%'),
                OrchidRecord.ai_description.ilike('%aroma%')
            ),
            OrchidRecord.google_drive_id.isnot(None),
            OrchidRecord.validation_status != 'rejected'
        )
        
        orchids = query.paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        fragrance_info = {
            'name': 'Fragrant Orchids',
            'description': 'Many orchids have evolved delightful fragrances to attract specific pollinators. From the vanilla scent of some Cattleyas to the complex perfumes of Angraecums, these orchids engage multiple senses.',
            'fragrance_types': ['Citrus', 'Vanilla', 'Spicy', 'Floral', 'Musky'],
            'peak_times': ['Morning', 'Evening', 'Night'],
            'common_fragrant_genera': ['Cattleya', 'Brassavola', 'Angraecum', 'Rhynchostylis']
        }
        
        return render_template('themed_galleries/fragrant_gallery.html', 
                             orchids=orchids, 
                             fragrance_info=fragrance_info,
                             title="Fragrant Orchids")
        
    except Exception as e:
        logger.error(f"Error loading fragrant orchids gallery: {e}")
        flash('Unable to load fragrant orchids gallery', 'error')
        return redirect(url_for('gallery'))

@app.route('/gallery/night-blooming')
def night_blooming_gallery():
    """Night-Blooming Orchids Gallery - Nocturnal flowering orchids"""
    page = request.args.get('page', 1, type=int)
    per_page = 24
    
    try:
        # Query night-blooming orchids
        query = OrchidRecord.query.filter(
            or_(
                OrchidRecord.ai_description.ilike('%night%bloom%'),
                OrchidRecord.ai_description.ilike('%evening%bloom%'),
                OrchidRecord.ai_description.ilike('%nocturnal%'),
                OrchidRecord.ai_description.ilike('%evening%flower%'),
                OrchidRecord.ai_description.ilike('%night%flower%')
            ),
            OrchidRecord.google_drive_id.isnot(None),
            OrchidRecord.validation_status != 'rejected'
        )
        
        orchids = query.paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        night_info = {
            'name': 'Night-Blooming Orchids',
            'description': 'Some orchids have adapted to bloom at night, often releasing intense fragrances to attract nocturnal pollinators like moths and bats. These mysterious beauties reveal their secrets under moonlight.',
            'pollinators': ['Moths', 'Bats', 'Night-flying insects'],
            'characteristics': ['Strong fragrance', 'Light-colored flowers', 'Waxy petals'],
            'common_night_genera': ['Brassavola', 'Angraecum', 'Aerangis', 'Stanhopea']
        }
        
        return render_template('themed_galleries/night_blooming_gallery.html', 
                             orchids=orchids, 
                             night_info=night_info,
                             title="Night-Blooming Orchids")
        
    except Exception as e:
        logger.error(f"Error loading night-blooming orchids gallery: {e}")
        flash('Unable to load night-blooming orchids gallery', 'error')
        return redirect(url_for('gallery'))

@app.route('/gallery/members')
def members_gallery():
    """Members Collection Gallery - Orchids from society members"""
    page = request.args.get('page', 1, type=int)
    per_page = 24
    
    try:
        # Query member-submitted orchids (based on data source or photographer)
        query = OrchidRecord.query.filter(
            or_(
                OrchidRecord.ingestion_source.ilike('%member%'),
                OrchidRecord.ai_description.ilike('%member submission%'),
                OrchidRecord.photographer.ilike('%member%'),
                OrchidRecord.image_source.ilike('%member%')
            ),
            OrchidRecord.google_drive_id.isnot(None),
            OrchidRecord.validation_status != 'rejected'
        )
        
        orchids = query.paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        members_info = {
            'name': 'Members Collection',
            'description': 'Showcase of beautiful orchids from our society members\' personal collections. These photos represent the passion and dedication of orchid enthusiasts who share their growing successes with our community.',
            'features': ['Personal collections', 'Growing achievements', 'Cultural success stories'],
            'photo_types': ['Home greenhouse', 'Garden displays', 'Show plants', 'Rare specimens']
        }
        
        return render_template('themed_galleries/members_gallery.html', 
                             orchids=orchids, 
                             members_info=members_info,
                             title="Members Collection")
        
    except Exception as e:
        logger.error(f"Error loading members gallery: {e}")
        flash('Unable to load members gallery', 'error')
        return redirect(url_for('gallery'))

@app.route('/themed-orchids')
@app.route('/themed-orchids/<theme>')
def themed_orchids(theme=None):
    """Display orchids organized by themes like fragrant, miniature, unusual, etc."""
    try:
        logger.info(f"🎨 Loading themed orchids page, theme: {theme}")
        
        # Get all themes for navigation
        themes = ORCHID_THEMES
        
        # If no theme specified, show theme selection page
        if not theme:
            return render_template('themed_orchids_index.html', themes=themes)
            
        # Validate theme exists
        if theme not in themes:
            flash(f"Theme '{theme}' not found", "error")
            return redirect(url_for('themed_orchids'))
            
        theme_info = themes[theme]
        
        # Get orchids for this theme
        orchids = get_orchids_by_theme(theme_info['keywords'])
        
        logger.info(f"✅ Found {len(orchids)} orchids for theme '{theme}'")
        
        return render_template('themed_orchids.html', 
                             orchids=orchids, 
                             theme=theme,
                             theme_info=theme_info,
                             themes=themes,
                             orchid_count=len(orchids))
                             
    except Exception as e:
        logger.error(f"❌ Error in themed orchids: {e}")
        flash("Error loading themed orchids", "error")
        return redirect(url_for('gallery'))

@app.route('/api/themed-orchids')
@app.route('/api/themed-orchids/<theme>')
def api_themed_orchids(theme=None):
    """API endpoint for themed orchids data"""
    try:
        if not theme:
            # Return all available themes
            return jsonify({
                'themes': ORCHID_THEMES,
                'success': True
            })
            
        if theme not in ORCHID_THEMES:
            return jsonify({'error': f'Theme {theme} not found', 'success': False}), 404
            
        theme_info = ORCHID_THEMES[theme]
        orchids = get_orchids_by_theme(theme_info['keywords'])
        
        orchid_data = []
        for orchid in orchids:
            orchid_data.append({
                'id': orchid.id,
                'scientific_name': orchid.scientific_name,
                'common_names': orchid.common_names,
                'ai_description': orchid.ai_description,
                'google_drive_id': orchid.google_drive_id,
                'image_url': orchid.image_url,
                'locality': orchid.locality,
                'event_date': orchid.event_date
            })
            
        return jsonify({
            'orchids': orchid_data,
            'theme': theme,
            'theme_info': theme_info,
            'count': len(orchid_data),
            'success': True
        })
        
    except Exception as e:
        logger.error(f"❌ Error in themed orchids API: {e}")
        return jsonify({'error': 'Internal server error', 'success': False}), 500

@app.route('/gallery')
def gallery():
    """Clean working gallery showing ALL 1,607 real orchid images"""
    page = request.args.get('page', 1, type=int)
    genus = request.args.get('genus', '')
    climate = request.args.get('climate', '')
    growth_habit = request.args.get('growth_habit', '')
    search_query = request.args.get('search', '')
    per_page = 48
    
    try:
        # Direct PostgreSQL connection - bypass ALL SQLAlchemy issues
        import psycopg2
        conn = psycopg2.connect(os.environ.get("DATABASE_URL"))
        cursor = conn.cursor()
        
        # Build WHERE clause
        where_conditions = ["google_drive_id IS NOT NULL AND google_drive_id != '' AND google_drive_id != 'None'"]
        params = []
        
        if genus:
            where_conditions.append("genus ILIKE %s")
            params.append(f'%{genus}%')
        if climate:
            where_conditions.append("climate_preference = %s")
            params.append(climate)
        if growth_habit:
            where_conditions.append("growth_habit = %s")
            params.append(growth_habit)
        if search_query:
            where_conditions.append("(display_name ILIKE %s OR scientific_name ILIKE %s OR genus ILIKE %s)")
            search_param = f'%{search_query}%'
            params.extend([search_param, search_param, search_param])
        
        where_clause = " AND ".join(where_conditions)
        
        # Get total count
        count_query = f"SELECT COUNT(*) FROM orchid_record WHERE {where_clause}"
        cursor.execute(count_query, params)
        total_count = cursor.fetchone()[0]
        
        # Get the main data
        main_query = f"""
            SELECT id, display_name, scientific_name, genus, species, author, 
                   region, native_habitat, bloom_time, growth_habit, climate_preference,
                   google_drive_id, photographer, ai_description, created_at, is_featured
            FROM orchid_record 
            WHERE {where_clause}
            ORDER BY created_at DESC 
            LIMIT %s OFFSET %s
        """
        
        main_params = params + [per_page, (page - 1) * per_page]
        cursor.execute(main_query, main_params)
        rows = cursor.fetchall()
        
        # Create clean orchid objects
        from datetime import datetime
        
        class CleanOrchid:
            def __init__(self, row):
                self.id = row[0]
                self.display_name = row[1] or 'Unknown Orchid'
                self.scientific_name = row[2] or 'Unknown Species'
                self.genus = row[3] or 'Unknown'
                self.species = row[4] or ''
                self.author = row[5] or ''
                self.region = row[6] or ''
                self.native_habitat = row[7] or ''
                self.bloom_time = row[8] or ''
                self.growth_habit = row[9] or ''
                self.climate_preference = row[10] or ''
                self.google_drive_id = row[11]
                self.photographer = row[12] or 'FCOS Collection'
                self.ai_description = row[13] or f'Beautiful {self.scientific_name} specimen'
                self.created_at = row[14] or datetime.now()
                self.is_featured = row[15] or False
                self.image_url = f'/api/drive-photo/{self.google_drive_id}' if self.google_drive_id else None
                self.ai_confidence = 0.95
        
        orchid_items = [CleanOrchid(row) for row in rows]
        
        # Create clean pagination
        class CleanPagination:
            def __init__(self, items, total, page, per_page):
                self.items = items
                self.total = total
                self.page = page
                self.per_page = per_page
                self.pages = (total + per_page - 1) // per_page if total > 0 else 1
                self.has_prev = page > 1
                self.has_next = page < self.pages
                self.prev_num = page - 1 if self.has_prev else None
                self.next_num = page + 1 if self.has_next else None
            
            def iter_pages(self, left_edge=2, left_current=2, right_current=3, right_edge=2):
                last = self.pages
                for num in range(1, last + 1):
                    if num <= left_edge or \
                       (self.page - left_current - 1 < num < self.page + right_current) or \
                       num > last - right_edge:
                        yield num
        
        orchids = CleanPagination(orchid_items, total_count, page, per_page)
        
        # Get filter options
        cursor.execute("SELECT DISTINCT genus FROM orchid_record WHERE genus IS NOT NULL AND genus != '' ORDER BY genus")
        genera = [row[0] for row in cursor.fetchall()]
        
        cursor.execute("SELECT DISTINCT climate_preference FROM orchid_record WHERE climate_preference IS NOT NULL AND climate_preference != '' ORDER BY climate_preference")
        climates = [row[0] for row in cursor.fetchall()]
        
        cursor.execute("SELECT DISTINCT growth_habit FROM orchid_record WHERE growth_habit IS NOT NULL AND growth_habit != '' ORDER BY growth_habit")
        growth_habits = [row[0] for row in cursor.fetchall()]
        
        cursor.close()
        conn.close()
        
        logger.info(f"✅ CLEAN GALLERY SUCCESS: Loaded {len(orchid_items)} from {total_count} total orchids")
        
        return render_template('gallery.html', 
            orchids=orchids,
            page=page,
            pages=orchids.pages,
            total=total_count,
            per_page=per_page,
            search_query=search_query,
            genus_filter=genus,
            climate_filter=climate,
            growth_habit_filter=growth_habit,
            genera=genera,
            climates=climates,
            growth_habits=growth_habits,
            current_genus=genus,
            current_climate=climate,
            current_growth_habit=growth_habit
        )
        
    except Exception as e:
        logger.error(f"❌ Clean gallery failed: {e}")
        # Minimal fallback
        class EmptyPagination:
            def __init__(self):
                self.items = []
                self.total = 0
                self.page = 1
                self.pages = 1
                self.per_page = 48
                self.has_prev = False
                self.has_next = False
                self.prev_num = None
                self.next_num = None
        
        return render_template('gallery.html', 
            orchids=EmptyPagination(),
            page=1, pages=1, total=0, per_page=48,
            search_query='', genus_filter='', climate_filter='', growth_habit_filter='',
            genera=[], climates=[], growth_habits=[],
            current_genus='', current_climate='', current_growth_habit=''
        )

@app.route('/gallery-old')
def gallery_old():
    """Gallery with comprehensive photo failsafe protection"""
    try:
        # Primary attempt - database query
        page = request.args.get('page', 1, type=int)
        genus = request.args.get('genus', '')
        climate = request.args.get('climate', '')
        growth_habit = request.args.get('growth_habit', '')
        
        # Skip broken SQLAlchemy queries - go directly to working database access
        
        # DIRECT DATABASE ACCESS - All 1,607 Real Orchid Images
        logger.info("🎯 Loading ALL 1,607 real orchid images with direct database access")
        
        try:
            # Direct PostgreSQL connection to bypass SQLAlchemy model issues
            import psycopg2
            conn = psycopg2.connect(os.environ.get("DATABASE_URL"))
            cursor = conn.cursor()
            
            # Build WHERE clause for filters
            where_conditions = ["google_drive_id IS NOT NULL AND google_drive_id != '' AND google_drive_id != 'None'"]
            params = []
            
            if genus:
                where_conditions.append("genus ILIKE %s")
                params.append(f'%{genus}%')
                
            if climate:
                where_conditions.append("climate_preference = %s")
                params.append(climate)
                
            if growth_habit:
                where_conditions.append("growth_habit = %s")
                params.append(growth_habit)
            
            where_clause = " AND ".join(where_conditions)
            
            # Get total count
            count_query = f"SELECT COUNT(*) FROM orchid_record WHERE {where_clause}"
            cursor.execute(count_query, params)
            total_count = cursor.fetchone()[0]
            
            # Get the main data
            main_query = f"""
                SELECT id, display_name, scientific_name, genus, species, author, 
                       region, native_habitat, bloom_time, growth_habit, climate_preference,
                       google_drive_id, photographer, ai_description, created_at, is_featured
                FROM orchid_record 
                WHERE {where_clause}
                ORDER BY created_at DESC 
                LIMIT %s OFFSET %s
            """
            
            per_page = 48
            main_params = params + [per_page, (page - 1) * per_page]
            cursor.execute(main_query, main_params)
            rows = cursor.fetchall()
            
            # Create orchid objects with clean data structure
            class DirectOrchid:
                def __init__(self, row):
                    self.id = row[0]
                    self.display_name = row[1] or 'Unknown Orchid'
                    self.scientific_name = row[2] or 'Unknown Species'
                    self.genus = row[3] or 'Unknown'
                    self.species = row[4] or ''
                    self.author = row[5] or ''
                    self.region = row[6] or ''
                    self.native_habitat = row[7] or ''
                    self.bloom_time = row[8] or ''
                    self.growth_habit = row[9] or ''
                    self.climate_preference = row[10] or ''
                    self.google_drive_id = row[11]
                    self.photographer = row[12] or 'FCOS Collection'
                    self.ai_description = row[13] or f'Beautiful {self.scientific_name} specimen'
                    self.created_at = row[14] or datetime.now()
                    self.is_featured = row[15] or False
                    
                    # Computed properties
                    self.image_url = f'/api/drive-photo/{self.google_drive_id}' if self.google_drive_id else None
                    self.ai_confidence = 0.95
            
            orchid_items = [DirectOrchid(row) for row in rows]
            
            # Create pagination object
            class DirectPagination:
                def __init__(self, items, total, page, per_page):
                    self.items = items
                    self.total = total
                    self.page = page
                    self.per_page = per_page
                    self.pages = (total + per_page - 1) // per_page if total > 0 else 1
                    self.has_prev = page > 1
                    self.has_next = page < self.pages
                    self.prev_num = page - 1 if self.has_prev else None
                    self.next_num = page + 1 if self.has_next else None
            
            orchids = DirectPagination(orchid_items, total_count, page, per_page)
            
            # Close connection
            cursor.close()
            conn.close()
            
            logger.info(f"✅ DIRECT DATABASE: Loaded {len(orchid_items)} from {total_count} total orchids with Google Drive IDs")
            
        except Exception as e:
            logger.error(f"❌ Direct database access failed: {e}")
            # Emergency fallback to prevent complete failure
            class EmptyOrchid:
                def __init__(self):
                    self.id = 0
                    self.display_name = "Database Connection Error"
                    self.scientific_name = "Please contact support"
                    self.genus = "Error"
                    self.photographer = "System"
                    self.ai_description = "Database connection failed - using emergency backup"
                    self.image_url = "/static/images/orchid_placeholder.svg"
                    self.is_featured = False
                    self.ai_confidence = 0.0
                    self.created_at = datetime.now()
            
            class EmptyPagination:
                def __init__(self):
                    self.items = [EmptyOrchid()]
                    self.total = 0
                    self.page = 1
                    self.pages = 1
                    self.per_page = 48
                    self.has_prev = False
                    self.has_next = False
                    self.prev_num = None
                    self.next_num = None
            
            orchids = EmptyPagination()
            logger.warning("🆘 Using emergency backup due to database failure")
        
        # Simple filter data for the template
        genera = ['Trichocentrum', 'Cattleya', 'Brassolaeliocattleya', 'Potinara', 'Angcm']
        climates = ['cool', 'intermediate', 'warm']
        growth_habits = ['epiphyte', 'terrestrial', 'lithophyte']
        
        return render_template('gallery.html', 
            orchids=orchids,
            page=orchids.page,
            pages=orchids.pages,
            total=orchids.total,
            per_page=orchids.per_page,
            search_query='',
            genus_filter=genus,
            climate_filter=climate,
            growth_habit_filter=growth_habit,
            genera=genera,
            climates=climates,
            growth_habits=growth_habits,
            current_genus=genus,
            current_climate=climate,
            current_growth_habit=growth_habit
        )
        
        # Original database logic as fallback
        if orchids.total < 3:
            logger.warning(f"⚠️ Gallery filter returned {orchids.total} orchids for climate={climate}, genus={genus}, growth_habit={growth_habit}")
            # Fall back to all orchids without filters if specific filter returns too few results
            query = OrchidRecord.query.filter(
                OrchidRecord.google_drive_id.isnot(None),
                OrchidRecord.google_drive_id != '',
                OrchidRecord.google_drive_id != 'None'
            )
            orchids = query.order_by(OrchidRecord.created_at.desc()).paginate(
                page=1, per_page=12, error_out=False
            )
        
        # FORCE only Google Drive images - block all external URLs
        working_orchids = []
        for orchid in orchids.items:
            # Only include orchids with Google Drive IDs - these ALWAYS work
            if (hasattr(orchid, 'google_drive_id') and orchid.google_drive_id and 
                str(orchid.google_drive_id).strip() and str(orchid.google_drive_id) != 'None'):
                working_orchids.append(orchid)
        
        # ALWAYS ensure we have enough photos - NEVER show empty gallery
        if len(working_orchids) < 12:
            logging.warning(f"⚠️ Gallery protection: Only {len(working_orchids)} Google Drive images, filling remaining {12 - len(working_orchids)} slots")
            
            # Get more Google Drive orchids from database with STRICT filtering
            additional_orchids = OrchidRecord.query.filter(
                OrchidRecord.google_drive_id.isnot(None),
                OrchidRecord.google_drive_id != '',
                OrchidRecord.google_drive_id != 'None',
                ~OrchidRecord.id.in_([o.id for o in working_orchids])
            ).limit(12 - len(working_orchids)).all()
            
            working_orchids.extend(additional_orchids)
            
            # If still not enough, use failsafe system
            if len(working_orchids) < 6:
                from photo_failsafe_system import get_photos_guaranteed
                backup_photos, _ = get_photos_guaranteed(12 - len(working_orchids))
                
                # Convert backup photos to orchid-like objects for template compatibility
                class MockOrchid:
                    def __init__(self, photo_data):
                        # Ensure ID is always numeric for template compatibility
                        backup_id = photo_data.get('id', 9000 + len(working_orchids))
                        self.id = int(backup_id) if isinstance(backup_id, str) and backup_id.isdigit() else backup_id
                        self.display_name = photo_data.get('common_name', 'Beautiful Orchid')
                        self.scientific_name = photo_data.get('scientific_name', 'Orchidaceae sp.')
                        self.genus = photo_data.get('scientific_name', 'Unknown').split()[0]
                        self.google_drive_id = 'failsafe'
                        self.ai_description = photo_data.get('description', 'Stunning orchid specimen')
                        self.created_at = datetime.now()
                        self.backup_image_url = photo_data.get('image_url')
                        self.is_emergency_backup = True
                
                mock_orchids = [MockOrchid(photo) for photo in backup_photos]
                working_orchids.extend(mock_orchids)
        
        # Update the pagination object with working orchids only
        orchids.items = working_orchids[:12]  # Limit to 12 for clean gallery
        orchids.total = len(working_orchids)
        
        # Get filter options
        genera = db.session.query(OrchidRecord.genus).distinct().filter(
            OrchidRecord.genus.isnot(None)
        ).all()
        genera = [g[0] for g in genera if g[0]]
        
        climates = ['cool', 'intermediate', 'warm']
        growth_habits = ['epiphytic', 'terrestrial', 'lithophytic']
        
        return render_template('gallery.html',
                             orchids=orchids,
                             genera=genera,
                             climates=climates,
                             growth_habits=growth_habits,
                             current_genus=genus,
                             current_climate=climate,
                             current_growth_habit=growth_habit)
    
    except Exception as e:
        logging.error(f"🚨 GALLERY FAILURE: {e}")
        
        # EMERGENCY FAILSAFE - NEVER show empty gallery
        try:
            from photo_failsafe_system import get_photos_guaranteed
            
            photos, recovery_info = get_photos_guaranteed(12)
            
            # Create mock pagination object for template compatibility
            class MockPagination:
                def __init__(self, items):
                    self.items = items
                    self.total = len(items)
                    self.pages = 1
                    self.page = 1
                    self.per_page = len(items)
                    self.has_prev = False
                    self.has_next = False
                    self.prev_num = None
                    self.next_num = None
            
            mock_orchids = MockPagination(photos)
            
            return render_template('gallery.html',
                                 orchids=mock_orchids,
                                 genera=['Phalaenopsis', 'Dendrobium', 'Cattleya'],
                                 climates=['cool', 'intermediate', 'warm'],
                                 growth_habits=['epiphytic', 'terrestrial'],
                                 current_genus='',
                                 current_climate='',
                                 current_growth_habit='',
                                 emergency_mode=True,
                                 recovery_info=recovery_info)
        
        except Exception as backup_error:
            logging.critical(f"🆘 BACKUP SYSTEM FAILED: {backup_error}")
            
            # Ultimate fallback - minimal working gallery
            return render_template('error.html', 
                                 error_message="Gallery temporarily unavailable. Please try again in a few moments.")

@app.route('/search')
def search():
    """Comprehensive orchid search with advanced filtering capabilities"""
    # Get all search parameters
    query_text = request.args.get('q', '').strip()
    
    # Basic filters
    genus = request.args.get('genus', '')
    species = request.args.get('species', '')
    author = request.args.get('author', '')
    
    # Geographic filters
    country = request.args.get('country', '')
    state_province = request.args.get('state_province', '')
    region = request.args.get('region', '')
    habitat_type = request.args.get('habitat_type', '')
    
    # Parentage/breeding filters
    orchid_type = request.args.get('orchid_type', '')  # species, hybrid, both
    has_parentage = request.args.get('has_parentage', '')
    generation = request.args.get('generation', '')
    pod_parent = request.args.get('pod_parent', '')
    pollen_parent = request.args.get('pollen_parent', '')
    rhs_registered = request.args.get('rhs_registered', '')
    
    # Flowering characteristics
    flowering_status = request.args.get('flowering_status', '')
    flowering_stage = request.args.get('flowering_stage', '')
    bloom_season = request.args.get('bloom_season', '')
    flower_size = request.args.get('flower_size', '')
    
    # Growing characteristics  
    growth_habit = request.args.get('growth_habit', '')
    climate_preference = request.args.get('climate_preference', '')
    substrate_type = request.args.get('substrate_type', '')
    growing_environment = request.args.get('growing_environment', '')
    natural_vs_cultivated = request.args.get('natural_vs_cultivated', '')
    
    # Cultural requirements (using actual database fields)
    light_requirements = request.args.get('light_requirements', '')
    temperature_range = request.args.get('temperature_range', '')
    water_requirements = request.args.get('water_requirements', '')
    
    # Plant morphology
    pseudobulb_presence = request.args.get('pseudobulb_presence', '')
    leaf_form = request.args.get('leaf_form', '')
    plant_maturity = request.args.get('plant_maturity', '')
    
    # Conservation and identification
    identification_status = request.args.get('identification_status', '')
    conservation_concern = request.args.get('conservation_concern', '')
    validation_status = request.args.get('validation_status', '')
    
    # NEW: COMPREHENSIVE BIOLOGICAL FILTERS
    # AI Analysis filters
    ai_confidence_min = request.args.get('ai_confidence_min', '')
    ai_confidence_max = request.args.get('ai_confidence_max', '')
    has_ai_description = request.args.get('has_ai_description', '')
    ai_metadata_category = request.args.get('ai_metadata_category', '')
    
    # Trait Discovery filters
    has_predicted_traits = request.args.get('has_predicted_traits', '')
    has_observed_traits = request.args.get('has_observed_traits', '')
    trait_inheritance_pattern = request.args.get('trait_inheritance_pattern', '')
    breeding_value_min = request.args.get('breeding_value_min', '')
    breeding_value_max = request.args.get('breeding_value_max', '')
    
    # Pollinator Relationship filters
    pollination_method_filter = request.args.get('pollination_method_filter', '')
    pollinator_type = request.args.get('pollinator_type', '')
    pollination_strategy = request.args.get('pollination_strategy', '')
    has_pollinator_data = request.args.get('has_pollinator_data', '')
    
    # Mycorrhizal Association filters
    fungal_partner_genus = request.args.get('fungal_partner_genus', '')
    symbiotic_relationship_type = request.args.get('symbiotic_relationship_type', '')
    has_mycorrhizal_data = request.args.get('has_mycorrhizal_data', '')
    
    # Ecological Interaction filters
    has_companion_plants = request.args.get('has_companion_plants', '')
    environmental_indicator = request.args.get('environmental_indicator', '')
    habitat_complexity = request.args.get('habitat_complexity', '')
    ecosystem_role = request.args.get('ecosystem_role', '')
    
    # Morphological Trait filters
    root_type = request.args.get('root_type', '')
    maturity_stage = request.args.get('maturity_stage', '')
    flower_complexity = request.args.get('flower_complexity', '')
    morphological_adaptation = request.args.get('morphological_adaptation', '')
    
    # Behavioral Mechanism filters
    mimicry_type = request.args.get('mimicry_type', '')
    deception_strategy = request.args.get('deception_strategy', '')
    attraction_mechanism = request.args.get('attraction_mechanism', '')
    behavioral_adaptation = request.args.get('behavioral_adaptation', '')
    
    # Conservation Status filters
    conservation_priority = request.args.get('conservation_priority', '')
    rarity_indicator = request.args.get('rarity_indicator', '')
    threat_level = request.args.get('threat_level', '')
    protection_status = request.args.get('protection_status', '')
    
    # System filters
    data_source = request.args.get('data_source', '')
    ingestion_source = request.args.get('ingestion_source', '')
    has_image = request.args.get('has_image', '')
    is_featured = request.args.get('is_featured', '')
    photographer = request.args.get('photographer', '')
    
    # Sorting and display options
    sort_by = request.args.get('sort_by', 'relevance')
    results_per_page = int(request.args.get('results_per_page', '50'))
    
    # Build base query
    query = OrchidRecord.query
    
    # 1. COMPREHENSIVE TEXT SEARCH across all text fields
    if query_text:
        search_filters = [
            # Primary identification fields
            OrchidRecord.display_name.ilike(f'%{query_text}%'),
            OrchidRecord.scientific_name.ilike(f'%{query_text}%'),
            OrchidRecord.genus.ilike(f'%{query_text}%'),
            OrchidRecord.species.ilike(f'%{query_text}%'),
            OrchidRecord.common_names.ilike(f'%{query_text}%'),
            OrchidRecord.grex_name.ilike(f'%{query_text}%'),
            OrchidRecord.clone_name.ilike(f'%{query_text}%'),
            OrchidRecord.author.ilike(f'%{query_text}%'),
            
            # Geographic and habitat fields
            OrchidRecord.region.ilike(f'%{query_text}%'),
            OrchidRecord.country.ilike(f'%{query_text}%'),
            OrchidRecord.state_province.ilike(f'%{query_text}%'),
            OrchidRecord.locality.ilike(f'%{query_text}%'),
            OrchidRecord.native_habitat.ilike(f'%{query_text}%'),
            OrchidRecord.collector.ilike(f'%{query_text}%'),
            
            # Parentage fields
            OrchidRecord.pod_parent.ilike(f'%{query_text}%'),
            OrchidRecord.pollen_parent.ilike(f'%{query_text}%'),
            OrchidRecord.parentage_formula.ilike(f'%{query_text}%'),
            OrchidRecord.registrant.ilike(f'%{query_text}%'),
            
            # Descriptive fields
            OrchidRecord.cultural_notes.ilike(f'%{query_text}%'),
            OrchidRecord.ai_description.ilike(f'%{query_text}%'),
            OrchidRecord.water_requirements.ilike(f'%{query_text}%'),
            OrchidRecord.fertilizer_needs.ilike(f'%{query_text}%'),
            
            # Growing and environmental fields
            OrchidRecord.bloom_time.ilike(f'%{query_text}%'),
            OrchidRecord.growth_habit.ilike(f'%{query_text}%'),
            OrchidRecord.climate_preference.ilike(f'%{query_text}%'),
            OrchidRecord.leaf_form.ilike(f'%{query_text}%'),
            OrchidRecord.light_requirements.ilike(f'%{query_text}%'),
            OrchidRecord.temperature_range.ilike(f'%{query_text}%'),
            
            # Enhanced habitat fields
            OrchidRecord.growing_environment.ilike(f'%{query_text}%'),
            OrchidRecord.substrate_type.ilike(f'%{query_text}%'),
            OrchidRecord.mounting_evidence.ilike(f'%{query_text}%'),
            OrchidRecord.setting_type.ilike(f'%{query_text}%'),
            OrchidRecord.companion_plants.ilike(f'%{query_text}%'),
            OrchidRecord.elevation_indicators.ilike(f'%{query_text}%'),
            OrchidRecord.conservation_status_clues.ilike(f'%{query_text}%'),
            
            # Morphology fields  
            OrchidRecord.root_visibility.ilike(f'%{query_text}%'),
            OrchidRecord.flowering_stage.ilike(f'%{query_text}%'),
            OrchidRecord.bloom_season_indicator.ilike(f'%{query_text}%'),
            
            # Source and photographer fields
            OrchidRecord.photographer.ilike(f'%{query_text}%'),
            OrchidRecord.image_source.ilike(f'%{query_text}%'),
            OrchidRecord.data_source.ilike(f'%{query_text}%'),
            
            # OCR and AI fields
            OrchidRecord.ocr_text.ilike(f'%{query_text}%'),
            OrchidRecord.ai_extracted_metadata.ilike(f'%{query_text}%')
        ]
        query = query.filter(or_(*search_filters))
    
    # 2. TAXONOMIC FILTERS
    if genus:
        query = query.filter(OrchidRecord.genus.ilike(f'%{genus}%'))
    if species:
        query = query.filter(OrchidRecord.species.ilike(f'%{species}%'))
    if author:
        query = query.filter(OrchidRecord.author.ilike(f'%{author}%'))
    
    # 3. GEOGRAPHIC FILTERS
    if country:
        query = query.filter(OrchidRecord.country.ilike(f'%{country}%'))
    if state_province:
        query = query.filter(OrchidRecord.state_province.ilike(f'%{state_province}%'))
    if region:
        query = query.filter(OrchidRecord.region.ilike(f'%{region}%'))
    if habitat_type:
        query = query.filter(OrchidRecord.native_habitat.ilike(f'%{habitat_type}%'))
    
    # 4. PARENTAGE/BREEDING FILTERS
    if orchid_type == 'species':
        query = query.filter(OrchidRecord.is_species == True)
    elif orchid_type == 'hybrid':
        query = query.filter(OrchidRecord.is_hybrid == True)
        
    if has_parentage == 'yes':
        query = query.filter(or_(
            OrchidRecord.pod_parent.isnot(None),
            OrchidRecord.pollen_parent.isnot(None),
            OrchidRecord.parentage_formula.isnot(None)
        ))
    elif has_parentage == 'no':
        query = query.filter(and_(
            OrchidRecord.pod_parent.is_(None),
            OrchidRecord.pollen_parent.is_(None),
            OrchidRecord.parentage_formula.is_(None)
        ))
    
    if generation:
        try:
            gen_num = int(generation)
            query = query.filter(OrchidRecord.generation == gen_num)
        except ValueError:
            pass
            
    if pod_parent:
        query = query.filter(OrchidRecord.pod_parent.ilike(f'%{pod_parent}%'))
    if pollen_parent:
        query = query.filter(OrchidRecord.pollen_parent.ilike(f'%{pollen_parent}%'))
    
    if rhs_registered == 'yes':
        query = query.filter(OrchidRecord.rhs_registration_id.isnot(None))
    elif rhs_registered == 'no':
        query = query.filter(OrchidRecord.rhs_registration_id.is_(None))
    
    # 5. FLOWERING CHARACTERISTICS
    if flowering_status == 'flowering':
        query = query.filter(OrchidRecord.is_flowering == True)
    elif flowering_status == 'not_flowering':
        query = query.filter(OrchidRecord.is_flowering == False)
        
    if flowering_stage:
        query = query.filter(OrchidRecord.flowering_stage == flowering_stage)
        
    if bloom_season:
        query = query.filter(OrchidRecord.bloom_season_indicator.ilike(f'%{bloom_season}%'))
    
    if flower_size:
        if flower_size == 'small':
            query = query.filter(OrchidRecord.flower_size_mm < 50)
        elif flower_size == 'medium':
            query = query.filter(and_(OrchidRecord.flower_size_mm >= 50, OrchidRecord.flower_size_mm < 100))
        elif flower_size == 'large':
            query = query.filter(OrchidRecord.flower_size_mm >= 100)
    
    # 6. GROWING CHARACTERISTICS
    if growth_habit:
        query = query.filter(OrchidRecord.growth_habit == growth_habit)
    if climate_preference:
        query = query.filter(OrchidRecord.climate_preference == climate_preference)
    if substrate_type:
        query = query.filter(OrchidRecord.substrate_type == substrate_type)
    if growing_environment:
        query = query.filter(OrchidRecord.growing_environment == growing_environment)
    if natural_vs_cultivated:
        query = query.filter(OrchidRecord.natural_vs_cultivated == natural_vs_cultivated)
    
    # 7. CULTURAL REQUIREMENTS (using actual database fields)
    if light_requirements:
        query = query.filter(OrchidRecord.light_requirements.ilike(f'%{light_requirements}%'))
    if temperature_range:
        query = query.filter(OrchidRecord.temperature_range.ilike(f'%{temperature_range}%'))
    if water_requirements:
        query = query.filter(OrchidRecord.water_requirements.ilike(f'%{water_requirements}%'))
    
    # 8. PLANT MORPHOLOGY
    if pseudobulb_presence == 'yes':
        query = query.filter(OrchidRecord.pseudobulb_presence == True)
    elif pseudobulb_presence == 'no':
        query = query.filter(OrchidRecord.pseudobulb_presence == False)
        
    if leaf_form:
        query = query.filter(OrchidRecord.leaf_form.ilike(f'%{leaf_form}%'))
    if plant_maturity:
        query = query.filter(OrchidRecord.plant_maturity == plant_maturity)
    
    # 9. CONSERVATION AND IDENTIFICATION
    if identification_status:
        query = query.filter(OrchidRecord.identification_status == identification_status)
    if conservation_concern == 'yes':
        query = query.filter(OrchidRecord.conservation_status_clues.isnot(None))
    elif conservation_concern == 'no':
        query = query.filter(OrchidRecord.conservation_status_clues.is_(None))
    if validation_status:
        query = query.filter(OrchidRecord.validation_status == validation_status)
    
    # 10. NEW: COMPREHENSIVE BIOLOGICAL FILTERS
    
    # 10.1 AI ANALYSIS FILTERS
    if ai_confidence_min:
        try:
            min_confidence = float(ai_confidence_min)
            query = query.filter(OrchidRecord.ai_confidence >= min_confidence)
        except ValueError:
            pass
    
    if ai_confidence_max:
        try:
            max_confidence = float(ai_confidence_max)
            query = query.filter(OrchidRecord.ai_confidence <= max_confidence)
        except ValueError:
            pass
    
    if has_ai_description == 'yes':
        query = query.filter(OrchidRecord.ai_description.isnot(None))
    elif has_ai_description == 'no':
        query = query.filter(OrchidRecord.ai_description.is_(None))
    
    if ai_metadata_category:
        query = query.filter(OrchidRecord.ai_extracted_metadata.ilike(f'%{ai_metadata_category}%'))
    
    # 10.2 TRAIT DISCOVERY FILTERS
    if has_predicted_traits == 'yes':
        # Search in AI descriptions and cultural notes for trait analysis mentions
        query = query.filter(or_(
            OrchidRecord.ai_description.ilike('%trait%'),
            OrchidRecord.ai_description.ilike('%inheritance%'),
            OrchidRecord.cultural_notes.ilike('%trait%')
        ))
    elif has_predicted_traits == 'no':
        query = query.filter(and_(
            or_(
                OrchidRecord.ai_description.is_(None),
                and_(
                    OrchidRecord.ai_description.notlike('%trait%'),
                    OrchidRecord.ai_description.notlike('%inheritance%')
                )
            ),
            or_(
                OrchidRecord.cultural_notes.is_(None),
                OrchidRecord.cultural_notes.notlike('%trait%')
            )
        ))
    
    if has_observed_traits == 'yes':
        # Search for observed characteristics and traits in descriptions
        query = query.filter(or_(
            OrchidRecord.ai_description.ilike('%observed%'),
            OrchidRecord.ai_description.ilike('%characteristic%'),
            OrchidRecord.cultural_notes.ilike('%observed%'),
            OrchidRecord.cultural_notes.ilike('%feature%')
        ))
    elif has_observed_traits == 'no':
        query = query.filter(and_(
            or_(
                OrchidRecord.ai_description.is_(None),
                and_(
                    OrchidRecord.ai_description.notlike('%observed%'),
                    OrchidRecord.ai_description.notlike('%characteristic%')
                )
            ),
            or_(
                OrchidRecord.cultural_notes.is_(None),
                and_(
                    OrchidRecord.cultural_notes.notlike('%observed%'),
                    OrchidRecord.cultural_notes.notlike('%feature%')
                )
            )
        ))
    
    if trait_inheritance_pattern:
        query = query.filter(or_(
            OrchidRecord.ai_description.ilike(f'%{trait_inheritance_pattern}%'),
            OrchidRecord.cultural_notes.ilike(f'%{trait_inheritance_pattern}%'),
            OrchidRecord.ai_extracted_metadata.ilike(f'%{trait_inheritance_pattern}%')
        ))
    
    # Breeding value filters using flower quality scores and AI analysis
    if breeding_value_min or breeding_value_max:
        # Search for breeding value indicators in AI descriptions
        query = query.filter(or_(
            OrchidRecord.ai_description.ilike('%breeding%'),
            OrchidRecord.ai_description.ilike('%quality%'),
            OrchidRecord.cultural_notes.ilike('%breeding%'),
            OrchidRecord.cultural_notes.ilike('%award%')
        ))
    
    # 10.3 POLLINATOR RELATIONSHIP FILTERS
    if pollination_method_filter:
        query = query.filter(OrchidRecord.pollination_method.ilike(f'%{pollination_method_filter}%'))
    
    if pollinator_type:
        # Search in AI descriptions and cultural notes for pollinator mentions
        query = query.filter(or_(
            OrchidRecord.ai_description.ilike(f'%{pollinator_type}%'),
            OrchidRecord.cultural_notes.ilike(f'%{pollinator_type}%'),
            OrchidRecord.ai_extracted_metadata.ilike(f'%{pollinator_type}%')
        ))
    
    if pollination_strategy:
        # Search for specific strategies in descriptions
        query = query.filter(or_(
            OrchidRecord.ai_description.ilike(f'%{pollination_strategy}%'),
            OrchidRecord.cultural_notes.ilike(f'%{pollination_strategy}%'),
            OrchidRecord.ai_extracted_metadata.ilike(f'%{pollination_strategy}%')
        ))
    
    if has_pollinator_data == 'yes':
        query = query.filter(or_(
            OrchidRecord.pollination_method.isnot(None),
            OrchidRecord.ai_description.ilike('%pollinator%'),
            OrchidRecord.ai_description.ilike('%pollination%')
        ))
    elif has_pollinator_data == 'no':
        query = query.filter(and_(
            OrchidRecord.pollination_method.is_(None),
            or_(
                OrchidRecord.ai_description.is_(None),
                and_(
                    OrchidRecord.ai_description.notlike('%pollinator%'),
                    OrchidRecord.ai_description.notlike('%pollination%')
                )
            )
        ))
    
    # 10.4 MYCORRHIZAL ASSOCIATION FILTERS
    if fungal_partner_genus:
        # Search for fungal genus names in descriptions and metadata
        query = query.filter(or_(
            OrchidRecord.ai_description.ilike(f'%{fungal_partner_genus}%'),
            OrchidRecord.cultural_notes.ilike(f'%{fungal_partner_genus}%'),
            OrchidRecord.ai_extracted_metadata.ilike(f'%{fungal_partner_genus}%'),
            OrchidRecord.companion_plants.ilike(f'%{fungal_partner_genus}%')
        ))
    
    if symbiotic_relationship_type:
        query = query.filter(or_(
            OrchidRecord.ai_description.ilike(f'%{symbiotic_relationship_type}%'),
            OrchidRecord.cultural_notes.ilike(f'%{symbiotic_relationship_type}%'),
            OrchidRecord.ai_extracted_metadata.ilike(f'%{symbiotic_relationship_type}%')
        ))
    
    if has_mycorrhizal_data == 'yes':
        query = query.filter(or_(
            OrchidRecord.ai_description.ilike('%mycorrhiza%'),
            OrchidRecord.ai_description.ilike('%fungal%'),
            OrchidRecord.ai_description.ilike('%symbiotic%'),
            OrchidRecord.cultural_notes.ilike('%mycorrhiza%'),
            OrchidRecord.cultural_notes.ilike('%fungal%')
        ))
    elif has_mycorrhizal_data == 'no':
        query = query.filter(and_(
            or_(
                OrchidRecord.ai_description.is_(None),
                and_(
                    OrchidRecord.ai_description.notlike('%mycorrhiza%'),
                    OrchidRecord.ai_description.notlike('%fungal%'),
                    OrchidRecord.ai_description.notlike('%symbiotic%')
                )
            ),
            or_(
                OrchidRecord.cultural_notes.is_(None),
                and_(
                    OrchidRecord.cultural_notes.notlike('%mycorrhiza%'),
                    OrchidRecord.cultural_notes.notlike('%fungal%')
                )
            )
        ))
    
    # 10.5 ECOLOGICAL INTERACTION FILTERS
    if has_companion_plants == 'yes':
        query = query.filter(OrchidRecord.companion_plants.isnot(None))
    elif has_companion_plants == 'no':
        query = query.filter(OrchidRecord.companion_plants.is_(None))
    
    if environmental_indicator:
        query = query.filter(or_(
            OrchidRecord.humidity_indicators.ilike(f'%{environmental_indicator}%'),
            OrchidRecord.temperature_indicators.ilike(f'%{environmental_indicator}%'),
            OrchidRecord.light_conditions.ilike(f'%{environmental_indicator}%'),
            OrchidRecord.mounting_evidence.ilike(f'%{environmental_indicator}%')
        ))
    
    if habitat_complexity:
        query = query.filter(or_(
            OrchidRecord.setting_type.ilike(f'%{habitat_complexity}%'),
            OrchidRecord.growing_environment.ilike(f'%{habitat_complexity}%'),
            OrchidRecord.native_habitat.ilike(f'%{habitat_complexity}%')
        ))
    
    if ecosystem_role:
        query = query.filter(or_(
            OrchidRecord.ai_description.ilike(f'%{ecosystem_role}%'),
            OrchidRecord.cultural_notes.ilike(f'%{ecosystem_role}%'),
            OrchidRecord.companion_plants.ilike(f'%{ecosystem_role}%')
        ))
    
    # 10.6 MORPHOLOGICAL TRAIT FILTERS
    if root_type:
        query = query.filter(OrchidRecord.root_visibility.ilike(f'%{root_type}%'))
    
    if maturity_stage:
        query = query.filter(OrchidRecord.plant_maturity.ilike(f'%{maturity_stage}%'))
    
    if flower_complexity:
        # Search in AI descriptions for flower complexity indicators
        query = query.filter(or_(
            OrchidRecord.ai_description.ilike(f'%{flower_complexity}%'),
            OrchidRecord.cultural_notes.ilike(f'%{flower_complexity}%'),
            cast(OrchidRecord.flower_measurements, String).ilike(f'%{flower_complexity}%')
        ))
    
    if morphological_adaptation:
        query = query.filter(or_(
            OrchidRecord.ai_description.ilike(f'%{morphological_adaptation}%'),
            OrchidRecord.root_visibility.ilike(f'%{morphological_adaptation}%'),
            OrchidRecord.mounting_evidence.ilike(f'%{morphological_adaptation}%'),
            OrchidRecord.leaf_form.ilike(f'%{morphological_adaptation}%')
        ))
    
    # 10.7 BEHAVIORAL MECHANISM FILTERS
    if mimicry_type:
        query = query.filter(or_(
            OrchidRecord.ai_description.ilike(f'%{mimicry_type}%'),
            OrchidRecord.cultural_notes.ilike(f'%{mimicry_type}%'),
            OrchidRecord.ai_extracted_metadata.ilike(f'%{mimicry_type}%')
        ))
    
    if deception_strategy:
        query = query.filter(or_(
            OrchidRecord.ai_description.ilike(f'%{deception_strategy}%'),
            OrchidRecord.cultural_notes.ilike(f'%{deception_strategy}%'),
            OrchidRecord.ai_extracted_metadata.ilike(f'%{deception_strategy}%')
        ))
    
    if attraction_mechanism:
        query = query.filter(or_(
            OrchidRecord.ai_description.ilike(f'%{attraction_mechanism}%'),
            OrchidRecord.cultural_notes.ilike(f'%{attraction_mechanism}%'),
            OrchidRecord.ai_extracted_metadata.ilike(f'%{attraction_mechanism}%')
        ))
    
    if behavioral_adaptation:
        query = query.filter(or_(
            OrchidRecord.ai_description.ilike(f'%{behavioral_adaptation}%'),
            OrchidRecord.cultural_notes.ilike(f'%{behavioral_adaptation}%'),
            OrchidRecord.ai_extracted_metadata.ilike(f'%{behavioral_adaptation}%')
        ))
    
    # 10.8 CONSERVATION STATUS FILTERS
    if conservation_priority:
        query = query.filter(OrchidRecord.conservation_status_clues.ilike(f'%{conservation_priority}%'))
    
    if rarity_indicator:
        query = query.filter(or_(
            OrchidRecord.conservation_status_clues.ilike(f'%{rarity_indicator}%'),
            OrchidRecord.ai_description.ilike(f'%{rarity_indicator}%'),
            OrchidRecord.cultural_notes.ilike(f'%{rarity_indicator}%')
        ))
    
    if threat_level:
        query = query.filter(or_(
            OrchidRecord.conservation_status_clues.ilike(f'%{threat_level}%'),
            OrchidRecord.ai_description.ilike(f'%{threat_level}%')
        ))
    
    if protection_status:
        query = query.filter(or_(
            OrchidRecord.conservation_status_clues.ilike(f'%{protection_status}%'),
            OrchidRecord.ai_description.ilike(f'%{protection_status}%')
        ))
    
    # 11. SYSTEM FILTERS
    if data_source:
        query = query.filter(OrchidRecord.data_source == data_source)
    if ingestion_source:
        query = query.filter(OrchidRecord.ingestion_source == ingestion_source)
    if has_image == 'yes':
        query = query.filter(or_(
            OrchidRecord.google_drive_id.isnot(None),
            OrchidRecord.image_url.isnot(None),
            OrchidRecord.image_filename.isnot(None)
        ))
    elif has_image == 'no':
        query = query.filter(and_(
            OrchidRecord.google_drive_id.is_(None),
            OrchidRecord.image_url.is_(None),
            OrchidRecord.image_filename.is_(None)
        ))
    if is_featured == 'yes':
        query = query.filter(OrchidRecord.is_featured == True)
    elif is_featured == 'no':
        query = query.filter(OrchidRecord.is_featured == False)
    if photographer:
        query = query.filter(OrchidRecord.photographer.ilike(f'%{photographer}%'))
    
    # SORTING OPTIONS
    if sort_by == 'alphabetical':
        query = query.order_by(OrchidRecord.scientific_name.asc())
    elif sort_by == 'genus_species':
        query = query.order_by(OrchidRecord.genus.asc(), OrchidRecord.species.asc())
    elif sort_by == 'popularity':
        query = query.order_by(OrchidRecord.view_count.desc())
    elif sort_by == 'newest':
        query = query.order_by(OrchidRecord.created_at.desc())
    elif sort_by == 'featured':
        query = query.order_by(OrchidRecord.is_featured.desc(), OrchidRecord.view_count.desc())
    elif sort_by == 'flowering':
        query = query.order_by(OrchidRecord.is_flowering.desc(), OrchidRecord.flower_count.desc())
    else:  # relevance (default)
        query = query.order_by(OrchidRecord.view_count.desc())
    
    # Get results with limit
    has_filters = any([query_text, genus, species, country, region, orchid_type, flowering_status, 
                      growth_habit, climate_preference, light_requirements, has_image, is_featured,
                      # NEW: Include biological filters in has_filters check
                      ai_confidence_min, ai_confidence_max, has_ai_description, ai_metadata_category,
                      has_predicted_traits, has_observed_traits, trait_inheritance_pattern, 
                      breeding_value_min, breeding_value_max, pollination_method_filter, 
                      pollinator_type, pollination_strategy, has_pollinator_data,
                      fungal_partner_genus, symbiotic_relationship_type, has_mycorrhizal_data,
                      has_companion_plants, environmental_indicator, habitat_complexity, ecosystem_role,
                      root_type, maturity_stage, flower_complexity, morphological_adaptation,
                      mimicry_type, deception_strategy, attraction_mechanism, behavioral_adaptation,
                      conservation_priority, rarity_indicator, threat_level, protection_status])
    
    orchids = query.limit(results_per_page).all() if has_filters else []
    
    # Get search statistics
    search_stats = {
        'total_found': len(orchids),
        'has_images': sum(1 for o in orchids if o.google_drive_id or o.image_url or o.image_filename),
        'species_count': sum(1 for o in orchids if o.is_species),
        'hybrid_count': sum(1 for o in orchids if o.is_hybrid),
        'flowering_count': sum(1 for o in orchids if o.is_flowering),
        'featured_count': sum(1 for o in orchids if o.is_featured)
    }
    
    # Get filter options for dropdowns (get distinct values from database)
    try:
        filter_options = {
            'genera': [g[0] for g in db.session.query(OrchidRecord.genus).distinct().filter(OrchidRecord.genus.isnot(None)).order_by(OrchidRecord.genus).limit(100).all() if g[0]],
            'countries': [c[0] for c in db.session.query(OrchidRecord.country).distinct().filter(OrchidRecord.country.isnot(None)).order_by(OrchidRecord.country).limit(50).all() if c[0]],
            'growth_habits': [g[0] for g in db.session.query(OrchidRecord.growth_habit).distinct().filter(OrchidRecord.growth_habit.isnot(None)).order_by(OrchidRecord.growth_habit).all() if g[0]],
            'climate_preferences': [c[0] for c in db.session.query(OrchidRecord.climate_preference).distinct().filter(OrchidRecord.climate_preference.isnot(None)).order_by(OrchidRecord.climate_preference).all() if c[0]],
            'data_sources': [d[0] for d in db.session.query(OrchidRecord.data_source).distinct().filter(OrchidRecord.data_source.isnot(None)).order_by(OrchidRecord.data_source).all() if d[0]],
            'ingestion_sources': [i[0] for i in db.session.query(OrchidRecord.ingestion_source).distinct().filter(OrchidRecord.ingestion_source.isnot(None)).order_by(OrchidRecord.ingestion_source).all() if i[0]],
            'photographers': [p[0] for p in db.session.query(OrchidRecord.photographer).distinct().filter(OrchidRecord.photographer.isnot(None)).order_by(OrchidRecord.photographer).limit(50).all() if p[0]]
        }
    except Exception as e:
        logger.error(f"Error getting filter options: {e}")
        filter_options = {'genera': [], 'countries': [], 'growth_habits': [], 'climate_preferences': [], 'data_sources': [], 'ingestion_sources': [], 'photographers': []}
    
    # EXPORT FUNCTIONALITY
    export_format = request.args.get('export', '')
    if export_format == 'csv' and orchids:
        return export_search_results_csv(orchids, search_params={
            'query': query_text, 'genus': genus, 'species': species, 'country': country,
            'orchid_type': orchid_type, 'flowering_status': flowering_status
        })
    
    # ENHANCED MAPPING INTEGRATION
    # Get coordinate data for search results to enable map displays
    orchid_coordinates = []
    regional_distribution = {}
    
    try:
        from simplified_orchid_mapping import simplified_mapper
        
        # Get coordinates for orchids with precise location data
        for orchid in orchids:
            if orchid.decimal_latitude and orchid.decimal_longitude:
                orchid_coordinates.append({
                    'id': orchid.id,
                    'lat': float(orchid.decimal_latitude),
                    'lng': float(orchid.decimal_longitude),
                    'name': orchid.display_name or orchid.scientific_name or f"Orchid {orchid.id}",
                    'genus': orchid.genus or 'Unknown',
                    'species': orchid.species or 'Unknown',
                    'location': orchid.locality or orchid.country or 'Unknown location',
                    'image': orchid.google_drive_id if orchid.google_drive_id else None,
                    'region': orchid.region or 'Unknown region'
                })
        
        # Get regional distribution data for results
        for orchid in orchids:
            region = orchid.region or 'Unknown'
            if region not in regional_distribution:
                regional_distribution[region] = {
                    'count': 0,
                    'genera': set(),
                    'species': set(),
                    'flowering': 0
                }
            regional_distribution[region]['count'] += 1
            if orchid.genus:
                regional_distribution[region]['genera'].add(orchid.genus)
            if orchid.scientific_name:
                regional_distribution[region]['species'].add(orchid.scientific_name)
            if orchid.is_flowering:
                regional_distribution[region]['flowering'] += 1
        
        # Convert sets to counts for JSON serialization
        for region_data in regional_distribution.values():
            region_data['genera'] = len(region_data['genera'])
            region_data['species'] = len(region_data['species'])
        
        # Get mapping statistics
        mapping_stats = {
            'total_with_coordinates': len(orchid_coordinates),
            'total_regions': len(regional_distribution),
            'coordinate_coverage': (len(orchid_coordinates) / len(orchids) * 100) if orchids else 0
        }
        
        logger.info(f"🗺️ Search mapping integration: {len(orchid_coordinates)} coordinates, {len(regional_distribution)} regions")
        
    except Exception as e:
        logger.error(f"❌ Error generating mapping data for search: {e}")
        orchid_coordinates = []
        regional_distribution = {}
        mapping_stats = {'total_with_coordinates': 0, 'total_regions': 0, 'coordinate_coverage': 0}

    return render_template('search.html', 
                         orchids=orchids,
                         search_stats=search_stats,
                         filter_options=filter_options,
                         # ENHANCED MAPPING DATA
                         orchid_coordinates=orchid_coordinates,
                         regional_distribution=regional_distribution,
                         mapping_stats=mapping_stats,
                         # Pass all search parameters back to template
                         query=query_text,
                         genus=genus,
                         species=species,
                         author=author,
                         country=country,
                         state_province=state_province,
                         region=region,
                         habitat_type=habitat_type,
                         orchid_type=orchid_type,
                         has_parentage=has_parentage,
                         generation=generation,
                         pod_parent=pod_parent,
                         pollen_parent=pollen_parent,
                         rhs_registered=rhs_registered,
                         flowering_status=flowering_status,
                         flowering_stage=flowering_stage,
                         bloom_season=bloom_season,
                         flower_size=flower_size,
                         growth_habit=growth_habit,
                         climate_preference=climate_preference,
                         substrate_type=substrate_type,
                         growing_environment=growing_environment,
                         natural_vs_cultivated=natural_vs_cultivated,
                         light_requirements=light_requirements,
                         temperature_range=temperature_range,
                         water_requirements=water_requirements,
                         pseudobulb_presence=pseudobulb_presence,
                         leaf_form=leaf_form,
                         plant_maturity=plant_maturity,
                         identification_status=identification_status,
                         conservation_concern=conservation_concern,
                         validation_status=validation_status,
                         data_source=data_source,
                         ingestion_source=ingestion_source,
                         has_image=has_image,
                         is_featured=is_featured,
                         photographer=photographer,
                         sort_by=sort_by,
                         results_per_page=results_per_page)

def export_search_results_csv(orchids, search_params=None):
    """Export search results to CSV format"""
    import csv
    from io import StringIO
    from datetime import datetime
    
    output = StringIO()
    writer = csv.writer(output)
    
    # CSV Headers
    headers = [
        'ID', 'Display_Name', 'Scientific_Name', 'Genus', 'Species', 'Author',
        'Country', 'Region', 'Native_Habitat', 'Climate_Preference', 'Growth_Habit',
        'Is_Species', 'Is_Hybrid', 'Pod_Parent', 'Pollen_Parent', 'Parentage_Formula',
        'RHS_Registration_ID', 'Is_Flowering', 'Flowering_Stage', 'Bloom_Season',
        'Light_Requirements', 'Temperature_Range', 'Water_Requirements',
        'Conservation_Status_Clues', 'Identification_Status', 'Validation_Status',
        'Data_Source', 'Photographer', 'View_Count', 'Created_At'
    ]
    writer.writerow(headers)
    
    # Data rows
    for orchid in orchids:
        row = [
            orchid.id,
            orchid.display_name or '',
            orchid.scientific_name or '',
            orchid.genus or '',
            orchid.species or '',
            orchid.author or '',
            orchid.country or '',
            orchid.region or '',
            orchid.native_habitat or '',
            orchid.climate_preference or '',
            orchid.growth_habit or '',
            'Yes' if orchid.is_species else 'No',
            'Yes' if orchid.is_hybrid else 'No',
            orchid.pod_parent or '',
            orchid.pollen_parent or '',
            orchid.parentage_formula or '',
            orchid.rhs_registration_id or '',
            'Yes' if orchid.is_flowering else 'No',
            orchid.flowering_stage or '',
            orchid.bloom_season_indicator or '',
            orchid.light_requirements or '',
            orchid.temperature_range or '',
            orchid.water_requirements or '',
            orchid.conservation_status_clues or '',
            orchid.identification_status or '',
            orchid.validation_status or '',
            orchid.data_source or '',
            orchid.photographer or '',
            orchid.view_count or 0,
            orchid.created_at.strftime('%Y-%m-%d') if orchid.created_at else ''
        ]
        writer.writerow(row)
    
    output.seek(0)
    
    # Generate filename with search parameters
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    search_desc = []
    if search_params:
        if search_params.get('query'):
            search_desc.append(f"query_{search_params['query'][:10]}")
        if search_params.get('genus'):
            search_desc.append(f"genus_{search_params['genus']}")
        if search_params.get('country'):
            search_desc.append(f"country_{search_params['country']}")
    
    filename_parts = ['orchid_search_results']
    if search_desc:
        filename_parts.extend(search_desc[:3])  # Limit to 3 parameters
    filename_parts.append(timestamp)
    filename = '_'.join(filename_parts) + '.csv'
    
    response = make_response(output.getvalue())
    response.headers['Content-Type'] = 'text/csv'
    response.headers['Content-Disposition'] = f'attachment; filename="{filename}"'
    
    return response

@app.route('/search/mapping-integration')
def search_mapping_integration():
    """Integrate search results with mapping system"""
    # Get search parameters from query string
    search_params = dict(request.args)
    
    # Perform search with same logic as main search route
    # (This would reuse the search logic above)
    query = OrchidRecord.query
    
    # Apply same filters as main search...
    # (Code abbreviated for brevity - would use same filter logic)
    
    orchids = query.filter(OrchidRecord.decimal_latitude.isnot(None)).filter(
        OrchidRecord.decimal_longitude.isnot(None)).all()
    
    # Prepare geographic data for mapping
    map_data = []
    for orchid in orchids:
        if orchid.decimal_latitude and orchid.decimal_longitude:
            map_data.append({
                'id': orchid.id,
                'name': orchid.display_name,
                'scientific_name': orchid.scientific_name,
                'genus': orchid.genus,
                'lat': float(orchid.decimal_latitude),
                'lng': float(orchid.decimal_longitude),
                'country': orchid.country,
                'region': orchid.region,
                'is_flowering': orchid.is_flowering,
                'climate_preference': orchid.climate_preference,
                'conservation_concern': bool(orchid.conservation_status_clues)
            })
    
    return render_template('search_map_integration.html', 
                         map_data=map_data, 
                         search_params=search_params,
                         orchids_count=len(orchids))

@app.route('/api/search-analytics')
def api_search_analytics():
    """API endpoint for search analytics and statistics"""
    try:
        # Basic database statistics
        total_orchids = OrchidRecord.query.count()
        
        # Search-relevant statistics
        analytics = {
            'total_records': total_orchids,
            'with_images': OrchidRecord.query.filter(
                or_(OrchidRecord.google_drive_id.isnot(None),
                    OrchidRecord.image_url.isnot(None))
            ).count(),
            'species_count': OrchidRecord.query.filter(OrchidRecord.is_species == True).count(),
            'hybrid_count': OrchidRecord.query.filter(OrchidRecord.is_hybrid == True).count(),
            'currently_flowering': OrchidRecord.query.filter(OrchidRecord.is_flowering == True).count(),
            'with_geographic_data': OrchidRecord.query.filter(
                and_(OrchidRecord.decimal_latitude.isnot(None),
                     OrchidRecord.decimal_longitude.isnot(None))
            ).count(),
            'with_parentage': OrchidRecord.query.filter(
                or_(OrchidRecord.pod_parent.isnot(None),
                    OrchidRecord.pollen_parent.isnot(None))
            ).count(),
            'rhs_registered': OrchidRecord.query.filter(OrchidRecord.rhs_registration_id.isnot(None)).count(),
            'featured_orchids': OrchidRecord.query.filter(OrchidRecord.is_featured == True).count(),
            
            # Top genera (convert to dict)
            'top_genera': [{'genus': row[0], 'count': row[1]} for row in 
                          db.session.query(OrchidRecord.genus, func.count(OrchidRecord.id))
                          .filter(OrchidRecord.genus.isnot(None))
                          .group_by(OrchidRecord.genus)
                          .order_by(func.count(OrchidRecord.id).desc())
                          .limit(10).all()],
            
            # Top countries (convert to dict)
            'top_countries': [{'country': row[0], 'count': row[1]} for row in
                             db.session.query(OrchidRecord.country, func.count(OrchidRecord.id))
                             .filter(OrchidRecord.country.isnot(None))
                             .group_by(OrchidRecord.country)
                             .order_by(func.count(OrchidRecord.id).desc())
                             .limit(10).all()],
            
            # Climate distribution (convert to dict)
            'climate_distribution': [{'climate': row[0], 'count': row[1]} for row in
                                   db.session.query(OrchidRecord.climate_preference, func.count(OrchidRecord.id))
                                   .filter(OrchidRecord.climate_preference.isnot(None))
                                   .group_by(OrchidRecord.climate_preference)
                                   .all()]
        }
        
        return jsonify({
            'success': True,
            'analytics': analytics
        })
        
    except Exception as e:
        logger.error(f"Search analytics error: {e}")
        return jsonify({
            'success': False,
            'error': str(e)
        })

@app.route('/breeding-search-integration')
def breeding_search_integration():
    """Integration page for breeding and parentage search"""
    # Get breeding-specific parameters
    breeding_type = request.args.get('breeding_type', 'all')  # species, hybrid, intergeneric
    generation = request.args.get('generation', '')
    parent_genus = request.args.get('parent_genus', '')
    
    query = OrchidRecord.query
    
    # Apply breeding-specific filters
    if breeding_type == 'species':
        query = query.filter(OrchidRecord.is_species == True)
    elif breeding_type == 'hybrid':
        query = query.filter(OrchidRecord.is_hybrid == True)
    elif breeding_type == 'intergeneric':
        # Look for different genera in parentage
        query = query.filter(
            and_(OrchidRecord.pod_parent.isnot(None),
                 OrchidRecord.pollen_parent.isnot(None))
        )
    
    if generation:
        try:
            gen_num = int(generation)
            query = query.filter(OrchidRecord.generation == gen_num)
        except ValueError:
            pass
    
    if parent_genus:
        query = query.filter(
            or_(OrchidRecord.pod_parent.ilike(f'{parent_genus}%'),
                OrchidRecord.pollen_parent.ilike(f'{parent_genus}%'))
        )
    
    # Get results
    orchids = query.order_by(OrchidRecord.rhs_registration_id.desc()).limit(100).all()
    
    # Prepare breeding analysis data
    breeding_stats = {
        'total_found': len(orchids),
        'rhs_registered': sum(1 for o in orchids if o.rhs_registration_id),
        'with_full_parentage': sum(1 for o in orchids if o.pod_parent and o.pollen_parent),
        'generation_distribution': {}
    }
    
    # Analyze generation distribution
    for orchid in orchids:
        if orchid.generation:
            gen = f"F{orchid.generation}"
            breeding_stats['generation_distribution'][gen] = breeding_stats['generation_distribution'].get(gen, 0) + 1
    
    return render_template('breeding_search_integration.html',
                         orchids=orchids,
                         breeding_stats=breeding_stats,
                         breeding_type=breeding_type,
                         generation=generation,
                         parent_genus=parent_genus)

@app.route('/care-wheel')
def care_wheel():
    """Care Wheel Generator page"""
    return render_template('care_wheel.html')

@app.route('/map')
def world_map():
    """Interactive world map showing orchid locations"""
    return render_template('map.html')

def get_database_statistics():
    """Calculate comprehensive database statistics"""
    try:
        # Basic counts
        total_orchids = OrchidRecord.query.count()
        total_genera = db.session.query(func.count(func.distinct(OrchidRecord.genus))).scalar() or 0
        
        # Species count
        species_count = OrchidRecord.query.filter(
            OrchidRecord.is_species == True
        ).count()
        
        # Hybrid count
        hybrid_count = OrchidRecord.query.filter(
            OrchidRecord.is_hybrid == True
        ).count()
        
        # Intergeneric count (hybrids with different genera in parentage)
        intergeneric_count = 0
        try:
            # Check for records with different genera in parentage
            intergenerics = OrchidRecord.query.filter(
                and_(
                    OrchidRecord.is_hybrid == True,
                    or_(
                        OrchidRecord.pod_parent.isnot(None),
                        OrchidRecord.pollen_parent.isnot(None),
                        OrchidRecord.parentage_formula.isnot(None)
                    )
                )
            ).all()
            
            for orchid in intergenerics:
                if is_intergeneric_hybrid(orchid):
                    intergeneric_count += 1
                    
        except Exception as e:
            logger.error(f"Error calculating intergeneric count: {str(e)}")
        
        return {
            'total_orchids': total_orchids,
            'genera': total_genera,
            'species': species_count,
            'hybrids': hybrid_count,
            'intergenerics': intergeneric_count
        }
    except Exception as e:
        logger.error(f"Error calculating database statistics: {str(e)}")
        return {
            'total_orchids': 0,
            'genera': 0,
            'species': 0,
            'hybrids': 0,
            'intergenerics': 0
        }

def is_intergeneric_hybrid(orchid):
    """Check if an orchid is an intergeneric hybrid"""
    try:
        # Check if parents have different genera
        if orchid.pod_parent and orchid.pollen_parent:
            pod_genus = extract_genus_from_name(orchid.pod_parent)
            pollen_genus = extract_genus_from_name(orchid.pollen_parent)
            
            if pod_genus and pollen_genus and pod_genus != pollen_genus:
                return True
        
        # Check parentage formula for genus crosses (e.g., "Cattleya × Laelia")
        if orchid.parentage_formula:
            formula = orchid.parentage_formula.lower()
            # Look for genus names separated by × or x
            if '×' in formula or ' x ' in formula:
                # Extract genera from the formula
                genera_in_formula = extract_genera_from_formula(formula)
                if len(set(genera_in_formula)) > 1:
                    return True
        
        # Check scientific name for intergeneric notation (usually starts with ×)
        if orchid.scientific_name and orchid.scientific_name.startswith('×'):
            return True
            
        return False
    except Exception:
        return False

def extract_genus_from_name(name):
    """Extract genus from a plant name"""
    if not name:
        return None
    
    # Take the first word as genus
    parts = name.strip().split()
    if parts:
        genus = parts[0].replace('×', '').strip()
        return genus.capitalize()
    return None

def extract_genera_from_formula(formula):
    """Extract all genus names from a parentage formula"""
    import re
    # Find genus names (capitalized words at the start of names)
    genera = []
    words = re.findall(r'\b[A-Z][a-z]+', formula)
    for word in words:
        if len(word) > 2 and word not in ['var', 'Ver', 'Var']:  # Skip common abbreviations
            genera.append(word)
    return genera

@app.route('/api/database-statistics')
def api_database_statistics():
    """API endpoint for database statistics"""
    try:
        stats = get_database_statistics()
        return jsonify(stats)
    except Exception as e:
        logger.error(f"Error fetching database statistics: {str(e)}")
        return jsonify({
            'total_orchids': 0,
            'genera': 0,
            'species': 0,
            'hybrids': 0,
            'intergenerics': 0
        }), 500

@app.route('/api/live-stats')
def api_live_statistics():
    """API endpoint for real-time live statistics with auto-update support"""
    try:
        from orchid_statistics import get_homepage_statistics
        from datetime import datetime
        
        stats = get_homepage_statistics()
        
        # Add auto-update metadata
        stats['last_updated'] = datetime.now().strftime('%H:%M:%S')
        stats['timestamp'] = datetime.now().isoformat()
        
        # Force refresh from database for most current counts
        current_count = db.session.query(func.count(OrchidRecord.id)).scalar()
        photos_count = db.session.query(func.count(OrchidRecord.id)).filter(
            OrchidRecord.google_drive_id.isnot(None)
        ).scalar()
        
        # Override with live counts
        stats['total_orchids'] = current_count or stats.get('total_orchids', 4164)
        stats['photos_available'] = photos_count or stats.get('photos_available', 1337)
        
        return jsonify(stats)
    except Exception as e:
        logger.error(f"Error loading live statistics: {e}")
        return jsonify({
            'total_orchids': 4164,
            'total_genera': 396,
            'total_species': 2053,
            'photos_available': 1337,
            'last_updated': 'Error',
            'error': 'Statistics temporarily unavailable'
        }), 500

@app.route('/api/orchid-locations')
def orchid_locations_api():
    """API endpoint to get orchid location data for the map"""
    try:
        # Get all orchids with geographic data
        orchids = OrchidRecord.query.filter(
            or_(
                OrchidRecord.region.isnot(None),
                OrchidRecord.native_habitat.isnot(None)
            )
        ).all()
        
        locations = []
        for orchid in orchids:
            location_data = {}
            
            # Try to extract coordinates from AI extracted metadata if available
            if orchid.ai_extracted_metadata:
                try:
                    metadata = json.loads(orchid.ai_extracted_metadata) if isinstance(orchid.ai_extracted_metadata, str) else orchid.ai_extracted_metadata
                    if metadata and 'location' in metadata:
                        location_info = metadata['location']
                        if 'latitude' in location_info and 'longitude' in location_info:
                            location_data['lat'] = float(location_info['latitude'])
                            location_data['lng'] = float(location_info['longitude'])
                except (json.JSONDecodeError, KeyError, ValueError, TypeError):
                    pass
            
            # If no coordinates from metadata, try to derive approximate location from region/habitat
            if 'lat' not in location_data:
                location_data.update(_get_approximate_location(orchid.region, orchid.native_habitat))
            
            # Only include orchids with valid location data
            if 'lat' in location_data and 'lng' in location_data:
                location_data.update({
                    'id': orchid.id,
                    'name': orchid.display_name,
                    'scientific_name': orchid.scientific_name,
                    'genus': orchid.genus,
                    'species': orchid.species,
                    'region': orchid.region,
                    'habitat': orchid.native_habitat,
                    'image_url': orchid.image_url,
                    'climate': orchid.climate_preference,
                    'growth_habit': orchid.growth_habit,
                    'bloom_time': orchid.bloom_time
                })
                locations.append(location_data)
        
        return jsonify({
            'success': True,
            'locations': locations,
            'total_count': len(locations)
        })
    
    except Exception as e:
        logger.error(f"Error fetching orchid locations: {str(e)}")
        return jsonify({
            'success': False,
            'error': 'Failed to fetch orchid locations',
            'locations': [],
            'total_count': 0
        })

def _get_approximate_location(region, habitat):
    """Get approximate coordinates based on region/habitat text"""
    # Simple mapping of common regions to approximate coordinates
    region_coordinates = {
        # Asia
        'thailand': {'lat': 15.87, 'lng': 100.99},
        'malaysia': {'lat': 4.21, 'lng': 101.97},
        'singapore': {'lat': 1.35, 'lng': 103.82},
        'indonesia': {'lat': -0.79, 'lng': 113.92},
        'philippines': {'lat': 12.88, 'lng': 121.77},
        'vietnam': {'lat': 14.06, 'lng': 108.28},
        'laos': {'lat': 19.85, 'lng': 102.50},
        'cambodia': {'lat': 12.57, 'lng': 104.99},
        'myanmar': {'lat': 21.92, 'lng': 95.96},
        'china': {'lat': 35.86, 'lng': 104.20},
        'india': {'lat': 20.59, 'lng': 78.96},
        'japan': {'lat': 36.20, 'lng': 138.25},
        'south korea': {'lat': 35.91, 'lng': 127.77},
        
        # Americas
        'ecuador': {'lat': -1.83, 'lng': -78.18},
        'colombia': {'lat': 4.57, 'lng': -74.30},
        'peru': {'lat': -9.19, 'lng': -75.02},
        'brazil': {'lat': -14.24, 'lng': -51.93},
        'costa rica': {'lat': 9.75, 'lng': -83.75},
        'guatemala': {'lat': 15.78, 'lng': -90.23},
        'mexico': {'lat': 23.63, 'lng': -102.55},
        'panama': {'lat': 8.54, 'lng': -80.78},
        'venezuela': {'lat': 6.42, 'lng': -66.58},
        'bolivia': {'lat': -16.29, 'lng': -63.59},
        
        # Africa  
        'madagascar': {'lat': -18.77, 'lng': 46.87},
        'south africa': {'lat': -30.56, 'lng': 22.94},
        'kenya': {'lat': -0.02, 'lng': 37.91},
        'tanzania': {'lat': -6.37, 'lng': 34.89},
        'cameroon': {'lat': 7.37, 'lng': 12.35},
        
        # Oceania
        'australia': {'lat': -25.27, 'lng': 133.78},
        'new zealand': {'lat': -40.90, 'lng': 174.89},
        'new guinea': {'lat': -5.32, 'lng': 141.00},
        'papua new guinea': {'lat': -6.31, 'lng': 143.96},
        
        # Europe
        'mediterranean': {'lat': 40.00, 'lng': 18.00}
    }
    
    # Check region first
    if region:
        region_lower = region.lower()
        for key, coords in region_coordinates.items():
            if key in region_lower:
                return coords
    
    # Check habitat text
    if habitat:
        habitat_lower = habitat.lower()
        for key, coords in region_coordinates.items():
            if key in habitat_lower:
                return coords
    
    return {}

def _get_origin_summary(orchid):
    """Get comprehensive origin summary for an orchid"""
    origin_parts = []
    
    if orchid.country:
        origin_parts.append(orchid.country)
    if orchid.state_province:
        origin_parts.append(orchid.state_province)
    if orchid.region:
        origin_parts.append(orchid.region)
    
    if origin_parts:
        return f"Native to {', '.join(origin_parts)}"
    elif orchid.native_habitat:
        return f"Found in {orchid.native_habitat}"
    else:
        return "Geographic origin not specified"

def _get_habitat_description(orchid):
    """Get detailed habitat description"""
    habitat_parts = []
    
    if orchid.native_habitat:
        habitat_parts.append(orchid.native_habitat)
    
    if orchid.growth_habit:
        habit_desc = {
            'epiphytic': 'grows on trees and other plants',
            'terrestrial': 'grows in soil on the ground',
            'lithophytic': 'grows on rocks and cliff faces'
        }
        if orchid.growth_habit in habit_desc:
            habitat_parts.append(habit_desc[orchid.growth_habit])
    
    if orchid.climate_preference:
        climate_desc = {
            'cool': 'prefers cool climates (55-75°F)',
            'intermediate': 'thrives in intermediate climates (65-85°F)',
            'warm': 'requires warm climates (70-90°F)'
        }
        if orchid.climate_preference in climate_desc:
            habitat_parts.append(climate_desc[orchid.climate_preference])
    
    return '. '.join(habitat_parts).capitalize() if habitat_parts else None

def _get_climate_analysis(orchid):
    """Get climate analysis based on available data"""
    analysis = []
    
    if orchid.bloom_time:
        analysis.append(f"Blooming season: {orchid.bloom_time}")
    
    if orchid.temperature_range:
        analysis.append(f"Temperature range: {orchid.temperature_range}")
    
    if orchid.light_requirements:
        analysis.append(f"Light needs: {orchid.light_requirements}")
    
    return analysis

def _get_conservation_notes(orchid):
    """Get conservation and rarity information"""
    notes = []
    
    if orchid.conservation_status_clues:
        notes.append(orchid.conservation_status_clues)
    
    # Add general conservation awareness
    if orchid.native_habitat and any(keyword in orchid.native_habitat.lower() 
                                   for keyword in ['cloud forest', 'montane', 'endemic']):
        notes.append("This orchid may come from a sensitive ecosystem. Please support conservation efforts.")
    
    return notes

def _get_cultural_recommendations(orchid):
    """Get cultural recommendations from database and external sources"""
    recommendations = {
        'light': orchid.light_requirements,
        'temperature': orchid.temperature_range,
        'water': orchid.water_requirements,
        'fertilizer': orchid.fertilizer_needs,
        'general_notes': orchid.cultural_notes,
        'growing_tips': []
    }
    
    # Add specific growing tips based on growth habit
    if orchid.growth_habit == 'epiphytic':
        recommendations['growing_tips'].append("Use well-draining bark mix or mount on tree fern")
        recommendations['growing_tips'].append("Provide good air circulation around roots")
    elif orchid.growth_habit == 'terrestrial':
        recommendations['growing_tips'].append("Use terrestrial orchid mix with good drainage")
        recommendations['growing_tips'].append("Keep evenly moist during growing season")
    
    # Add climate-based tips
    if orchid.climate_preference == 'cool':
        recommendations['growing_tips'].append("Provide cool nights (10-15°F drop)")
    elif orchid.climate_preference == 'warm':
        recommendations['growing_tips'].append("Maintain consistent warmth year-round")
    
    return recommendations

@app.route('/orchid/<int:id>')
def orchid_detail(id):
    """Display detailed orchid information with enhanced metadata and care recommendations"""
    try:
        # Query with specific columns to avoid column mapping issues
        orchid = db.session.query(OrchidRecord).filter(OrchidRecord.id == id).first()
        
        if not orchid:
            abort(404)
        
        # Safely increment view count
        try:
            if orchid.view_count is None:
                orchid.view_count = 1
            else:
                orchid.view_count += 1
            db.session.commit()
        except Exception as e:
            logger.warning(f"Could not update view count: {e}")
            db.session.rollback()
        
        # Get related orchids with error handling
        related_orchids = []
        try:
            if orchid.genus:
                related_orchids = db.session.query(OrchidRecord).filter(
                    OrchidRecord.genus == orchid.genus,
                    OrchidRecord.id != orchid.id,
                    OrchidRecord.google_drive_id.isnot(None)
                ).limit(4).all()
        except Exception as e:
            logger.warning(f"Could not load related orchids: {e}")
        
        # Get comprehensive care data from multiple sources
        care_data = None
        care_available = False
        baker_notes = None
        
        try:
            from care_wheel_generator import ORCHID_CARE_DATA, extrapolate_species_care
            from attribution_system import attribution_manager, Sources
            
            if orchid.genus in ORCHID_CARE_DATA:
                care_available = True
                # Get species-specific care if available
                if orchid.species:
                    care_data = extrapolate_species_care(orchid.genus, orchid.species)
                else:
                    care_data = ORCHID_CARE_DATA[orchid.genus].copy()
                
                # Add attribution information
                sources_used = [Sources.BAKER_CULTURE, Sources.AOS, Sources.RHS]
                care_data['attribution'] = attribution_manager.create_attribution_block(
                    sources_used, format_type='html'
                )
                
        except Exception as e:
            logger.warning(f"Could not load care data for {orchid.genus}: {e}")
        
        # Get habitat and geographic insights
        habitat_info = {
            'origin_summary': _get_origin_summary(orchid),
            'habitat_description': _get_habitat_description(orchid),
            'climate_analysis': _get_climate_analysis(orchid),
            'conservation_notes': _get_conservation_notes(orchid)
        }
        
        # Get cultural recommendations from database fields
        cultural_recommendations = _get_cultural_recommendations(orchid)
        
        return render_template('orchid_detail.html', 
                             orchid=orchid, 
                             related_orchids=related_orchids,
                             care_data=care_data,
                             care_available=care_available,
                             habitat_info=habitat_info,
                             cultural_recommendations=cultural_recommendations)
        
    except Exception as e:
        logger.error(f"Error loading orchid detail for ID {id}: {e}")
        flash(f"Error loading orchid details: {str(e)}", 'error')
        return redirect(url_for('gallery'))

@app.route('/mission')
def mission():
    """Display mission statement and support information"""
    return render_template('mission.html')

@app.route('/admin/baker-extrapolation')
def admin_baker_extrapolation():
    """Admin interface for Baker culture extrapolation system"""
    return render_template('admin/baker_extrapolation.html')

@app.route('/admin')
def admin():
    """Admin interface for batch operations"""
    try:
        # Get recent uploads with error handling
        recent_uploads = []
        try:
            recent_uploads = UserUpload.query.order_by(UserUpload.created_at.desc()).limit(10).all()
        except Exception as e:
            logger.warning(f"Could not load recent uploads: {e}")
        
        # Get scraping logs with error handling
        recent_scrapes = []
        try:
            recent_scrapes = ScrapingLog.query.order_by(ScrapingLog.created_at.desc()).limit(10).all()
        except Exception as e:
            logger.warning(f"Could not load scraping logs: {e}")
        
        # Get statistics with error handling
        stats = {
            'total_orchids': 0,
            'pending_uploads': 0,
            'validated_orchids': 0,
            'featured_orchids': 0
        }
        
        try:
            stats['total_orchids'] = OrchidRecord.query.count()
            stats['validated_orchids'] = OrchidRecord.query.filter_by(validation_status='validated').count()
            stats['featured_orchids'] = OrchidRecord.query.filter_by(is_featured=True).count()
        except Exception as e:
            logger.warning(f"Could not load statistics: {e}")
        
        try:
            stats['pending_uploads'] = UserUpload.query.filter_by(processing_status='pending').count()
        except Exception as e:
            logger.warning(f"Could not load upload stats: {e}")
        
        return render_template('admin.html',
                             recent_uploads=recent_uploads,
                             recent_scrapes=recent_scrapes,
                             stats=stats)
    except Exception as e:
        logger.error(f"Admin page error: {e}")
        return render_template('error.html', error="Admin dashboard temporarily unavailable", details=str(e)), 500

@app.route('/admin/drive-import')
def admin_drive_import():
    """Admin Google Drive import interface"""
    return render_template('admin/drive_import.html')

@app.route('/api/drive-import/preview-folder', methods=['POST'])
def api_preview_drive_folder():
    """Preview contents of a Google Drive folder"""
    try:
        from google_drive_service import get_folder_contents
        from drive_importer import extract_folder_id_from_url
        
        data = request.get_json()
        folder_url = data.get('folder_url', '')
        
        # Extract folder ID from URL
        folder_id = extract_folder_id_from_url(folder_url)
        if not folder_id:
            return jsonify({'error': 'Invalid Google Drive folder URL'}), 400
        
        # Get folder contents
        files = get_folder_contents(folder_id)
        
        return jsonify({
            'success': True,
            'folder_id': folder_id,
            'file_count': len(files),
            'files': files[:20],  # Preview first 20 files
            'total_files': len(files)
        })
        
    except Exception as e:
        logger.error(f"Error previewing folder: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/drive-import/import-folder', methods=['POST'])
def api_import_drive_folder():
    """Import orchid images from a Google Drive folder"""
    try:
        from google_drive_service import batch_import_from_drive_folder
        from drive_importer import extract_folder_id_from_url
        
        data = request.get_json()
        folder_url = data.get('folder_url', '')
        limit = int(data.get('limit', 50))
        auto_process = data.get('auto_process', True)
        
        # Extract folder ID from URL
        folder_id = extract_folder_id_from_url(folder_url)
        if not folder_id:
            return jsonify({'error': 'Invalid Google Drive folder URL'}), 400
        
        # Perform batch import
        import_result = batch_import_from_drive_folder(folder_id, limit)
        
        if not import_result.get('success'):
            return jsonify({'error': import_result.get('error', 'Import failed')}), 500
        
        return jsonify({
            'success': True,
            'imported_count': import_result.get('imported_count', 0),
            'skipped_count': import_result.get('skipped_count', 0),
            'error_count': import_result.get('error_count', 0),
            'details': import_result.get('details', [])
        })
        
    except Exception as e:
        logger.error(f"Error importing folder: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/scrape/<source>')
def trigger_scrape(source):
    """Trigger scraping for a specific source"""
    try:
        if source == 'gary':
            results = scrape_gary_yong_gee()
        elif source == 'roberta':
            results = scrape_roberta_fox()
        else:
            return jsonify({'error': 'Unknown source'}), 400
        
        return jsonify(results)
    except Exception as e:
        logger.error(f"Scraping error: {str(e)}")
        return jsonify({'error': str(e)}), 500

@app.route('/weather')
def weather_dashboard():
    """Weather dashboard for orchid growing conditions"""
    try:
        # Get user locations
        locations = UserLocation.query.filter_by(track_weather=True).all()
        
        # Get recent weather data
        recent_weather = WeatherData.query.filter(
            WeatherData.recorded_at >= datetime.utcnow() - timedelta(days=7),
            WeatherData.is_forecast == False
        ).order_by(WeatherData.recorded_at.desc()).limit(50).all()
        
        # Get active weather alerts
        active_alerts = WeatherAlert.query.filter(
            WeatherAlert.is_active == True,
            WeatherAlert.expires_at > datetime.utcnow()
        ).order_by(WeatherAlert.triggered_at.desc()).all()
        
        # Get forecast data
        forecast_data = WeatherData.query.filter(
            WeatherData.is_forecast == True,
            WeatherData.recorded_at >= datetime.utcnow()
        ).order_by(WeatherData.recorded_at.asc()).limit(14).all()
        
        return render_template('weather/dashboard.html',
                             locations=locations,
                             recent_weather=recent_weather,
                             active_alerts=active_alerts,
                             forecast_data=forecast_data)
        
    except Exception as e:
        logger.error(f"Error loading weather dashboard: {str(e)}")
        flash('Error loading weather dashboard', 'error')
        return redirect(url_for('index'))

@app.route('/weather/location/add', methods=['GET', 'POST'])
def add_weather_location():
    """Add a new weather tracking location"""
    if request.method == 'POST':
        try:
            data = request.get_json() if request.is_json else request.form
            
            location_name = data.get('name')
            city = data.get('city')
            state_province = data.get('state_province', '')
            country = data.get('country', '')
            growing_type = data.get('growing_type', 'outdoor')
            
            # Try to get coordinates
            full_location = f"{city}, {state_province}, {country}".strip(', ')
            coordinates = get_coordinates_from_location(full_location)
            
            if not coordinates:
                return jsonify({'error': 'Could not find coordinates for this location'}), 400
            
            latitude, longitude = coordinates
            
            # Create new location
            location = UserLocation()
            location.name = location_name
            location.city = city
            location.state_province = state_province
            location.country = country
            location.latitude = latitude
            location.longitude = longitude
            location.growing_type = growing_type
            location.track_weather = True
            
            db.session.add(location)
            db.session.commit()
            
            # Fetch initial weather data
            if location_name:
                WeatherService.get_current_weather(latitude, longitude, location_name)
            
            if request.is_json:
                return jsonify({'success': True, 'location_id': location.id})
            else:
                flash(f'Added weather tracking for {location_name}', 'success')
                return redirect(url_for('weather_dashboard'))
                
        except Exception as e:
            logger.error(f"Error adding weather location: {str(e)}")
            if request.is_json:
                return jsonify({'error': str(e)}), 500
            else:
                flash('Error adding location', 'error')
                return redirect(url_for('weather_dashboard'))
    
    return render_template('weather/add_location.html')

@app.route('/weather/current/<int:location_id>')
def get_current_weather(location_id):
    """Get current weather for a specific location"""
    try:
        location = UserLocation.query.get_or_404(location_id)
        
        # Fetch fresh weather data
        weather = WeatherService.get_current_weather(
            location.latitude, 
            location.longitude, 
            location.name
        )
        
        if not weather:
            return jsonify({'error': 'Could not fetch weather data'}), 500
        
        # Check for alerts
        alerts = WeatherService.check_orchid_weather_alerts(location)
        
        # Get growing conditions summary
        conditions = WeatherService.get_growing_conditions_summary(location, 7)
        
        return jsonify({
            'weather': {
                'temperature': weather.temperature,
                'humidity': weather.humidity,
                'pressure': weather.pressure,
                'wind_speed': weather.wind_speed,
                'description': weather.description,
                'vpd': weather.vpd,
                'orchid_friendly': weather.is_orchid_friendly(),
                'recorded_at': weather.recorded_at.isoformat()
            },
            'alerts': [
                {
                    'type': alert.alert_type,
                    'severity': alert.severity,
                    'title': alert.title,
                    'message': alert.message,
                    'advice': alert.orchid_care_advice
                } for alert in alerts
            ],
            'conditions': conditions
        })
        
    except Exception as e:
        logger.error(f"Error fetching current weather: {str(e)}")
        return jsonify({'error': str(e)}), 500

@app.route('/weather/historical/<int:location_id>')
def get_historical_weather(location_id):
    """Get historical weather data for analysis"""
    try:
        location = UserLocation.query.get_or_404(location_id)
        days = request.args.get('days', 30, type=int)
        
        # Get existing historical data
        existing_data = WeatherData.query.filter(
            WeatherData.latitude == location.latitude,
            WeatherData.longitude == location.longitude,
            WeatherData.recorded_at >= datetime.utcnow() - timedelta(days=days),
            WeatherData.is_forecast == False
        ).order_by(WeatherData.recorded_at.desc()).all()
        
        # If we have less than expected, fetch more
        if len(existing_data) < days * 0.8:  # Allow for some missing days
            end_date = datetime.utcnow()
            start_date = end_date - timedelta(days=days)
            
            new_data = WeatherService.get_historical_weather(
                location.latitude,
                location.longitude,
                start_date,
                end_date,
                location.name
            )
            
            # Refresh query after fetching new data
            existing_data = WeatherData.query.filter(
                WeatherData.latitude == location.latitude,
                WeatherData.longitude == location.longitude,
                WeatherData.recorded_at >= start_date,
                WeatherData.is_forecast == False
            ).order_by(WeatherData.recorded_at.desc()).all()
        
        # Format data for charts
        chart_data = []
        for weather in existing_data:
            chart_data.append({
                'date': weather.recorded_at.strftime('%Y-%m-%d'),
                'temperature': weather.temperature,
                'temperature_min': weather.temperature_min,
                'temperature_max': weather.temperature_max,
                'humidity': weather.humidity,
                'precipitation': weather.precipitation,
                'vpd': weather.vpd,
                'orchid_friendly': weather.is_orchid_friendly()
            })
        
        return jsonify({
            'data': chart_data,
            'summary': WeatherService.get_growing_conditions_summary(location, days)
        })
        
    except Exception as e:
        logger.error(f"Error fetching historical weather: {str(e)}")
        return jsonify({'error': str(e)}), 500

@app.route('/weather/forecast/<int:location_id>')
def get_weather_forecast(location_id):
    """Get weather forecast for planning"""
    try:
        location = UserLocation.query.get_or_404(location_id)
        days = request.args.get('days', 7, type=int)
        
        # Fetch fresh forecast
        forecast = WeatherService.get_weather_forecast(
            location.latitude,
            location.longitude,
            days,
            location.name
        )
        
        forecast_data = []
        for weather in forecast:
            forecast_data.append({
                'date': weather.recorded_at.strftime('%Y-%m-%d'),
                'temperature_min': weather.temperature_min,
                'temperature_max': weather.temperature_max,
                'humidity': weather.humidity,
                'precipitation': weather.precipitation,
                'description': weather.description,
                'weather_code': weather.weather_code,
                'orchid_friendly': weather.is_orchid_friendly()
            })
        
        return jsonify({'forecast': forecast_data})
        
    except Exception as e:
        logger.error(f"Error fetching weather forecast: {str(e)}")
        return jsonify({'error': str(e)}), 500

@app.route('/weather/alerts')
def weather_alerts():
    """Display weather alerts for orchid care"""
    try:
        # Get all active alerts
        alerts = WeatherAlert.query.filter(
            WeatherAlert.is_active == True,
            WeatherAlert.expires_at > datetime.utcnow()
        ).order_by(WeatherAlert.severity.desc(), WeatherAlert.triggered_at.desc()).all()
        
        return render_template('weather/alerts.html', alerts=alerts)
        
    except Exception as e:
        logger.error(f"Error loading weather alerts: {str(e)}")
        flash('Error loading weather alerts', 'error')
        return redirect(url_for('weather_dashboard'))

@app.route('/weather/alert/<int:alert_id>/acknowledge', methods=['POST'])
def acknowledge_alert(alert_id):
    """Acknowledge a weather alert"""
    try:
        alert = WeatherAlert.query.get_or_404(alert_id)
        alert.is_acknowledged = True
        alert.acknowledged_at = datetime.utcnow()
        
        db.session.commit()
        
        if request.is_json:
            return jsonify({'success': True})
        else:
            flash('Alert acknowledged', 'success')
            return redirect(url_for('weather_alerts'))
            
    except Exception as e:
        logger.error(f"Error acknowledging alert: {str(e)}")
        if request.is_json:
            return jsonify({'error': str(e)}), 500
        else:
            flash('Error acknowledging alert', 'error')
            return redirect(url_for('weather_alerts'))

@app.route('/weather/orchid-correlation')
def weather_orchid_correlation():
    """Analyze correlation between weather and orchid performance"""
    try:
        # Get orchids with recent activity or photos
        recent_orchids = OrchidRecord.query.filter(
            OrchidRecord.created_at >= datetime.utcnow() - timedelta(days=90)
        ).all()
        
        # Get weather data for the same period
        recent_weather = WeatherData.query.filter(
            WeatherData.recorded_at >= datetime.utcnow() - timedelta(days=90),
            WeatherData.is_forecast == False
        ).all()
        
        # Group weather by location
        weather_by_location = {}
        for weather in recent_weather:
            key = f"{weather.latitude},{weather.longitude}"
            if key not in weather_by_location:
                weather_by_location[key] = []
            weather_by_location[key].append(weather)
        
        # Analyze correlations
        correlations = {
            'temperature_trends': [],
            'humidity_patterns': [],
            'vpd_analysis': [],
            'orchid_friendly_days': 0,
            'recommendations': []
        }
        
        # Calculate orchid-friendly days
        orchid_friendly_count = sum(1 for w in recent_weather if w.is_orchid_friendly())
        correlations['orchid_friendly_days'] = orchid_friendly_count
        
        # Add basic recommendations
        if recent_weather:
            avg_temp = sum(w.temperature for w in recent_weather if w.temperature) / len([w for w in recent_weather if w.temperature])
            avg_humidity = sum(w.humidity for w in recent_weather if w.humidity) / len([w for w in recent_weather if w.humidity])
            
            if avg_temp < 18:
                correlations['recommendations'].append("Temperatures below optimal range - consider supplemental heating")
            if avg_humidity < 50:
                correlations['recommendations'].append("Humidity below optimal range - increase humidity for better orchid health")
        
        return render_template('weather/correlation.html',
                             recent_orchids=recent_orchids,
                             weather_data=recent_weather,
                             correlations=correlations)
        
    except Exception as e:
        logger.error(f"Error analyzing weather-orchid correlation: {str(e)}")
        flash('Error analyzing correlations', 'error')
        return redirect(url_for('weather_dashboard'))

@app.route('/api/validate/<int:orchid_id>', methods=['POST'])
def validate_orchid(orchid_id):
    """Validate an orchid record"""
    orchid = OrchidRecord.query.get_or_404(orchid_id)
    action = request.json.get('action', 'validate')
    
    if action == 'validate':
        orchid.validation_status = 'validated'
    elif action == 'reject':
        orchid.validation_status = 'rejected'
    elif action == 'feature':
        orchid.is_featured = not orchid.is_featured
    
    db.session.commit()
    return jsonify({'status': 'success'})

@app.route('/api/orchid-of-day')
def api_orchid_of_day():
    """API endpoint for orchid of the day widget"""
    orchid = get_orchid_of_the_day()
    if orchid:
        # Fix image URL - never return GBIF occurrence URLs as images
        image_url = orchid.image_url
        
        # Check if the image_url is a broken GBIF occurrence URL
        if image_url and ('gbif.org/occurrence/' in image_url or 'occurrence/' in image_url):
            # Use Google Drive ID if available
            if orchid.google_drive_id:
                image_url = f"/api/drive-photo/{orchid.google_drive_id}"
            else:
                # Fallback to placeholder
                image_url = "/static/images/orchid_placeholder.svg"
                logger.warning(f"Fixed GBIF occurrence URL for orchid {orchid.id}: {orchid.image_url}")
        
        # Ensure image_url is valid
        if not image_url:
            if orchid.google_drive_id:
                image_url = f"/api/drive-photo/{orchid.google_drive_id}"
            else:
                image_url = "/static/images/orchid_placeholder.svg"
        
        return jsonify({
            'id': orchid.id,
            'name': orchid.display_name,
            'scientific_name': orchid.scientific_name,
            'description': orchid.ai_description,
            'image_url': image_url,
            'cultural_notes': orchid.cultural_notes
        })
    return jsonify({'error': 'No orchid found'}), 404

@app.route('/api/orchid/<int:orchid_id>/ecosystem')
def api_orchid_ecosystem(orchid_id):
    """API endpoint for individual orchid ecosystem data"""
    try:
        orchid = OrchidRecord.query.get_or_404(orchid_id)
        
        ecosystem_data = {
            'id': orchid.id,
            'scientific_name': orchid.scientific_name,
            'pollinators': [],  # Field doesn't exist in model
            'mycorrhizal_fungi': [],  # Field doesn't exist in model
            'native_habitat': orchid.native_habitat,
            'companion_plants': [],  # Field doesn't exist in model
            'temperature_range': orchid.temperature_range,
            'humidity_preference': 'Unknown',  # Field doesn't exist
            'light_requirements': orchid.light_requirements,
            'flowering_time': orchid.flowering_time,
            'climate_preference': orchid.climate_preference,
            'growth_habit': orchid.growth_habit,
            'environmental_zones': orchid.environmental_zones,
            'botanical_features': orchid.botanical_features
        }
        
        return jsonify(ecosystem_data)
        
    except Exception as e:
        logger.error(f"Ecosystem data error: {e}")
        return jsonify({'error': 'Failed to load ecosystem data'}), 500

@app.route('/api/orchid/<int:orchid_id>/cultural')
def api_orchid_cultural(orchid_id):
    """API endpoint for individual orchid cultural/care data"""
    try:
        orchid = OrchidRecord.query.get_or_404(orchid_id)
        
        cultural_data = {
            'id': orchid.id,
            'scientific_name': orchid.scientific_name,
            'water_requirements': orchid.water_requirements,
            'light_requirements': orchid.light_requirements,
            'temperature_range': orchid.temperature_range,
            'humidity_preference': orchid.humidity_preference,
            'fertilizer_needs': orchid.fertilizer_needs,
            'potting_media': orchid.potting_media,
            'repotting': orchid.repotting,
            'climate_preference': orchid.climate_preference,
            'growth_habit': orchid.growth_habit,
            'flowering_time': orchid.flowering_time,
            'cultural_notes': orchid.cultural_notes,
            'ai_description': orchid.ai_description
        }
        
        return jsonify(cultural_data)
        
    except Exception as e:
        logger.error(f"Cultural data error: {e}")
        return jsonify({'error': 'Failed to load cultural data'}), 500

@app.route('/api/orchid/<int:orchid_id>/coordinates')
def api_orchid_coordinates(orchid_id):
    """API endpoint for individual orchid coordinates"""
    try:
        orchid = OrchidRecord.query.get_or_404(orchid_id)
        
        coordinate_data = {
            'id': orchid.id,
            'latitude': float(orchid.decimal_latitude) if orchid.decimal_latitude else None,
            'longitude': float(orchid.decimal_longitude) if orchid.decimal_longitude else None,
            'region': orchid.region,
            'country': orchid.country,
            'native_habitat': orchid.native_habitat
        }
        
        return jsonify(coordinate_data)
        
    except Exception as e:
        logger.error(f"Coordinates error: {e}")
        return jsonify({'error': 'Failed to load coordinate data'}), 500

@app.route('/api/featured-orchids')
def api_featured_orchids():
    """API endpoint for featured orchids"""
    try:
        featured = OrchidRecord.query.filter(
            OrchidRecord.is_featured == True,
            OrchidRecord.google_drive_id.isnot(None)
        ).limit(6).all()
        
        if not featured:
            # Get recent high-quality orchids as fallback
            featured = OrchidRecord.query.filter(
                OrchidRecord.google_drive_id.isnot(None),
                OrchidRecord.ai_description.isnot(None)
            ).order_by(OrchidRecord.created_at.desc()).limit(6).all()
        
        orchids = []
        for orchid in featured:
            orchids.append({
                'id': orchid.id,
                'name': orchid.display_name,
                'scientific_name': orchid.scientific_name,
                'description': orchid.ai_description,
                'image_url': orchid.image_url,
                'genus': orchid.genus
            })
        
        return jsonify({'orchids': orchids})
        
    except Exception as e:
        logger.error(f"Featured orchids error: {e}")
        return jsonify({'error': 'Failed to fetch featured orchids'}), 500

@app.route('/api/orchids-map-data')
def api_orchids_map_data():
    """API endpoint for orchid map data with filtering"""
    try:
        # Get filter parameters
        genus_filter = request.args.get('genus', '').strip()
        climate_filter = request.args.get('climate', '').strip()
        
        # Build query
        query = OrchidRecord.query.filter(
            OrchidRecord.validation_status == 'validated'
        )
        
        # Apply filters
        if genus_filter:
            query = query.filter(OrchidRecord.genus == genus_filter)
        
        if climate_filter:
            query = query.filter(OrchidRecord.climate_preference == climate_filter)
        
        orchids = query.all()
        
        # Convert to map data format
        map_data = []
        for orchid in orchids:
            # Get location data using existing function
            location_data = _get_approximate_location(orchid.region, orchid.native_habitat)
            
            if location_data and 'lat' in location_data and 'lng' in location_data:  # Only include orchids with location data
                map_data.append({
                    'id': orchid.id,
                    'name': orchid.display_name,
                    'scientific_name': orchid.scientific_name,
                    'genus': orchid.genus,
                    'region': orchid.region,
                    'climate': orchid.climate_preference,
                    'growth_habit': orchid.growth_habit,
                    'image_url': orchid.image_url,
                    'lat': location_data.get('lat'),
                    'lng': location_data.get('lng'),
                    'description': orchid.ai_description[:200] + '...' if orchid.ai_description and len(orchid.ai_description) > 200 else orchid.ai_description
                })
        
        return jsonify({
            'orchids': map_data,
            'total': len(map_data)
        })
        
    except Exception as e:
        logger.error(f"Error fetching map data: {str(e)}")
        return jsonify({'error': 'Failed to fetch map data'}), 500

@app.route('/api/orchids-filters')
def api_orchids_filters():
    """API endpoint for available filter options"""
    try:
        # Get unique genera
        genera_query = db.session.query(OrchidRecord.genus).filter(
            OrchidRecord.genus.isnot(None),
            OrchidRecord.validation_status == 'validated'
        ).distinct()
        genera = [g[0] for g in genera_query if g[0]]
        
        # Get unique climate preferences
        climate_query = db.session.query(OrchidRecord.climate_preference).filter(
            OrchidRecord.climate_preference.isnot(None),
            OrchidRecord.validation_status == 'validated'
        ).distinct()
        climates = [c[0] for c in climate_query if c[0]]
        
        # Get unique growth habits
        habit_query = db.session.query(OrchidRecord.growth_habit).filter(
            OrchidRecord.growth_habit.isnot(None),
            OrchidRecord.validation_status == 'validated'
        ).distinct()
        growth_habits = [h[0] for h in habit_query if h[0]]
        
        return jsonify({
            'genera': sorted(genera),
            'climates': sorted(climates),
            'growth_habits': sorted(growth_habits)
        })
        
    except Exception as e:
        logger.error(f"Error fetching filter options: {str(e)}")
        return jsonify({'error': 'Failed to fetch filter options'}), 500

@app.route('/feedback', methods=['GET', 'POST'])
def feedback():
    """User feedback submission form"""
    if request.method == 'POST':
        try:
            # Get form data
            name = request.form.get('name', '').strip()
            email = request.form.get('email', '').strip()
            feedback_type = request.form.get('feedback_type', '').strip()
            subject = request.form.get('subject', '').strip()
            message = request.form.get('message', '').strip()
            
            # Basic validation
            if not all([name, email, feedback_type, subject, message]):
                flash('Please fill in all required fields.', 'error')
                return render_template('feedback.html')
            
            # Create feedback record
            feedback_record = UserFeedback()
            feedback_record.name = name
            feedback_record.email = email
            feedback_record.feedback_type = feedback_type
            feedback_record.subject = subject
            feedback_record.message = message
            feedback_record.page_url = request.referrer or request.url
            feedback_record.browser_info = request.headers.get('User-Agent', '')
            
            db.session.add(feedback_record)
            db.session.commit()
            
            flash(f'Thank you for your {feedback_type.replace("_", " ")}! We\'ll review it and get back to you if needed.', 'success')
            return redirect(url_for('feedback'))
            
        except Exception as e:
            logger.error(f"Error submitting feedback: {str(e)}")
            flash('There was an error submitting your feedback. Please try again.', 'error')
            db.session.rollback()
    
    return render_template('feedback.html')

@app.route('/admin/feedback')
def admin_feedback():
    """Admin view for managing feedback"""
    if not current_user.is_authenticated or not hasattr(current_user, 'is_admin') or not current_user.is_admin:
        flash('Access denied. Admin privileges required.', 'error')
        return redirect(url_for('index'))
    
    # Get feedback with pagination
    page = request.args.get('page', 1, type=int)
    status_filter = request.args.get('status', 'all')
    type_filter = request.args.get('type', 'all')
    
    query = UserFeedback.query
    
    if status_filter != 'all':
        query = query.filter(UserFeedback.status == status_filter)
    
    if type_filter != 'all':
        query = query.filter(UserFeedback.feedback_type == type_filter)
    
    feedback_items = query.order_by(UserFeedback.created_at.desc()).paginate(
        page=page, per_page=20, error_out=False
    )
    
    # Get summary statistics
    total_feedback = UserFeedback.query.count()
    new_feedback = UserFeedback.query.filter(UserFeedback.status == 'new').count()
    bug_reports = UserFeedback.query.filter(UserFeedback.feedback_type == 'bug_report').count()
    feature_requests = UserFeedback.query.filter(UserFeedback.feedback_type == 'feature_request').count()
    
    return render_template('admin/feedback.html', 
                         feedback_items=feedback_items,
                         total_feedback=total_feedback,
                         new_feedback=new_feedback,
                         bug_reports=bug_reports,
                         feature_requests=feature_requests,
                         status_filter=status_filter,
                         type_filter=type_filter)

@app.route('/admin/feedback/<int:feedback_id>/update', methods=['POST'])
def update_feedback_status(feedback_id):
    """Update feedback status and admin notes"""
    if not current_user.is_authenticated or not hasattr(current_user, 'is_admin') or not current_user.is_admin:
        return jsonify({'error': 'Access denied'}), 403
    
    feedback_item = UserFeedback.query.get_or_404(feedback_id)
    
    try:
        new_status = request.json.get('status')
        admin_notes = request.json.get('admin_notes', '')
        
        if new_status:
            feedback_item.status = new_status
        
        if admin_notes:
            feedback_item.admin_notes = admin_notes
        
        feedback_item.updated_at = datetime.utcnow()
        db.session.commit()
        
        return jsonify({'success': True, 'message': 'Feedback updated successfully'})
        
    except Exception as e:
        logger.error(f"Error updating feedback: {str(e)}")
        db.session.rollback()
        return jsonify({'error': 'Failed to update feedback'}), 500

@app.errorhandler(404)
def not_found_error(error):
    return render_template('404.html'), 404

@app.errorhandler(500)
def internal_error(error):
    db.session.rollback()
    return render_template('500.html'), 500

@app.route('/admin/data-import')
def admin_data_import():
    """Data import management interface"""
    # Get database statistics
    stats = {}
    try:
        stats['total_orchids'] = db.session.query(func.count(OrchidRecord.id)).scalar() or 0
        stats['validated_orchids'] = db.session.query(func.count(OrchidRecord.id)).filter_by(validation_status='validated').scalar() or 0
        
        # Data source breakdown
        stats['by_source'] = {}
        sources = ['upload', 'scrape_gary', 'scrape_roberta', 'legacy', 'drive_import']
        for source in sources:
            count = db.session.query(func.count(OrchidRecord.id)).filter_by(ingestion_source=source).scalar() or 0
            stats['by_source'][source] = count
            
    except Exception as e:
        logger.error(f"Error fetching data import stats: {str(e)}")
        db.session.rollback()
        stats = {
            'total_orchids': 0,
            'validated_orchids': 0,
            'by_source': {}
        }
    
    return render_template('admin_data_import.html', stats=stats)

@app.route('/admin/run-scraping', methods=['POST'])
def admin_run_scraping():
    """Trigger web scraping to collect orchid data"""
    try:
        data = request.get_json()
        source = data.get('source', 'all')
        
        results = {'success': True, 'new_records': 0, 'errors': 0}
        
        # Import and run scrapers
        try:
            from web_scraper import scrape_gary_yong_gee, scrape_roberta_fox
            
            if source in ['all', 'gary']:
                gary_results = scrape_gary_yong_gee()
                results['new_records'] += gary_results.get('processed', 0)
                results['errors'] += gary_results.get('errors', 0)
                
            if source in ['all', 'roberta']:
                roberta_results = scrape_roberta_fox()
                results['new_records'] += roberta_results.get('processed', 0)
                results['errors'] += roberta_results.get('errors', 0)
                
        except ImportError as e:
            return jsonify({'success': False, 'error': f'Scraping modules not available: {e}'}), 500
        except Exception as e:
            return jsonify({'success': False, 'error': f'Scraping failed: {e}'}), 500
        
        return jsonify(results)
        
    except Exception as e:
        logger.error(f"Error running scraping: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/drive-photo/<file_id>')
def get_drive_photo(file_id):
    """Proxy Google Drive photos for display - PRODUCTION READY"""
    try:
        # Construct Google Drive image URL
        drive_url = f'https://drive.google.com/uc?export=view&id={file_id}'
        
        # Get the image with optimized settings
        response = requests.get(drive_url, timeout=15, stream=True)
        if response.status_code == 200:
            # Add caching headers for better performance
            headers = {
                'Content-Type': response.headers.get('Content-Type', 'image/jpeg'),
                'Cache-Control': 'public, max-age=3600',  # Cache for 1 hour
                'ETag': f'drive-{file_id}',
                'X-Image-Source': 'Google Drive'
            }
            return response.content, 200, headers
        else:
            # Return placeholder if image not accessible
            logger.warning(f"Drive photo {file_id} returned {response.status_code}")
            return redirect('/static/images/orchid_placeholder.svg')
            
    except requests.Timeout:
        logger.warning(f"Drive photo {file_id} timed out - serving placeholder")
        return redirect('/static/images/orchid_placeholder.svg')
    except Exception as e:
        logger.error(f"Error loading Drive photo {file_id}: {e}")
        return redirect('/static/images/orchid_placeholder.svg')

@app.route('/admin/import-sheets-data', methods=['POST'])
def admin_import_sheets_data():
    """Import data from Five Cities Orchid Society Google Sheets"""
    try:
        from google_sheets_importer import GoogleSheetsImporter
        
        sheet_id = '1103vQ_D00Qio5W7PllFeRaFoFAzr7jd8ivOo79sdfgs'
        importer = GoogleSheetsImporter(sheet_id)
        
        # Import limited records for testing
        result = importer.import_all_records(max_records=100)
        
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"Error importing sheets data: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/check-member-submissions', methods=['GET', 'POST'])
def admin_check_member_submissions():
    """Check for new member photo submissions from Google Form"""
    try:
        from member_photo_submission_monitor import check_member_photo_submissions
        
        result = check_member_photo_submissions()
        
        return jsonify({
            'success': True,
            'new_submissions': result['new_submissions'],
            'imported_count': result['imported_count'],
            'message': f"Found {result['new_submissions']} new submissions, imported {result['imported_count']} photos"
        })
        
    except Exception as e:
        logger.error(f"Error checking member submissions: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/photo-submission-notifications')
def admin_photo_submission_notifications():
    """Get recent photo submission notifications"""
    try:
        from member_photo_submission_monitor import photo_monitor
        
        notifications = photo_monitor.get_recent_notifications()
        
        return jsonify({
            'success': True,
            'notifications': notifications
        })
        
    except Exception as e:
        logger.error(f"Error getting notifications: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

# Register AI widget builder blueprint  
try:
    from ai_widget_builder import ai_widget_bp
    app.register_blueprint(ai_widget_bp)
    logger.info("AI Widget Builder registered successfully")
except ImportError as e:
    logger.warning(f"AI Widget Builder not available: {e}")

# Register Orchid Debugger System
try:
    from orchid_debugger_system import debugger_bp
    app.register_blueprint(debugger_bp)
    logger.info("Orchid Debugger System registered successfully")
except ImportError as e:
    logger.warning(f"Orchid Debugger System not available: {e}")

# Register Photo Provenance System
try:
    from photo_provenance_system import provenance_bp
    app.register_blueprint(provenance_bp)
    logger.info("Photo Provenance System registered successfully")
except ImportError as e:
    logger.warning(f"Photo Provenance System not available: {e}")

# Register main widget system blueprint
try:
    from widget_system import widget_bp
    app.register_blueprint(widget_bp)
    logger.info("Widget System registered successfully")
except ImportError as e:
    logger.warning(f"Widget System not available: {e}")

# Register member feedback system
try:
    from member_feedback_system import register_feedback_system
    register_feedback_system()
    logger.info("Member Feedback System registered successfully")
except ImportError as e:
    logger.warning(f"Member Feedback System not available: {e}")

# Register admin monitoring dashboard
try:
    from admin_monitoring_dashboard import register_monitoring_system
    register_monitoring_system()
    logger.info("Admin Monitoring Dashboard registered successfully")
except ImportError as e:
    logger.warning(f"Admin Monitoring Dashboard not available: {e}")

# Register member authentication system
try:
    from member_authentication import register_member_authentication
    register_member_authentication()
    logger.info("Member Authentication System registered successfully")
except ImportError as e:
    logger.warning(f"Member Authentication System not available: {e}")

# Import admin_required decorator
from admin_system import admin_required

# Neon One CRM Integration Routes
@app.route('/api/neon-member-lookup/<email>')
def neon_member_lookup(email):
    """API endpoint to lookup member in Neon One CRM"""
    try:
        from neon_one_integration import fcos_automation
        member = fcos_automation.neon.get_member_by_email(email)
        if member:
            return jsonify({
                'found': True,
                'member': {
                    'account_id': member.account_id,
                    'name': f"{member.first_name} {member.last_name}",
                    'membership_status': member.membership_status,
                    'membership_level': member.membership_level
                }
            })
        return jsonify({'found': False})
    except Exception as e:
        logger.error(f"Neon One member lookup error: {e}")
        return jsonify({
            'found': False,
            'error': 'Member lookup service unavailable'
        }), 500

@app.route('/admin/neon-one-dashboard')
@admin_required
def admin_neon_one_dashboard():
    """Neon One CRM integration dashboard"""
    return render_template('admin/neon_one_dashboard.html')

@app.route('/admin/neon-workshop-reminders/<workshop_date>')
@admin_required
def admin_send_workshop_reminders(workshop_date):
    """Admin endpoint to send workshop reminders via Neon One"""
    try:
        from neon_one_integration import fcos_automation
        results = fcos_automation.send_workshop_reminders(workshop_date)
        return jsonify({
            'success': True,
            'reminder_results': results
        })
    except Exception as e:
        logger.error(f"Workshop reminders error: {e}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500

@app.route('/admin/neon-engagement-report')
@admin_required
def admin_engagement_report():
    """Admin endpoint for member engagement report"""
    try:
        from neon_one_integration import fcos_automation
        report = fcos_automation.generate_member_engagement_report()
        return jsonify(report)
    except Exception as e:
        logger.error(f"Engagement report error: {e}")
        return jsonify({
            'error': 'Report generation failed'
        }), 500

# Newsletter and Zoom Speaker Automation Routes
@app.route('/admin/newsletter-automation')
@admin_required
def admin_newsletter_automation():
    """Newsletter automation dashboard"""
    return render_template('admin/newsletter_automation.html')

@app.route('/api/generate-newsletter-content')
@admin_required
def generate_newsletter_content():
    """Generate automated newsletter content"""
    try:
        from enhanced_newsletter_automation import newsletter_automation
        content = newsletter_automation.generate_monthly_newsletter_content()
        
        return jsonify({
            'success': True,
            'content': {
                'featured_photos': content.featured_photos,
                'database_insights': content.database_insights,
                'member_spotlights': content.member_spotlights,
                'orchid_of_month': content.orchid_of_month,
                'upcoming_events': content.upcoming_events,
                'growth_statistics': content.growth_statistics
            }
        })
    except Exception as e:
        logger.error(f"Newsletter generation error: {e}")
        return jsonify({
            'success': False,
            'error': 'Newsletter generation failed'
        }), 500

@app.route('/admin/zoom-speaker-setup')
@admin_required
def admin_zoom_speaker_setup():
    """Zoom speaker event setup and management"""
    return render_template('admin/zoom_speaker_setup.html')

@app.route('/api/schedule-zoom-speaker', methods=['POST'])
@admin_required
def schedule_zoom_speaker():
    """Schedule new Zoom speaker and trigger marketing"""
    try:
        data = request.get_json()
        from enhanced_newsletter_automation import zoom_automation
        
        speaker = zoom_automation.schedule_monthly_speaker(data)
        marketing_results = zoom_automation.trigger_speaker_marketing_campaign(speaker)
        
        return jsonify({
            'success': True,
            'speaker': {
                'name': speaker.speaker_name,
                'topic': speaker.topic,
                'date': speaker.date,
                'time': speaker.time,
                'registration_link': speaker.registration_link
            },
            'marketing_results': marketing_results
        })
    except Exception as e:
        logger.error(f"Zoom speaker setup error: {e}")
        return jsonify({
            'success': False,
            'error': 'Speaker setup failed'
        }), 500

@app.route('/events/register-speaker/<speaker_id>')
def zoom_speaker_registration(speaker_id):
    """Public Zoom speaker registration page"""
    return render_template('events/zoom_speaker_registration.html', speaker_id=speaker_id)

@app.route('/admin/society-outreach/<speaker_id>')
@admin_required
def admin_society_outreach(speaker_id):
    """Manage outreach to partner orchid societies"""
    try:
        from enhanced_newsletter_automation import zoom_automation
        
        # Get dummy speaker data for the interface (would be from database in production)
        speaker_data = {
            'speaker_name': 'Dr. Jane Smith',
            'topic': 'Advanced Orchid Propagation',
            'date': 'October 15, 2025',
            'time': '7:00 PM PST'
        }
        
        speaker = zoom_automation.schedule_monthly_speaker(speaker_data)
        outreach_campaigns = zoom_automation._create_society_outreach_campaign(speaker)
        
        return jsonify({
            'success': True,
            'outreach_campaigns': outreach_campaigns,
            'total_societies': len(outreach_campaigns),
            'estimated_reach': sum(campaign['estimated_reach'] for campaign in outreach_campaigns)
        })
    except Exception as e:
        logger.error(f"Society outreach error: {e}")
        return jsonify({
            'success': False,
            'error': 'Outreach generation failed'
        }), 500

@app.route('/admin/social-media-content/<speaker_id>')
@admin_required
def admin_social_media_content(speaker_id):
    """Generate social media content for speaker events"""
    try:
        from enhanced_newsletter_automation import zoom_automation
        
        # Get dummy speaker data (would be from database in production)
        speaker_data = {
            'speaker_name': 'Dr. Jane Smith',
            'topic': 'Advanced Orchid Propagation',
            'date': 'October 15, 2025',
            'time': '7:00 PM PST',
            'bio': 'Leading orchid researcher with 20+ years experience'
        }
        
        speaker = zoom_automation.schedule_monthly_speaker(speaker_data)
        social_content = zoom_automation._generate_comprehensive_social_media_content(speaker)
        
        return jsonify({
            'success': True,
            'social_content': social_content,
            'platforms': list(social_content.keys())
        })
    except Exception as e:
        logger.error(f"Social media content error: {e}")
        return jsonify({
            'success': False,
            'error': 'Content generation failed'
        }), 500

# Member Submission Email Campaigns
@app.route('/admin/member-submissions')
@admin_required
def admin_member_submissions():
    """Member submission management dashboard"""
    return render_template('admin/member_submissions.html')

@app.route('/api/trigger-submission-request', methods=['POST'])
@admin_required
def trigger_submission_request():
    """Trigger member submission request email campaign"""
    try:
        data = request.get_json() or {}
        test_mode = data.get('test_mode', False)
        test_email = data.get('test_email', 'jeffery@fivecitiesorchidsociety.org')
        
        from enhanced_newsletter_automation import submission_manager
        
        if test_mode:
            # Send test email to specific address
            results = submission_manager.send_test_submission_email(test_email)
        else:
            # Send to all FCOS members
            results = submission_manager.trigger_submission_request_campaign()
        
        return jsonify({
            'success': True,
            'campaign_results': results,
            'test_mode': test_mode
        })
    except Exception as e:
        logger.error(f"Submission request campaign error: {e}")
        return jsonify({
            'success': False,
            'error': 'Campaign trigger failed'
        }), 500

@app.route('/api/send-submission-reminders', methods=['POST'])
@admin_required
def send_submission_reminders():
    """Send submission deadline reminders"""
    try:
        from enhanced_newsletter_automation import submission_manager
        results = submission_manager.send_submission_reminders()
        
        return jsonify({
            'success': True,
            'reminder_results': results
        })
    except Exception as e:
        logger.error(f"Submission reminder error: {e}")
        return jsonify({
            'success': False,
            'error': 'Reminder sending failed'
        }), 500

@app.route('/api/members-garden-stats')
@admin_required
def members_garden_stats():
    """Get Members Garden submission statistics"""
    try:
        from enhanced_newsletter_automation import submission_manager
        stats = submission_manager.generate_members_garden_stats()
        
        return jsonify({
            'success': True,
            'stats': stats
        })
    except Exception as e:
        logger.error(f"Members Garden stats error: {e}")
        return jsonify({
            'success': False,
            'error': 'Stats generation failed'
        }), 500

# Register visitor teasers
try:
    from visitor_teasers import register_visitor_teasers, add_membership_filters
    from visitor_demo_system import visitor_demo_bp
    from game_infrastructure import game_infrastructure_bp
    from orchid_memory_game import orchid_memory_bp
    from rebus_puzzle_system import rebus_puzzle_bp
    register_visitor_teasers()
    add_membership_filters(app)
    app.register_blueprint(visitor_demo_bp)
    app.register_blueprint(game_infrastructure_bp)
    app.register_blueprint(orchid_memory_bp)
    app.register_blueprint(rebus_puzzle_bp)
    
    from member_personalization import member_personalization_bp
    from orchid_care_manager import orchid_care_manager_bp
    app.register_blueprint(member_personalization_bp)
    app.register_blueprint(orchid_care_manager_bp)
    
    logger.info("Visitor Teasers System registered successfully")
    logger.info("Visitor Demo System registered successfully")
    logger.info("Game Infrastructure registered successfully")
    logger.info("Memory Game System registered successfully") 
    logger.info("Rebus Puzzle System registered successfully")
# Professor BloomBot Curator removed - replaced by Replit Agent
    logger.info("Member Personalization System registered successfully")
    logger.info("Orchid Care Manager registered successfully")
except ImportError as e:
    logger.warning(f"Visitor Teasers System not available: {e}")

# API ENDPOINTS FOR DISCOVERY CENTER SEARCH
@app.route('/api/search-orchids')
def api_search_orchids():
    """API endpoint for searching orchids in discovery center"""
    import sqlite3
    import os
    
    query = request.args.get('q', '').strip()
    genus = request.args.get('genus', '').strip()
    climate = request.args.get('climate', '').strip()
    limit = int(request.args.get('limit', 12))
    
    try:
        # Connect directly to the database to bypass ORM issues
        db_path = os.environ.get('DATABASE_URL', 'orchid_continuum.db')
        if db_path.startswith('postgresql://'):
            # For PostgreSQL, use the configured connection
            from models import OrchidRecord
            orchid_query = db.session.query(OrchidRecord)
            
            # Apply filters
            if query:
                search_filter = db.or_(
                    OrchidRecord.display_name.ilike(f'%{query}%'),
                    OrchidRecord.scientific_name.ilike(f'%{query}%'),
                    OrchidRecord.genus.ilike(f'%{query}%'),
                    OrchidRecord.species.ilike(f'%{query}%')
                )
                orchid_query = orchid_query.filter(search_filter)
            
            if genus:
                orchid_query = orchid_query.filter(OrchidRecord.genus.ilike(f'%{genus}%'))
            
            if climate:
                orchid_query = orchid_query.filter(OrchidRecord.climate_preference.ilike(f'%{climate}%'))
            
            # Order by relevance and limit results
            orchids = orchid_query.order_by(OrchidRecord.created_at.desc()).limit(limit).all()
            
            # Convert to JSON format
            results = []
            for orchid in orchids:
                result = {
                    'id': orchid.id,
                    'name': orchid.display_name,
                    'scientific_name': orchid.scientific_name,
                    'genus': orchid.genus,
                    'species': orchid.species,
                    'habitat': orchid.native_habitat,
                    'climate': orchid.climate_preference,
                    'image_url': orchid.image_url if orchid.image_url and orchid.image_url != '/static/images/orchid_placeholder.svg' else None,
                    'google_drive_id': orchid.google_drive_id
                }
                results.append(result)
        else:
            # For SQLite, use direct connection
            conn = sqlite3.connect('orchid_continuum.db')
            conn.row_factory = sqlite3.Row
            cursor = conn.cursor()
            
            # Build SQL query with filters
            sql = """
                SELECT id, display_name, scientific_name, genus, species, 
                       native_habitat, climate_preference, image_url, google_drive_id
                FROM orchid_record 
                WHERE 1=1
            """
            params = []
            
            if query:
                sql += " AND (display_name LIKE ? OR scientific_name LIKE ? OR genus LIKE ? OR species LIKE ?)"
                search_term = f'%{query}%'
                params.extend([search_term, search_term, search_term, search_term])
            
            if genus:
                sql += " AND genus LIKE ?"
                params.append(f'%{genus}%')
            
            if climate:
                sql += " AND climate_preference LIKE ?"
                params.append(f'%{climate}%')
            
            sql += " ORDER BY created_at DESC LIMIT ?"
            params.append(limit)
            
            cursor.execute(sql, params)
            rows = cursor.fetchall()
            
            results = []
            for row in rows:
                result = {
                    'id': row['id'],
                    'name': row['display_name'],
                    'scientific_name': row['scientific_name'],
                    'genus': row['genus'],
                    'species': row['species'],
                    'habitat': row['native_habitat'],
                    'climate': row['climate_preference'],
                    'image_url': row['image_url'] if row['image_url'] and row['image_url'] != '/static/images/orchid_placeholder.svg' else None,
                    'google_drive_id': row['google_drive_id']
                }
                results.append(result)
            
            conn.close()
        
        return jsonify({
            'orchids': results,
            'total': len(results),
            'query': query,
            'filters': {'genus': genus, 'climate': climate}
        })
        
    except Exception as e:
        logger.error(f"Search API error: {e}")
        # Return filtered version of the simple orchids as fallback
        from simple_api import get_simple_orchids
        simple_orchids = get_simple_orchids()
        
        # Apply basic filtering to the fallback data
        filtered_orchids = []
        for orchid in simple_orchids:
            matches = True
            if query:
                query_lower = query.lower()
                matches = (query_lower in orchid.get('display_name', '').lower() or 
                          query_lower in orchid.get('scientific_name', '').lower() or
                          query_lower in orchid.get('genus', '').lower())
            
            if genus and matches:
                matches = genus.lower() in orchid.get('genus', '').lower()
                
            if matches:
                # Convert to expected format
                result = {
                    'id': orchid['id'],
                    'name': orchid['display_name'],
                    'scientific_name': orchid['scientific_name'],
                    'genus': orchid.get('genus', ''),
                    'species': orchid.get('species', ''),
                    'habitat': orchid.get('native_habitat', ''),
                    'climate': orchid.get('climate_preference', ''),
                    'image_url': None,
                    'google_drive_id': orchid['google_drive_id']
                }
                filtered_orchids.append(result)
        
        return jsonify({
            'orchids': filtered_orchids[:limit],
            'total': len(filtered_orchids),
            'query': query,
            'filters': {'genus': genus, 'climate': climate},
            'fallback': True
        })

# PRODUCTION-READY INDIVIDUAL WIDGET ROUTES FOR NEON ONE INTEGRATION
@app.route('/widget/featured')
def standalone_featured_widget():
    """Standalone Featured Orchid Widget for embedding"""
    from widget_system import widget_system
    widget_data = widget_system.get_widget_data('featured')
    return render_template('widgets/featured_widget.html', 
                         widget_data=widget_data, 
                         standalone=True)

@app.route('/widget/gallery')
def standalone_gallery_widget():
    """Standalone Gallery Widget for embedding"""  
    from widget_system import widget_system
    limit = request.args.get('limit', 6, type=int)
    genus = request.args.get('genus', None)
    widget_data = widget_system.get_widget_data('gallery', limit=limit, genus=genus)
    return render_template('widgets/gallery_widget.html', 
                         widget_data=widget_data, 
                         standalone=True)

@app.route('/widgets/gallery')
def plural_gallery_widget():
    """Gallery Widget - plural route for consistency"""
    try:
        # Simple version without widget_system dependency
        limit = request.args.get('limit', 6, type=int)
        genus = request.args.get('genus', None)
        
        # Get orchids directly from database
        query = OrchidRecord.query.filter(
            OrchidRecord.google_drive_id.isnot(None),
            OrchidRecord.validation_status != 'rejected'
        )
        
        if genus:
            query = query.filter(OrchidRecord.genus.ilike(f'%{genus}%'))
        
        orchids = query.limit(limit).all()
        
        # Convert to template format
        orchids_data = []
        for orchid in orchids:
            orchids_data.append({
                'id': orchid.id,
                'display_name': orchid.display_name or f"{orchid.genus} {orchid.species}",
                'scientific_name': orchid.scientific_name,
                'genus': orchid.genus,
                'species': orchid.species,
                'ai_description': orchid.ai_description,
                'image_url': f"/api/drive-photo/{orchid.google_drive_id}" if orchid.google_drive_id else None
            })
        
        widget_data = {
            'orchids': orchids_data,
            'total_count': len(orchids_data)
        }
        
        return render_template('widgets/gallery_widget.html', 
                             widget_data=widget_data, 
                             standalone=True)
                             
    except Exception as e:
        logger.error(f"Gallery widget error: {e}")
        # Emergency fallback
        return render_template('widgets/gallery_widget.html', 
                             widget_data={'orchids': [], 'total_count': 0}, 
                             standalone=True)

@app.route('/widget/discovery')  
def standalone_discovery_widget():
    """Standalone Discovery Widget for embedding"""
    from widget_system import widget_system
    from models import OrchidRecord
    
    # Get actual database statistics
    total_orchids = db.session.query(OrchidRecord).count()
    recent_orchids = db.session.query(OrchidRecord).order_by(OrchidRecord.created_at.desc()).limit(8).all()
    
    # Convert to dictionary format for template
    orchids_data = []
    for orchid in recent_orchids:
        orchid_data = {
            'id': orchid.id,
            'display_name': orchid.display_name,
            'scientific_name': orchid.scientific_name,
            'genus': orchid.genus,
            'species': orchid.species,
            'native_habitat': orchid.native_habitat,
            'climate_preference': orchid.climate_preference,
            'image_url': orchid.image_url,
            'google_drive_id': orchid.google_drive_id
        }
        orchids_data.append(orchid_data)
    
    widget_data = {
        'orchids': orchids_data,
        'total_orchids': total_orchids
    }
    
    return render_template('widgets/discovery_widget.html', 
                         widget_data=widget_data, 
                         standalone=True)

@app.route('/widget/orchid-of-the-day')
def standalone_orchid_of_day_widget():
    """Standalone Orchid of the Day Widget for embedding"""
    try:
        # Enhanced orchid of day is disabled, using fallback
        from utils import get_orchid_of_day
        orchid_data = get_orchid_of_day()
        widget_data = {'orchid': orchid_data} if orchid_data else {}
    except ImportError:
        # Fallback to regular featured orchid
        from widget_system import widget_system
        widget_data = widget_system.get_widget_data('featured')
    
    return render_template('widgets/featured_widget.html', 
                         widget_data=widget_data, 
                         standalone=True,
                         title="Orchid of the Day")

@app.route('/widget/climate')
def standalone_climate_widget():
    """Standalone Climate Habitat Comparator Widget for embedding"""
    from widget_system import widget_system
    orchid_id = request.args.get('orchid_id', None)
    mode = request.args.get('mode', 'seasonal')
    user_lat = request.args.get('user_lat', None)
    user_lon = request.args.get('user_lon', None)
    
    widget_data = widget_system.get_widget_data('climate', 
                                              orchid_id=orchid_id,
                                              mode=mode,
                                              user_lat=user_lat,
                                              user_lon=user_lon)
    
    return render_template('widgets/climate_widget.html', 
                         widget_data=widget_data, 
                         standalone=True)

# ============================================================================
# PHILOSOPHY QUIZ SYSTEM - Enhanced 16-philosophy quiz with Google Sheets
# ============================================================================

@app.route('/enhanced-philosophy-quiz')
def enhanced_philosophy_quiz():
    """Enhanced Philosophy Quiz - Discover Your Orchid Growing Philosophy"""
    return render_template('widgets/enhanced_philosophy_quiz.html')

# =============================================================================
# DUAL QUIZ ACCESS FLOWS - NeonOne Widget vs Social Media Lead Generation
# =============================================================================

@app.route('/quiz/philosophy/widget')
def philosophy_quiz_widget():
    """NeonOne Widget Route - Direct quiz access for existing members
    
    Features:
    - Direct quiz access without email collection
    - Inline results display
    - NO newsletter signup or CRM integration
    - Optimized for iframe embedding in NeonOne
    """
    try:
        from philosophy_quiz_system import PhilosophyQuizEngine
        
        # Initialize quiz engine and get questions
        quiz_engine = PhilosophyQuizEngine()
        questions = quiz_engine.questions
        
        # Check for existing philosophy badge for logged in users
        existing_philosophy = None
        if session.get('user_id'):
            try:
                from models import PhilosophyBadge
                existing_philosophy = PhilosophyBadge.query.filter_by(
                    user_id=session['user_id']
                ).first()
            except:
                pass
        
        return render_template('widgets/philosophy_quiz_widget.html', 
                             questions=questions,
                             existing_philosophy=existing_philosophy,
                             widget_mode=True)
                             
    except Exception as e:
        logger.error(f"Philosophy quiz widget error: {e}")
        return render_template('widgets/philosophy_quiz_widget.html', 
                             questions=[], 
                             error="Quiz temporarily unavailable")

@app.route('/quiz/philosophy/lead')
def philosophy_quiz_lead():
    """Social Media Landing Route - Lead generation flow
    
    Features:
    - Landing page collects name/email first
    - Shows quiz after email collection
    - Emails results and subscribes to newsletter
    - Full CRM integration for lead capture
    """
    return render_template('widgets/philosophy_quiz_lead_landing.html')

@app.route('/quiz/philosophy/lead/with-email', methods=['POST'])
def philosophy_quiz_lead_with_email():
    """Handle email collection and show quiz to lead"""
    try:
        # Get email collection data
        user_email = request.form.get('email')
        user_name = request.form.get('name', 'Orchid Grower')
        newsletter_opt_in = request.form.get('newsletter') == 'on'
        
        if not user_email or not user_name:
            flash('Both name and email are required')
            return redirect(url_for('philosophy_quiz_lead'))
            
        # Store in session for quiz submission
        session['lead_email'] = user_email
        session['lead_name'] = user_name
        session['lead_newsletter'] = newsletter_opt_in
        
        # Get quiz questions
        from philosophy_quiz_system import PhilosophyQuizEngine
        quiz_engine = PhilosophyQuizEngine()
        questions = quiz_engine.questions
        
        # Render quiz with lead context
        return render_template('widgets/philosophy_quiz.html', 
                             questions=questions,
                             lead_mode=True,
                             user_name=user_name,
                             user_email=user_email)
                             
    except Exception as e:
        logger.error(f"Lead email collection failed: {e}")
        flash('An error occurred. Please try again.')
        return redirect(url_for('philosophy_quiz_lead'))

@app.route('/quiz/philosophy/widget/submit', methods=['POST'])
def submit_philosophy_quiz_widget():
    """Process NeonOne widget quiz submission - NO email collection or CRM"""
    try:
        # Handle form data - answers only, no email collection
        answers = request.form.to_dict()
        
        # Remove any email/name fields that might have been submitted
        answers = {k: v for k, v in answers.items() if k not in ['email', 'name']}
        
        # Import quiz system
        from philosophy_quiz_system import PhilosophyQuizEngine
        from authentic_philosophy_data import get_philosophy_data
        
        # Initialize quiz engine
        quiz_engine = PhilosophyQuizEngine()
        
        # Calculate result
        philosophy_result = quiz_engine.calculate_philosophy_result(answers)
        philosophy_data = get_philosophy_data(philosophy_result)
        
        # Create result object for widget display
        result = {
            'philosophy_data': philosophy_data,
            'philosophy': philosophy_result,
            'widget_mode': True,
            'inline_display': True
        }
        
        # Save badge for logged in users
        if session.get('user_id'):
            try:
                from models import PhilosophyBadge
                existing_badge = PhilosophyBadge.query.filter_by(
                    user_id=session['user_id']
                ).first()
                
                if not existing_badge:
                    new_badge = PhilosophyBadge(
                        user_id=session['user_id'],
                        philosophy_type=philosophy_result,
                        badge_name=philosophy_data.get('badge_name', philosophy_result),
                        badge_emoji=philosophy_data.get('badge_emoji', '🌺'),
                        description=philosophy_data.get('life_philosophy', ''),
                        earned_date=datetime.now().isoformat()
                    )
                    db.session.add(new_badge)
                    db.session.commit()
                    result['badge_earned'] = True
            except Exception as e:
                logger.error(f"Failed to save philosophy badge: {e}")
        
        return render_template('widgets/philosophy_quiz_widget_result.html', result=result)
        
    except Exception as e:
        logger.error(f"Widget quiz submission failed: {e}")
        return render_template('widgets/philosophy_quiz_widget_result.html', 
                             result={'error': 'Quiz processing failed'})

@app.route('/quiz/philosophy/lead/submit', methods=['POST'])
def submit_philosophy_quiz_lead():
    """Process social media lead quiz submission - FULL email and CRM integration"""
    try:
        # Handle form data including email collection
        answers = request.form.to_dict()
        user_email = request.form.get('email')
        user_name = request.form.get('name', 'Orchid Grower')
        
        if not user_email:
            flash('Email is required for this quiz version')
            return redirect(url_for('philosophy_quiz_lead'))
        
        # Remove email/name from answers dict
        quiz_answers = {k: v for k, v in answers.items() if k not in ['email', 'name']}
        
        # Import systems
        from philosophy_quiz_system import PhilosophyQuizEngine
        from authentic_philosophy_data import get_philosophy_data
        from sendgrid_email_automation import PhilosophyQuizEmailer
        from fcos_integrations import process_quiz_lead
        
        # Initialize quiz engine
        quiz_engine = PhilosophyQuizEngine()
        
        # Calculate result
        philosophy_result = quiz_engine.calculate_philosophy_result(quiz_answers)
        philosophy_data = get_philosophy_data(philosophy_result)
        
        # Create result object
        result = {
            'philosophy_data': philosophy_data,
            'philosophy': philosophy_result,
            'user_name': user_name,
            'user_email': user_email,
            'lead_mode': True
        }
        
        # Send email with results
        try:
            emailer = PhilosophyQuizEmailer()
            emailer.send_philosophy_result_email(
                user_email=user_email,
                user_name=user_name,
                philosophy_result=philosophy_result
            )
            result['email_sent'] = True
        except Exception as e:
            logger.error(f"Failed to send philosophy result email: {e}")
            result['email_sent'] = False
        
        # LEAD GENERATION: Auto-subscribe to newsletter & add to CRM
        try:
            lead_results = process_quiz_lead(
                email=user_email,
                name=user_name,
                philosophy_result=philosophy_result,
                quiz_answers=quiz_answers
            )
            result['lead_processed'] = lead_results['lead_processed']
            result['newsletter_subscribed'] = lead_results['newsletter_subscribed']
            result['crm_added'] = lead_results['crm_added']
        except Exception as e:
            logger.error(f"Lead processing failed: {e}")
            result['lead_processed'] = False
        
        return render_template('widgets/philosophy_quiz_lead_result.html', result=result)
        
    except Exception as e:
        logger.error(f"Lead quiz submission failed: {e}")
        return render_template('widgets/philosophy_quiz_lead_result.html', 
                             result={'error': 'Quiz processing failed'})

@app.route('/widgets/philosophy-quiz')
def widgets_philosophy_quiz():
    """Philosophy Quiz Widget Route - Discover Your Orchid Growing Philosophy"""
    from philosophy_quiz_system import PhilosophyQuizEngine
    
    # Initialize quiz engine and get questions
    quiz_engine = PhilosophyQuizEngine()
    questions = quiz_engine.questions  # Use the questions attribute directly
    
    return render_template('widgets/philosophy_quiz.html', questions=questions)

@app.route('/widgets/philosophy-quiz-submit', methods=['POST'])
def submit_widgets_philosophy_quiz():
    """Process regular philosophy quiz submission"""
    try:
        # Handle form data from regular quiz
        answers = request.form.to_dict()
        user_email = request.form.get('email')
        user_name = request.form.get('name', 'Orchid Grower')
        
        # Import authentic data and quiz system
        from authentic_philosophy_data import get_philosophy_data
        from philosophy_quiz_system import PhilosophyQuizEngine
        from sendgrid_email_automation import PhilosophyQuizEmailer
        
        # Initialize quiz engine
        quiz_engine = PhilosophyQuizEngine()
        
        # Calculate result using authentic scoring
        philosophy_result = quiz_engine.calculate_philosophy_result(answers)
        
        # Get authentic philosophy data
        philosophy_data = get_philosophy_data(philosophy_result)
        
        # Create result object
        result = {
            'philosophy_data': philosophy_data,
            'philosophy': philosophy_result,  # Template expects this key
            'score': philosophy_result,
            'user_name': user_name,
            'user_email': user_email
        }
        
        # Send email if email provided
        if user_email:
            try:
                emailer = PhilosophyQuizEmailer()
                emailer.send_philosophy_result_email(
                    user_email=user_email,
                    user_name=user_name,
                    philosophy_result=philosophy_result
                )
                result['email_sent'] = True
            except Exception as e:
                logger.error(f"Failed to send philosophy result email: {e}")
                result['email_sent'] = False
                
            # LEAD GENERATION: Auto-subscribe to newsletter & add to CRM
            try:
                from fcos_integrations import process_quiz_lead
                lead_results = process_quiz_lead(
                    email=user_email,
                    name=user_name,
                    philosophy_result=philosophy_result,
                    quiz_answers=answers
                )
                result['lead_capture'] = lead_results
                logger.info(f"Lead capture completed for {user_email}: {lead_results}")
            except Exception as e:
                logger.error(f"Lead capture failed for {user_email}: {e}")
                result['lead_capture'] = {'lead_processed': False}
        
        # Render result template with proper data
        return render_template('widgets/philosophy_quiz_result.html', result=result)
        
    except Exception as e:
        logger.error(f"Philosophy quiz submission failed: {e}")
        return jsonify({
            'success': False,
            'error': 'Failed to process quiz submission'
        }), 500

@app.route('/api/enhanced-philosophy-quiz-data')
def get_enhanced_philosophy_quiz_data():
    """API endpoint to fetch quiz data from Google Sheets"""
    try:
        quiz_data = philosophy_quiz_service.get_complete_quiz_data()
        return jsonify({
            'success': True,
            'data': quiz_data
        })
    except Exception as e:
        logger.error(f"Failed to fetch philosophy quiz data: {e}")
        return jsonify({
            'success': False,
            'error': 'Failed to load quiz data',
            'fallback': True
        }), 500

@app.route('/api/enhanced-philosophy-quiz-submit', methods=['POST'])
def submit_enhanced_philosophy_quiz():
    """Process quiz submission using authentic data from response file"""
    try:
        # Handle both JSON and form data
        if request.is_json:
            answers = request.json.get('answers', {})
            user_email = request.json.get('email')
            user_name = request.json.get('name', 'Orchid Grower')
        else:
            answers = request.form.to_dict()
            user_email = request.form.get('email')
            user_name = request.form.get('name', 'Orchid Grower')
        
        # Import authentic data
        from authentic_philosophy_data import get_philosophy_data
        from philosophy_quiz_system import PhilosophyQuizEngine
        from sendgrid_email_automation import PhilosophyQuizEmailer
        
        # Initialize quiz engine
        quiz_engine = PhilosophyQuizEngine()
        
        # Calculate result using authentic scoring
        philosophy_result = quiz_engine.calculate_philosophy_result(answers)
        
        # Get authentic philosophy data
        philosophy_data = get_philosophy_data(philosophy_result)
        
        # Get user info from session or form
        user_id = session.get('user_id', 1)  # Default for demo
        
        # Award badge
        badge_awarded = quiz_engine.award_philosophy_badge(user_id, philosophy_result)
        
        # Send email if email provided
        email_sent = False
        if user_email:
            try:
                emailer = PhilosophyQuizEmailer()
                email_sent = emailer.send_philosophy_result_email(
                    user_email, user_name, philosophy_result
                )
                logger.info(f"✅ Philosophy quiz email sent to {user_email}: {philosophy_result}")
            except Exception as e:
                logger.error(f"Email sending failed: {e}")
        
        # Return result data
        result_data = {
            'success': True,
            'philosophy': philosophy_result,
            'philosophy_data': philosophy_data,
            'badge_awarded': badge_awarded,
            'email_sent': email_sent,
            'answers_processed': len(answers),
            'timestamp': datetime.now().isoformat()
        }
        
        logger.info(f"✅ Philosophy quiz completed: {philosophy_result} for user {user_id}")
        
        if request.is_json:
            return jsonify(result_data)
        else:
            # Redirect to results page for form submission
            return render_template('widgets/philosophy_quiz_result.html', 
                                 result=result_data, 
                                 philosophy_data=philosophy_data,
                                 philosophy=philosophy_result)
        
    except Exception as e:
        logger.error(f"Philosophy quiz submission error: {e}")
        error_data = {
            'success': False,
            'error': 'Failed to process quiz results',
            'details': str(e)
        }
        
        if request.is_json:
            return jsonify(error_data), 500
        else:
            return render_template('widgets/philosophy_quiz_result.html', 
                                 error=error_data)

# Register care wheel generator system
try:
    from care_wheel_generator import care_wheel_bp
    app.register_blueprint(care_wheel_bp)
    logger.info("Care Wheel Generator registered successfully")
except ImportError as e:
    logger.warning(f"Care Wheel Generator not available: {e}")

# ============================================================================
# RESEARCH-GRADE CITATION EXPORT API ENDPOINTS
# ============================================================================

@app.route('/api/citations/<format_type>')
def export_citations_api(format_type):
    """API endpoint for exporting citations in various academic formats"""
    try:
        from attribution_system import export_citations, Sources
        
        # Get sources from request parameters
        source_param = request.args.get('sources', '')
        if source_param:
            source_keys = source_param.split(',')
        else:
            # Default to common research sources
            source_keys = [Sources.AOS, Sources.GBIF, Sources.WORLD_ORCHIDS]
        
        # Validate format type
        valid_formats = ['bibtex', 'ris', 'endnote', 'apa', 'mla', 'chicago']
        if format_type not in valid_formats:
            return jsonify({
                'error': f'Invalid format. Supported formats: {", ".join(valid_formats)}'
            }), 400
        
        citations = export_citations(source_keys, format_type)
        
        # Set appropriate content type
        if format_type == 'bibtex':
            content_type = 'application/x-bibtex'
            extension = '.bib'
        elif format_type == 'ris':
            content_type = 'application/x-research-info-systems'
            extension = '.ris'
        elif format_type == 'endnote':
            content_type = 'application/x-endnote-refer'
            extension = '.enw'
        else:
            content_type = 'text/plain'
            extension = '.txt'
        
        # Return as downloadable file
        if request.args.get('download') == 'true':
            response = make_response('\n\n'.join(citations))
            response.headers['Content-Type'] = content_type
            response.headers['Content-Disposition'] = f'attachment; filename="orchid_citations_{format_type}{extension}"'
            return response
        
        # Return as JSON for API usage
        return jsonify({
            'format': format_type,
            'sources': source_keys,
            'citations': citations,
            'count': len(citations),
            'generated_at': datetime.now().isoformat()
        })
        
    except Exception as e:
        logger.error(f"Citation export error: {e}")
        return jsonify({'error': 'Citation export failed'}), 500

@app.route('/api/research-report')
def research_attribution_report():
    """Generate comprehensive research-grade attribution report"""
    try:
        from attribution_system import get_research_report, Sources, AIInferences
        
        # Get parameters
        source_param = request.args.get('sources', '')
        ai_param = request.args.get('ai_inferences', '')
        
        # Parse sources
        if source_param:
            source_keys = source_param.split(',')
        else:
            source_keys = [Sources.AOS, Sources.GBIF, Sources.FCOS]
        
        # Parse AI inferences
        ai_inferences = []
        if ai_param:
            ai_keys = ai_param.split(',')
            ai_mapping = {
                'image_analysis': AIInferences.IMAGE_ANALYSIS,
                'care_recommendations': AIInferences.CARE_RECOMMENDATIONS,
                'metadata_extraction': AIInferences.METADATA_EXTRACTION,
                'similarity_analysis': AIInferences.SIMILARITY_ANALYSIS
            }
            ai_inferences = [ai_mapping.get(key) for key in ai_keys if key in ai_mapping]
        
        report = get_research_report(source_keys, ai_inferences)
        
        return jsonify(report)
        
    except Exception as e:
        logger.error(f"Research report error: {e}")
        return jsonify({'error': 'Research report generation failed'}), 500

@app.route('/api/data-lineage')
def data_lineage_api():
    """API endpoint for data lineage documentation"""
    try:
        from attribution_system import create_data_lineage_trace
        
        # Get parameters
        original_source = request.args.get('source', 'Unknown')
        steps_param = request.args.get('steps', '')
        final_output = request.args.get('output', 'Analysis result')
        
        # Parse processing steps
        if steps_param:
            processing_steps = [step.strip() for step in steps_param.split(',')]
        else:
            processing_steps = ['Data collection', 'Processing', 'Analysis']
        
        lineage_trace = create_data_lineage_trace(original_source, processing_steps, final_output)
        
        return jsonify(lineage_trace)
        
    except Exception as e:
        logger.error(f"Data lineage error: {e}")
        return jsonify({'error': 'Data lineage documentation failed'}), 500

@app.route('/research-attribution-demo')
def research_attribution_demo():
    """Demonstration page for research-grade attribution system"""
    try:
        from attribution_system import (
            attribution_manager, Sources, AIInferences, 
            get_research_report, export_citations
        )
        
        # Example data for demonstration
        demo_sources = [Sources.AOS, Sources.GBIF, Sources.WORLD_ORCHIDS, Sources.OPENAI]
        demo_ai = [AIInferences.IMAGE_ANALYSIS, AIInferences.CARE_RECOMMENDATIONS]
        
        # Generate sample research report
        research_report = get_research_report(demo_sources, demo_ai)
        
        # Generate sample citations
        sample_citations = {
            'bibtex': export_citations(demo_sources[:3], 'bibtex'),
            'apa': export_citations(demo_sources[:3], 'apa'),
            'mla': export_citations(demo_sources[:3], 'mla')
        }
        
        # Create attribution HTML
        attribution_html = attribution_manager.create_attribution_block(demo_sources, demo_ai, 'html')
        
        demo_data = {
            'research_report': research_report,
            'sample_citations': sample_citations,
            'attribution_html': attribution_html,
            'available_sources': list(attribution_manager.data_sources.keys()),
            'source_details': attribution_manager.data_sources
        }
        
        return render_template('research_attribution_demo.html', **demo_data)
        
    except Exception as e:
        logger.error(f"Research attribution demo error: {e}")
        return "Research attribution demonstration unavailable", 500

# Register media migration system
try:
    from media_migration_system import migration_bp
    app.register_blueprint(migration_bp)
    logger.info("Media Migration System registered successfully")
except ImportError as e:
    logger.warning(f"Media Migration System not available: {e}")

# ============================================================================
# FCOS PAGES ROUTES - Structured content from Five Cities Orchid Society
# ============================================================================

@app.route('/fcos/top-10-orchid-tips')
def fcos_top_10_tips():
    """Top 10 orchid growing tips from FCOS experts"""
    return render_template('fcos_pages/top_10_orchid_tips.html')

@app.route('/fcos/guest-information')
def fcos_guest_info():
    """Information for visitors and newcomers to FCOS"""
    return render_template('fcos_pages/guest_information.html')

@app.route('/fcos/orchid-fest-2025')
def fcos_orchid_fest():
    """Complete OrchidFest 2025 event information"""
    return render_template('fcos_pages/orchid_fest_2025.html')

@app.route('/fcos/member-benefits')
def fcos_member_benefits():
    """Benefits of FCOS membership"""
    return render_template('fcos_pages/member_benefits.html')

@app.route('/fcos/how-to-join')
def fcos_how_to_join():
    """How to join Five Cities Orchid Society"""
    return render_template('fcos_pages/how_to_join.html')

@app.route('/orchid-maps')
def orchid_maps():
    """Interactive orchid mapping system with topological data"""
    return render_template('orchid_maps.html')

@app.route('/orchid-map')
def orchid_map():
    """Interactive orchid mapping system (singular route)"""
    return render_template('orchid_map.html')

@app.route('/api/orchid-map-data')
def api_orchid_map_data():
    """API endpoint for orchid mapping data"""
    try:
        from orchid_mapping_system import get_orchid_map_html
        map_html = get_orchid_map_html()
        return jsonify({'map_html': map_html, 'success': True})
    except Exception as e:
        logger.error(f"Error generating orchid map: {e}")
        return jsonify({'error': str(e), 'success': False}), 500

@app.route('/thematic-maps')
def thematic_maps():
    """Thematic orchid distribution maps"""
    return render_template('thematic_maps.html')

@app.route('/fcos/additional-information')
def fcos_additional_info():
    """Additional comprehensive FCOS information"""
    return render_template('fcos_pages/additional_fcos_information.html')

@app.route('/fcos/merchandise')
def fcos_merchandise():
    """FCOS merchandise store and products"""
    return render_template('fcos_pages/fcos_merchandise.html')

# FCOS main redirect for convenience
@app.route('/fcos')
def fcos_main():
    """Main FCOS page redirects to additional information"""
    return redirect(url_for('fcos_additional_info'))

# ============================================================================
# DARWIN CORE EXPORT ROUTES - GBIF Integration
# ============================================================================

@app.route('/export/darwin_core')
def export_darwin_core():
    """Export orchid data as Darwin Core Archive for GBIF publishing"""
    try:
        logger.info("Starting Darwin Core Archive export...")
        
        exporter = DarwinCoreExporter()
        archive_file = exporter.export_to_dwc_archive()
        
        # Generate download filename with timestamp
        download_name = f'orchid_atlas_dwc_{datetime.now().strftime("%Y%m%d_%H%M%S")}.zip'
        
        logger.info(f"Darwin Core Archive ready for download: {download_name}")
        
        return send_file(
            archive_file,
            as_attachment=True,
            download_name=download_name,
            mimetype='application/zip'
        )
        
    except Exception as e:
        logger.error(f"Error creating Darwin Core Archive: {e}")
        flash('Error creating Darwin Core export. Please try again.', 'error')
        return redirect(url_for('admin_dashboard'))

@app.route('/export/darwin_core/preview')
def preview_darwin_core():
    """Preview Darwin Core mapping without downloading"""
    try:
        exporter = DarwinCoreExporter()
        
        # Get sample records for preview
        sample_records = OrchidRecord.query.limit(5).all()
        mapped_samples = []
        
        for record in sample_records:
            mapped_record = exporter.map_orchid_to_dwc(record)
            mapped_samples.append({
                'original': {
                    'id': record.id,
                    'display_name': record.display_name,
                    'scientific_name': record.scientific_name,
                    'genus': record.genus,
                    'species': record.species,
                    'region': record.region,
                    'photographer': record.photographer
                },
                'darwin_core': mapped_record
            })
        
        # Get total counts
        total_records = OrchidRecord.query.count()
        records_with_coords = OrchidRecord.query.filter(
            OrchidRecord.region.like('%Lat:%')
        ).count()
        records_with_images = OrchidRecord.query.filter(
            OrchidRecord.image_url.isnot(None)
        ).count()
        
        export_stats = {
            'total_records': total_records,
            'records_with_coordinates': records_with_coords,
            'records_with_images': records_with_images,
            'coordinate_percentage': round((records_with_coords / total_records * 100) if total_records > 0 else 0, 1),
            'image_percentage': round((records_with_images / total_records * 100) if total_records > 0 else 0, 1)
        }
        
        return render_template('admin/darwin_core_preview.html',
                             mapped_samples=mapped_samples,
                             export_stats=export_stats,
                             dwc_fields=exporter.dwc_fields)
        
    except Exception as e:
        logger.error(f"Error generating Darwin Core preview: {e}")
        flash('Error generating preview. Please try again.', 'error')
        return redirect(url_for('admin_dashboard'))

@app.route("/compare/<species_name>")
def compare_orchid_specimens(species_name):
    """Compare multiple specimens of the same orchid species"""
    from collections import defaultdict
    
    # Get all specimens of this species
    records = OrchidRecord.query.filter(
        func.lower(OrchidRecord.display_name) == species_name.lower()
    ).all()
    
    if not records:
        flash("No specimens found for this species", "error")
        return redirect(url_for("gallery"))
    
    # Group specimens by different characteristics for comparison
    comparison_data = {
        "species_name": species_name,
        "specimen_count": len(records),
        "specimens": records,
        "photographers": list(set(r.photographer for r in records if r.photographer)),
        "sources": list(set(r.ingestion_source for r in records if r.ingestion_source)),
        "with_photos": [r for r in records if r.google_drive_id],
        "climate_conditions": list(set(r.climate_preference for r in records if r.climate_preference)),
        "growth_habits": list(set(r.growth_habit for r in records if r.growth_habit))
    }
    
    return render_template("comparison.html", data=comparison_data)

@app.route("/research_dashboard") 
def research_dashboard():
    """Dashboard for orchid growth research and comparison analysis"""
    # Get species with multiple specimens
    from collections import defaultdict
    
    species_groups = defaultdict(list)
    all_records = OrchidRecord.query.all()
    
    for record in all_records:
        if record.display_name:
            species_key = record.display_name.strip().lower()
            species_groups[species_key].append(record)
    
    # Filter to research candidates (multiple specimens)
    research_candidates = []
    for species, records in species_groups.items():
        if len(records) >= 2:  # Multiple specimens for comparison
            photos = sum(1 for r in records if r.google_drive_id)
            photographers = len(set(r.photographer for r in records if r.photographer))
            
            research_candidates.append({
                "species": species,
                "specimens": len(records),
                "photos": photos,
                "photographers": photographers,
                "photo_percentage": (photos/len(records)*100) if len(records) > 0 else 0
            })
    
    # Sort by research value (specimens with photos)
    research_candidates.sort(key=lambda x: (x["photos"], x["specimens"]), reverse=True)
    
    return render_template("research_dashboard.html", candidates=research_candidates[:50])


# Discovery alerts route removed - replaced by Replit Agent insights
# Professor BloomBot functionality discontinued
# All BloomBot functionality removed

# Featured discovery route removed - replaced by Replit Agent insights
# All BloomBot discovery functionality removed

# FCOS Pages Routes
@app.route('/fcos/merchandise')
def fcos_merchandise_page():
    """FCOS merchandise page"""
    return render_template('fcos_pages/fcos_merchandise.html')

@app.route('/fcos/member-benefits')
def fcos_member_benefits_page():
    """FCOS member benefits page"""
    return render_template('fcos_pages/member_benefits.html')

@app.route('/fcos/orchid-fest-2025')
def fcos_orchid_fest_page():
    """FCOS OrchidFest 2025 page"""
    return render_template('fcos_pages/orchid_fest_2025.html')

@app.route('/fcos/top-10-tips')
def fcos_top_tips_page():
    """FCOS top 10 orchid tips page"""
    return render_template('fcos_pages/top_10_orchid_tips.html')

@app.route('/fcos/how-to-join')
def fcos_how_to_join_page():
    """FCOS how to join page"""
    return render_template('fcos_pages/how_to_join.html')

@app.route('/fcos/guest-information')
def fcos_guest_info_page():
    """FCOS guest information page"""
    return render_template('fcos_pages/guest_information.html')

@app.route('/fcos/additional-information')
def fcos_additional_info_page():
    """FCOS additional information page"""
    return render_template('fcos_pages/additional_fcos_information.html')

# Register AOS-Baker Culture Sheet System
try:
    from aos_baker_culture_routes import register_culture_routes
    register_culture_routes(app)
    logger.info("AOS-Baker Culture Sheet system registered successfully")
except Exception as e:
    logger.error(f"Failed to register culture sheet system: {str(e)}")

# ==============================================================================
# ADMIN SCRAPING DASHBOARD ROUTES - Real-time monitoring and control
# ==============================================================================

@app.route('/admin/scraping/dashboard-stats')
def scraping_dashboard_stats():
    """Get real-time scraping dashboard statistics"""
    try:
        stats = scraping_dashboard.get_dashboard_stats()
        return jsonify(stats)
    except Exception as e:
        logger.error(f"Dashboard stats error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/admin/scraping/start', methods=['POST'])
def start_methodical_scraping():
    """Start methodical one-plant-at-a-time scraping"""
    try:
        success = scraping_dashboard.start_methodical_scraping()
        if success:
            return jsonify({'success': True, 'message': 'Methodical scraping started'})
        else:
            return jsonify({'success': False, 'error': 'Scraping already running'})
    except Exception as e:
        logger.error(f"Start scraping error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/scraping/stop', methods=['POST'])
def stop_scraping():
    """Stop all scraping operations"""
    try:
        scraping_dashboard.stop_scraping()
        return jsonify({'success': True, 'message': 'Scraping stopped'})
    except Exception as e:
        logger.error(f"Stop scraping error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/scraping/manual-trigger', methods=['POST'])
def manual_trigger_scraper():
    """Manually trigger a specific scraper"""
    try:
        data = request.get_json()
        scraper_name = data.get('scraper')
        
        if not scraper_name:
            return jsonify({'success': False, 'error': 'Scraper name required'}), 400
        
        result = scraping_dashboard.manual_trigger_scraper(scraper_name)
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"Manual trigger error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

# ==============================================================================
# VIGILANT MONITORING ROUTES - 30-second checks and auto-recovery
# ==============================================================================

@app.route('/admin/vigilant/start', methods=['POST'])
def start_vigilant_monitoring():
    """Start vigilant 30-second monitoring"""
    try:
        if vigilant_monitor:
            success = vigilant_monitor.start_vigilant_monitoring()
        else:
            success = False
        if success:
            return jsonify({'success': True, 'message': 'Vigilant monitoring started'})
        else:
            return jsonify({'success': False, 'error': 'Monitoring already running'})
    except Exception as e:
        logger.error(f"Start monitoring error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/vigilant/stop', methods=['POST'])
def stop_vigilant_monitoring():
    """Stop vigilant monitoring"""
    try:
        if vigilant_monitor:
            vigilant_monitor.stop_monitoring()
        else:
            raise Exception('Vigilant monitor not available')
        return jsonify({'success': True, 'message': 'Vigilant monitoring stopped'})
    except Exception as e:
        logger.error(f"Stop monitoring error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/vigilant/stats')
def vigilant_monitor_stats():
    """Get vigilant monitoring statistics"""
    try:
        if vigilant_monitor:
            stats = vigilant_monitor.get_monitor_stats()
        else:
            stats = {'error': 'Monitor not available'}
        return jsonify(stats)
    except Exception as e:
        logger.error(f"Monitor stats error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/admin/vigilant/force-backup', methods=['POST'])
def force_database_backup():
    """Force immediate database backup"""
    try:
        if vigilant_monitor:
            result = vigilant_monitor.force_backup()
            backup_url = vigilant_monitor.get_backup_download_url()
        else:
            result = {'error': 'Monitor not available'}
            backup_url = None
        
        return jsonify({
            'success': True, 
            'message': result,
            'backup_url': backup_url
        })
    except Exception as e:
        logger.error(f"Force backup error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/download-backup/<filename>')
def download_backup(filename):
    """Download database backup file"""
    try:
        from pathlib import Path
        backup_path = Path('database_backups') / filename
        
        if backup_path.exists() and backup_path.name.startswith('orchid_db_backup_'):
            return send_file(backup_path, as_attachment=True)
        else:
            return "Backup file not found", 404
            
    except Exception as e:
        logger.error(f"Download backup error: {e}")
        return f"Error downloading backup: {e}", 500

# ==============================================================================
# VIGILANT MONITOR AUTO-START - Start vigilant monitoring on app startup
# ==============================================================================

# ==============================================================================
# REAL GALLERY API - For testing actual user-visible images
# ==============================================================================

@app.route('/api/recent-orchids')
@widget_error_handler.with_error_handling(timeout=30, cache_ttl=300, fallback_data={'orchids': [], 'count': 0})
def api_recent_orchids():
    """API endpoint for recent orchids - FIXED TO USE CORRECT DATABASE DATA"""
    try:
        # Get orchids with Google Drive photos from the database with CORRECT names
        recent_orchids = OrchidRecord.query.filter(
            OrchidRecord.google_drive_id.isnot(None),
            OrchidRecord.google_drive_id != ''
        ).order_by(OrchidRecord.id.asc()).limit(8).all()
        
        result = []
        for orchid in recent_orchids:
            result.append({
                "id": orchid.id,
                "scientific_name": orchid.scientific_name or orchid.display_name,
                "display_name": orchid.display_name,
                "genus": orchid.genus,
                "species": orchid.species,
                "google_drive_id": orchid.google_drive_id,
                "photographer": orchid.photographer or "FCOS Collection",
                "ai_description": orchid.ai_description or f"Beautiful {orchid.genus or 'orchid'} specimen",
                "decimal_latitude": orchid.decimal_latitude,
                "decimal_longitude": orchid.decimal_longitude,
                "image_url": f"/api/drive-photo/{orchid.google_drive_id}"
            })
            
        # SAFEGUARD: Log any potential photo-name mismatches to prevent future issues
        for orchid in result:
            if orchid.get('google_drive_id') in ['185MlwyxBU8Dy6bqGdwXXPeBXTlhg5M0I', '1142ajwZe7_LbGt-BPy-HqVkLpNczcfZY']:
                expected_names = {
                    '185MlwyxBU8Dy6bqGdwXXPeBXTlhg5M0I': 'Cattleya "Pink"',
                    '1142ajwZe7_LbGt-BPy-HqVkLpNczcfZY': 'Epidendrum longipetalum'
                }
                expected = expected_names.get(orchid['google_drive_id'])
                actual = orchid['display_name']
                if expected != actual:
                    logger.error(f"🚨 PHOTO-NAME MISMATCH DETECTED: Photo {orchid['google_drive_id']} shows '{actual}' but should be '{expected}'")
                else:
                    logger.info(f"✅ Photo {orchid['google_drive_id']} correctly shows '{actual}'")
        
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"Error in api_recent_orchids: {e}")
        # Fallback to database records with correct names
        fallback_orchids = [
            {
                "id": 372,
                "scientific_name": "Cattleya \"Pink\"",
                "display_name": "Cattleya \"Pink\"",
                "genus": "Cattleya",
                "species": None,
                "google_drive_id": "185MlwyxBU8Dy6bqGdwXXPeBXTlhg5M0I",
                "photographer": "FCOS Collection",
                "ai_description": "Beautiful pink Cattleya orchid",
                "decimal_latitude": None,
                "decimal_longitude": None,
                "image_url": f"/api/drive-photo/185MlwyxBU8Dy6bqGdwXXPeBXTlhg5M0I"
            },
            {
                "id": 1391,
                "scientific_name": "Epidendrum longipetalum",
                "display_name": "Epidendrum longipetalum",
                "genus": "Epidendrum",
                "species": "longipetalum",
                "google_drive_id": "1142ajwZe7_LbGt-BPy-HqVkLpNczcfZY",
                "photographer": "FCOS Collection",
                "ai_description": "Elegant Epidendrum with long petals",
                "decimal_latitude": None,
                "decimal_longitude": None,
                "image_url": f"/api/drive-photo/1142ajwZe7_LbGt-BPy-HqVkLpNczcfZY"
            },
        {
            "id": 1003,
            "scientific_name": "Trichocentrum longiscott",
            "display_name": "Trichocentrum 'Longiscott'",
            "google_drive_id": "1bUDCfCrZCLeRWgDrDQfLbDbOmXTDQHjH",
            "photographer": "FCOS Collection",
            "ai_description": "Stunning trichocentrum hybrid with spotted patterns",
            "decimal_latitude": 10.0,
            "decimal_longitude": -84.0,
            "image_url": f"/api/drive-photo/1bUDCfCrZCLeRWgDrDQfLbDbOmXTDQHjH"
        },
        {
            "id": 1004,
            "scientific_name": "Angraecum didieri",
            "display_name": "Angraecum didieri",
            "google_drive_id": "1gd9BbXslt1IzAgMpeMWYQUfcJHWtHzhS",
            "photographer": "FCOS Collection",
            "ai_description": "White star-shaped angraecum with distinctive spur",
            "decimal_latitude": -20.0,
            "decimal_longitude": 47.0,
            "image_url": f"/api/drive-photo/1gd9BbXslt1IzAgMpeMWYQUfcJHWtHzhS"
        }
    ]
    
        # Apply any filters requested
        limit = int(request.args.get('limit', 20))
        with_coordinates = request.args.get('with_coordinates', 'false').lower() == 'true'
        
        if with_coordinates:
            # Filter to only orchids with coordinates
            fallback_orchids = [o for o in fallback_orchids if o.get('decimal_latitude') and o.get('decimal_longitude')]
        
        return jsonify(fallback_orchids[:limit])


# ==============================================================================
# EOL INTEGRATION ROUTES - Encyclopedia of Life enhancement
# ==============================================================================

@app.route('/admin/eol-enhancement')
def eol_enhancement_dashboard():
    """Admin dashboard for EOL enhancement status and controls"""
    try:
        eol = EOLIntegrator()
        status = eol.get_enhancement_status()
        
        return render_template('admin/eol_dashboard.html', 
                             status=status,
                             title="EOL Enhancement Dashboard")
    except Exception as e:
        logger.error(f"EOL dashboard error: {e}")
        flash(f"Error loading EOL dashboard: {str(e)}", 'error')
        return redirect(url_for('admin_dashboard'))

@app.route('/admin/eol-enhance-all', methods=['POST'])
def eol_enhance_all_orchids():
    """Start EOL enhancement for all orchid records"""
    try:
        eol = EOLIntegrator()
        stats = eol.enhance_all_orchids()
        
        flash(f"EOL Enhancement completed! Enhanced: {stats['successfully_enhanced']}, "
              f"Not found: {stats['not_found_in_eol']}, Errors: {stats['errors']}", 'success')
        
        return jsonify({
            'success': True,
            'stats': stats
        })
        
    except Exception as e:
        logger.error(f"EOL enhancement error: {e}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500

@app.route('/admin/eol-test')
def test_eol_integration():
    """Test EOL integration with a sample orchid"""
    try:
        eol = EOLIntegrator()
        
        # Test with Cattleya trianae (your first orchid)
        test_result = eol.search_eol_species("Cattleya trianae")
        
        if test_result:
            page_data = eol.get_eol_page_data(test_result['id'])
            traits = eol.extract_trait_data(page_data) if page_data else None
            
            return jsonify({
                'success': True,
                'test_species': 'Cattleya trianae',
                'eol_page_id': test_result.get('id'),
                'traits_found': len(traits.get('descriptions', [])) if traits else 0,
                'images_found': len(traits.get('images', [])) if traits else 0,
                'common_names': traits.get('common_names', []) if traits else []
            })
        else:
            return jsonify({
                'success': False,
                'message': 'Cattleya trianae not found in EOL'
            })
            
    except Exception as e:
        logger.error(f"EOL test error: {e}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500

@app.route('/api/orchid/<int:orchid_id>/eol-enhance', methods=['POST'])
def enhance_single_orchid_with_eol(orchid_id):
    """Enhance a single orchid with EOL data"""
    try:
        orchid = OrchidRecord.query.get_or_404(orchid_id)
        eol = EOLIntegrator()
        
        # Search for the orchid in EOL
        search_result = eol.search_eol_species(orchid.scientific_name)
        
        if search_result:
            # Get detailed page data
            page_data = eol.get_eol_page_data(search_result['id'])
            
            if page_data:
                # Extract traits
                traits = eol.extract_trait_data(page_data)
                
                # Update orchid record
                success = eol.update_orchid_with_eol_data(orchid_id, traits)
                
                if success:
                    return jsonify({
                        'success': True,
                        'message': f'Enhanced {orchid.scientific_name} with EOL data',
                        'eol_page_id': traits.get('eol_page_id'),
                        'traits_added': len(traits.get('descriptions', [])),
                        'images_found': len(traits.get('images', []))
                    })
                else:
                    return jsonify({
                        'success': False,
                        'error': 'Failed to update orchid record'
                    }), 500
            else:
                return jsonify({
                    'success': False,
                    'error': 'Failed to get EOL page data'
                }), 404
        else:
            return jsonify({
                'success': False,
                'error': f'{orchid.scientific_name} not found in EOL'
            }), 404
            
    except Exception as e:
        logger.error(f"Single orchid EOL enhancement error: {e}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500

@app.route('/api/eol-status')
def get_eol_status():
    """Get current EOL enhancement status"""
    try:
        eol = EOLIntegrator()
        status = eol.get_enhancement_status()
        return jsonify(status)
    except Exception as e:
        logger.error(f"EOL status error: {e}")
        return jsonify({'error': str(e)}), 500

# ==============================================================================
# WIDGET INTEGRATION ROUTES - Cross-widget functionality and mobile optimization
# ==============================================================================

@app.route('/api/widget-session')
def get_widget_session():
    """Get unified widget session data"""
    try:
        session_data = widget_hub.get_user_session()
        return jsonify(session_data)
    except Exception as e:
        logger.error(f"Widget session error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/widget-interaction', methods=['POST'])
def track_widget_interaction_api():
    """Track widget interaction for cross-widget integration"""
    try:
        data = request.get_json()
        widget_name = data.get('widget_name')
        action = data.get('action', 'view')
        context_data = data.get('context', {})
        
        track_widget_interaction(widget_name, action, **context_data)
        
        # Get smart recommendations based on interaction
        recommendations = widget_hub.get_smart_recommendations(widget_name)
        
        return jsonify({
            'success': True,
            'recommendations': recommendations
        })
        
    except Exception as e:
        logger.error(f"Widget interaction tracking error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/widget-recommendations/<widget_type>')
def get_widget_recommendations(widget_type):
    """Get smart recommendations for a widget"""
    try:
        recommendations = widget_hub.get_smart_recommendations(widget_type)
        return jsonify(recommendations)
    except Exception as e:
        logger.error(f"Widget recommendations error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/favorites', methods=['GET', 'POST', 'DELETE'])
def manage_favorites():
    """Manage user favorites across all widgets"""
    try:
        if request.method == 'GET':
            return jsonify(widget_hub.manage_favorites('get'))
            
        elif request.method == 'POST':
            data = request.get_json()
            orchid_id = data.get('orchid_id')
            orchid_data = data.get('orchid_data')
            
            result = widget_hub.manage_favorites('add', orchid_id, orchid_data)
            return jsonify(result)
            
        elif request.method == 'DELETE':
            orchid_id = request.args.get('orchid_id', type=int)
            result = widget_hub.manage_favorites('remove', orchid_id)
            return jsonify(result)
            
    except Exception as e:
        logger.error(f"Favorites management error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/exploration-progress')
def get_exploration_progress():
    """Get user's exploration progress and achievements"""
    try:
        progress_data = widget_hub.track_exploration_progress({})
        return jsonify(progress_data)
    except Exception as e:
        logger.error(f"Exploration progress error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/mobile-config/<widget_type>')
def get_mobile_config(widget_type):
    """Get mobile-optimized configuration for widgets"""
    try:
        config = mobile_optimizer.get_mobile_optimized_config(widget_type)
        return jsonify(config)
    except Exception as e:
        logger.error(f"Mobile config error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/mobile-touch-controls.js')
def get_mobile_touch_controls_js():
    """Get JavaScript for mobile touch controls"""
    try:
        widget_type = request.args.get('widget_type', 'general')
        js_code = mobile_optimizer.get_touch_controls_javascript(widget_type)
        
        response = Response(js_code, mimetype='application/javascript')
        response.headers['Cache-Control'] = 'public, max-age=3600'
        return response
        
    except Exception as e:
        logger.error(f"Mobile JS error: {e}")
        return Response(f"console.error('Mobile controls error: {str(e)}');", 
                       mimetype='application/javascript'), 500

@app.route('/api/mobile-widget-styles.css')
def get_mobile_widget_styles():
    """Get CSS for mobile widget optimization"""
    try:
        css_code = mobile_optimizer.get_mobile_css()
        
        response = Response(css_code, mimetype='text/css')
        response.headers['Cache-Control'] = 'public, max-age=3600'
        return response
        
    except Exception as e:
        logger.error(f"Mobile CSS error: {e}")
        return Response(f"/* Mobile styles error: {str(e)} */", 
                       mimetype='text/css'), 500

# ==============================================================================
# USER COLLECTION HUB ROUTES - Personal orchid collection management
# ==============================================================================

@app.route('/my-collection')
def user_collection_dashboard():
    """Personal orchid collection dashboard"""
    try:
        dashboard_data = collection_hub.get_collection_dashboard_data()
        recommendations = collection_hub.get_personalized_recommendations()
        
        return render_template('collection/dashboard.html',
                             dashboard=dashboard_data,
                             recommendations=recommendations,
                             title="My Orchid Collection")
    except Exception as e:
        logger.error(f"Collection dashboard error: {e}")
        flash(f"Error loading collection: {str(e)}", 'error')
        return redirect(url_for('index'))

@app.route('/api/collection', methods=['GET', 'POST'])
def collection_api():
    """API for collection management"""
    try:
        if request.method == 'GET':
            collection_data = collection_hub.get_user_collection()
            return jsonify(collection_data)
            
        elif request.method == 'POST':
            data = request.get_json()
            action = data.get('action')
            
            if action == 'add_orchid':
                orchid_id = data.get('orchid_id')
                collection_type = data.get('collection_type', 'owned')
                care_data = data.get('care_data', {})
                
                result = collection_hub.add_to_collection(orchid_id, collection_type, care_data)
                return jsonify(result)
                
            elif action == 'log_care':
                orchid_id = data.get('orchid_id')
                care_type = data.get('care_type')
                notes = data.get('notes', '')
                
                result = collection_hub.log_care_activity(orchid_id, care_type, notes)
                return jsonify(result)
                
            else:
                return jsonify({'error': 'Unknown action'}), 400
                
    except Exception as e:
        logger.error(f"Collection API error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/care-reminders')
def get_care_reminders():
    """Get care reminders for user's collection"""
    try:
        reminders = collection_hub.get_care_reminders()
        return jsonify({'reminders': reminders})
    except Exception as e:
        logger.error(f"Care reminders error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/collection-recommendations')
def get_collection_recommendations():
    """Get personalized orchid recommendations"""
    try:
        recommendations = collection_hub.get_personalized_recommendations()
        return jsonify(recommendations)
    except Exception as e:
        logger.error(f"Collection recommendations error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/unified-dashboard')
def get_unified_dashboard():
    """Get comprehensive unified dashboard data"""
    try:
        dashboard_data = widget_hub.get_unified_dashboard_data()
        collection_data = collection_hub.get_collection_dashboard_data()
        
        unified_data = {
            'widget_integration': dashboard_data,
            'collection_hub': collection_data,
            'device_type': mobile_optimizer.detect_device_type(),
            'timestamp': datetime.now().isoformat()
        }
        
        return jsonify(unified_data)
        
    except Exception as e:
        logger.error(f"Unified dashboard error: {e}")
        return jsonify({'error': str(e)}), 500

# ==============================================================================
# SHOW & TELL ROUTES - Member gallery creation and admin approval workflow
# ==============================================================================

@app.route('/my-collection/show-tell')
def show_tell_gallery_page():
    """Show & Tell gallery management page"""
    try:
        show_tell_data = collection_hub.get_show_tell_dashboard_data()
        collection_data = collection_hub.get_user_collection()
        
        return render_template('collection/show_tell.html',
                             show_tell=show_tell_data,
                             collection=collection_data,
                             title="Show & Tell Galleries")
    except Exception as e:
        logger.error(f"Show & Tell page error: {e}")
        flash(f"Error loading Show & Tell: {str(e)}", 'error')
        return redirect(url_for('user_collection_dashboard'))

@app.route('/api/show-tell', methods=['GET', 'POST', 'PUT', 'DELETE'])
def show_tell_api():
    """API for Show & Tell gallery management"""
    try:
        if request.method == 'GET':
            status_filter = request.args.get('status')
            galleries = collection_hub.get_show_tell_galleries(status_filter)
            return jsonify({'galleries': galleries})
            
        elif request.method == 'POST':
            data = request.get_json()
            result = collection_hub.create_show_tell_gallery(data)
            return jsonify(result)
            
        elif request.method == 'PUT':
            data = request.get_json()
            gallery_id = data.get('gallery_id')
            updates = data.get('updates', {})
            
            result = collection_hub.update_show_tell_gallery(gallery_id, updates)
            return jsonify(result)
            
        elif request.method == 'DELETE':
            gallery_id = request.args.get('gallery_id')
            result = collection_hub.delete_show_tell_gallery(gallery_id)
            return jsonify(result)
            
    except Exception as e:
        logger.error(f"Show & Tell API error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/show-tell/public')
def get_public_show_tell():
    """Get all approved public Show & Tell galleries"""
    try:
        galleries = collection_hub.get_public_show_tell_galleries()
        return jsonify({'galleries': galleries})
    except Exception as e:
        logger.error(f"Public Show & Tell error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/show-tell/admin/queue')
def get_admin_approval_queue():
    """Get galleries pending admin approval"""
    try:
        queue = collection_hub.get_admin_approval_queue()
        return jsonify({'queue': queue})
    except Exception as e:
        logger.error(f"Admin queue error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/show-tell/admin/approve', methods=['POST'])
def admin_approve_show_tell():
    """Admin approval/rejection of Show & Tell gallery"""
    try:
        data = request.get_json()
        gallery_id = data.get('gallery_id')
        approve = data.get('approve', True)
        admin_notes = data.get('admin_notes', '')
        
        result = collection_hub.admin_approve_gallery(gallery_id, admin_notes, approve)
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"Admin approval error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/show-tell/public')
def public_show_tell_page():
    """Public Show & Tell display page"""
    try:
        galleries = collection_hub.get_public_show_tell_galleries()
        
        return render_template('show_tell/public.html',
                             galleries=galleries,
                             title="Member Show & Tell Galleries")
    except Exception as e:
        logger.error(f"Public Show & Tell page error: {e}")
        flash(f"Error loading galleries: {str(e)}", 'error')
        return redirect(url_for('index'))

@app.route('/show-tell/gallery/<gallery_id>')
def view_show_tell_gallery(gallery_id):
    """View individual Show & Tell gallery"""
    try:
        # Increment view count
        collection_hub.increment_gallery_views(gallery_id)
        
        # Get gallery data
        all_galleries = collection_hub.get_show_tell_galleries()
        gallery = next((g for g in all_galleries if g['id'] == gallery_id), None)
        
        if not gallery:
            flash('Gallery not found', 'error')
            return redirect(url_for('public_show_tell_page'))
        
        # Only show if public or user's own gallery
        if gallery['status'] not in ['public_approved', 'member_published']:
            flash('Gallery not available for viewing', 'error')
            return redirect(url_for('public_show_tell_page'))
        
        return render_template('show_tell/gallery_detail.html',
                             gallery=gallery,
                             title=f"Show & Tell: {gallery['title']}")
    except Exception as e:
        logger.error(f"Gallery view error: {e}")
        flash(f"Error loading gallery: {str(e)}", 'error')
        return redirect(url_for('public_show_tell_page'))

# ==============================================================================
# MONTHLY CONTEST ROUTES - Show & Tell Monthly Contest System
# ==============================================================================

from monthly_contest_system import monthly_contest

@app.route('/contest')
def monthly_contest_page():
    """Monthly Show & Tell Contest main page"""
    try:
        contest_period = monthly_contest.get_current_contest_period()
        contest_stats = monthly_contest.get_contest_stats()
        leaderboard = monthly_contest.get_full_leaderboard()
        user_voting_status = monthly_contest.get_user_voting_status()
        
        return render_template('contest/monthly_contest.html',
                             period=contest_period,
                             stats=contest_stats,
                             leaderboard=leaderboard,
                             categories=monthly_contest.categories,
                             voting_status=user_voting_status,
                             title="Monthly Show & Tell Contest")
    except Exception as e:
        logger.error(f"Monthly contest page error: {e}")
        flash(f"Error loading contest: {str(e)}", 'error')
        return redirect(url_for('index'))

@app.route('/contest/submit')
def contest_submission_page():
    """Contest submission page for members"""
    try:
        contest_period = monthly_contest.get_current_contest_period()
        
        if not contest_period['is_active']:
            flash('Submission deadline has passed for this month', 'warning')
            return redirect(url_for('monthly_contest_page'))
        
        # Get member's current submissions
        member_id = session.get('user_id', 'demo_member')  # In real app, get from auth
        member_submissions = monthly_contest.get_member_submissions(member_id)
        remaining_submissions = monthly_contest.max_submissions_per_member - len(member_submissions)
        
        return render_template('contest/submit_entry.html',
                             period=contest_period,
                             categories=monthly_contest.categories,
                             submissions=member_submissions,
                             remaining=remaining_submissions,
                             title="Submit Contest Entry")
    except Exception as e:
        logger.error(f"Contest submission page error: {e}")
        flash(f"Error loading submission page: {str(e)}", 'error')
        return redirect(url_for('monthly_contest_page'))

@app.route('/api/contest/submit', methods=['POST'])
def submit_contest_entry():
    """API endpoint for submitting contest entries"""
    try:
        data = request.get_json()
        member_id = session.get('user_id', 'demo_member')  # In real app, get from auth
        
        result = monthly_contest.submit_entry(member_id, data)
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"Contest submission error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/contest/vote', methods=['POST'])
def vote_contest_entry():
    """API endpoint for voting on contest entries"""
    try:
        data = request.get_json()
        entry_id = data.get('entry_id')
        voter_ip = request.remote_addr or 'unknown'
        
        result = monthly_contest.vote_for_entry(entry_id, voter_ip)
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"Contest voting error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/contest/entries')
def get_contest_entries():
    """Get contest entries by category"""
    try:
        category = request.args.get('category') or ''
        status = request.args.get('status', 'approved')
        
        entries = monthly_contest.get_contest_entries(category=category, status=status)
        return jsonify({'entries': entries})
        
    except Exception as e:
        logger.error(f"Contest entries error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/contest/leaderboard')
def get_contest_leaderboard():
    """Get contest leaderboard data"""
    try:
        category = request.args.get('category')
        
        if category:
            leaderboard = monthly_contest.get_category_leaderboard(category)
        else:
            leaderboard = monthly_contest.get_full_leaderboard()
            
        return jsonify({'leaderboard': leaderboard})
        
    except Exception as e:
        logger.error(f"Contest leaderboard error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/contest/admin/queue')
def get_contest_admin_queue():
    """Get entries pending admin approval"""
    try:
        queue = monthly_contest.get_admin_queue()
        return jsonify({'queue': queue})
        
    except Exception as e:
        logger.error(f"Contest admin queue error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/contest/admin/moderate', methods=['POST'])
def moderate_contest_entry():
    """Admin moderation of contest entries"""
    try:
        data = request.get_json()
        entry_id = data.get('entry_id')
        action = data.get('action')  # 'approve' or 'reject'
        admin_notes = data.get('admin_notes', '')
        
        result = monthly_contest.admin_moderate_entry(entry_id, action, admin_notes)
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"Contest moderation error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/contest/admin/export')
def export_contest_results():
    """Export contest results for admin"""
    try:
        contest_id = request.args.get('contest_id')
        format_type = request.args.get('format', 'json')
        
        result = monthly_contest.export_contest_results(contest_id, format_type)
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"Contest export error: {e}")
        return jsonify({'error': str(e)}), 500

# Enhanced Contest API Endpoints

@app.route('/api/contest/enhanced-submit', methods=['POST'])
def enhanced_submit_entry():
    """Enhanced submission endpoint with file upload and validation"""
    try:
        import hashlib
        from werkzeug.utils import secure_filename
        
        member_id = session.get('user_id', 'demo_member')
        
        # Check submission window
        contest_period = monthly_contest.get_current_contest_period()
        if not contest_period['is_active']:
            return jsonify({'success': False, 'error': 'Submission deadline has passed'})
        
        # Get form data
        plant_name = request.form.get('plant_name', '').strip()
        category = request.form.get('category', '').strip()
        caption = request.form.get('caption', '').strip()
        culture = request.form.get('culture', '').strip()
        is_draft = request.form.get('is_draft') == 'true'
        
        # Validate required fields (unless draft)
        if not is_draft:
            if not all([plant_name, category, caption]):
                return jsonify({'success': False, 'error': 'Missing required fields'})
        
        # Handle photo upload
        photo_file = request.files.get('photo')
        photo_hash = None
        photo_url = None
        
        if photo_file and photo_file.filename:
            # Validate file type
            allowed_extensions = {'jpg', 'jpeg', 'png'}
            file_ext = photo_file.filename.rsplit('.', 1)[1].lower() if '.' in photo_file.filename else ''
            if file_ext not in allowed_extensions:
                return jsonify({'success': False, 'error': 'Invalid file type. Use JPG or PNG.'})
            
            # Validate file size (10MB)
            if len(photo_file.read()) > 10 * 1024 * 1024:
                return jsonify({'success': False, 'error': 'File too large. Maximum 10MB.'})
            
            photo_file.seek(0)  # Reset file pointer
            
            # Generate photo hash for duplicate detection
            photo_content = photo_file.read()
            photo_hash = hashlib.sha256(photo_content).hexdigest()
            photo_file.seek(0)
            
            # Check for duplicate hash
            existing_entry = monthly_contest.find_by_photo_hash(photo_hash)
            flags = []
            if existing_entry:
                flags.append('possible_duplicate')
            
            # Save file (in production, would upload to cloud storage)
            filename = secure_filename(f"entry_{member_id}_{contest_period['contest_id']}_{photo_hash[:8]}.{file_ext}")
            photo_url = f"/static/uploads/{filename}"
            
            # For demo, use placeholder
            photo_url = "/static/images/placeholder.png"
        
        # Create entry data
        entry_data = {
            'plant_name': plant_name,
            'category': category,
            'caption': caption,
            'culture_notes': culture,
            'photo_url': photo_url,
            'photo_hash': photo_hash,
            'is_draft': is_draft,
            'original_photo_confirmed': request.form.get('original_photo') == 'true',
            'currently_blooming_confirmed': request.form.get('currently_blooming') == 'true'
        }
        
        # Submit entry
        result = monthly_contest.enhanced_submit_entry(member_id, entry_data)
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"Enhanced submission error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/contest/submission-count')
def get_submission_count():
    """Get current member's submission count for the month"""
    try:
        member_id = session.get('user_id', 'demo_member')
        submissions = monthly_contest.get_member_submissions(member_id)
        return jsonify({'count': len(submissions)})
        
    except Exception as e:
        logger.error(f"Submission count error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/contest/admin/entries')
def get_admin_entries():
    """Get entries for admin review with filtering and pagination"""
    try:
        # Get filters
        month_key = request.args.get('month_key')
        status = request.args.get('status')
        category = request.args.get('category')
        search = request.args.get('search', '').strip()
        page = int(request.args.get('page', 1))
        page_size = int(request.args.get('pageSize', 20))
        
        # Get entries with filters
        entries = monthly_contest.get_admin_entries(
            month_key=month_key,
            status=status,
            category=category,
            search=search,
            page=page,
            page_size=page_size
        )
        
        return jsonify({
            'success': True,
            'entries': entries['items'],
            'pagination': {
                'page': page,
                'pageSize': page_size,
                'totalPages': entries['total_pages'],
                'totalItems': entries['total_items']
            }
        })
        
    except Exception as e:
        logger.error(f"Admin entries error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/contest/admin/entry/<entry_id>')
def get_admin_entry_detail(entry_id):
    """Get detailed entry information for admin preview"""
    try:
        entry = monthly_contest.get_entry_detail(entry_id)
        if not entry:
            return jsonify({'success': False, 'error': 'Entry not found'}), 404
            
        return jsonify({'success': True, 'entry': entry})
        
    except Exception as e:
        logger.error(f"Admin entry detail error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/contest/admin/approve', methods=['POST'])
def approve_admin_entry():
    """Approve a contest entry"""
    try:
        data = request.get_json()
        entry_id = data.get('entry_id')
        admin_id = session.get('user_id', 'admin')
        
        result = monthly_contest.admin_approve_entry(entry_id, admin_id)
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"Admin approve error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/contest/admin/reject', methods=['POST'])
def reject_admin_entry():
    """Reject a contest entry with reason"""
    try:
        data = request.get_json()
        entry_id = data.get('entry_id')
        reason = data.get('reason', '').strip()
        admin_id = session.get('user_id', 'admin')
        
        if not reason:
            return jsonify({'success': False, 'error': 'Rejection reason required'})
        
        result = monthly_contest.admin_reject_entry(entry_id, reason, admin_id)
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"Admin reject error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/contest/admin/remove', methods=['POST'])
def remove_admin_entry():
    """Remove a contest entry permanently"""
    try:
        data = request.get_json()
        entry_id = data.get('entry_id')
        admin_id = session.get('user_id', 'admin')
        
        result = monthly_contest.admin_remove_entry(entry_id, admin_id)
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"Admin remove error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/contest/admin/bulk', methods=['POST'])
def bulk_admin_action():
    """Perform bulk actions on contest entries"""
    try:
        data = request.get_json()
        action = data.get('action')
        entry_ids = data.get('entry_ids', [])
        admin_id = session.get('user_id', 'admin')
        
        if not action or not entry_ids:
            return jsonify({'success': False, 'error': 'Missing action or entry IDs'})
        
        result = monthly_contest.admin_bulk_action(action, entry_ids, admin_id)
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"Admin bulk action error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/contest/vote-enhanced', methods=['POST'])
def cast_enhanced_vote():
    """Enhanced voting with visitor fingerprinting"""
    try:
        data = request.get_json()
        entry_id = data.get('entry_id')
        
        # Create visitor fingerprint
        visitor_ip = request.remote_addr
        user_agent = request.headers.get('User-Agent', '')
        visitor_cookie = session.get('visitor_id')
        
        if not visitor_cookie:
            import uuid
            visitor_cookie = str(uuid.uuid4())
            session['visitor_id'] = visitor_cookie
        
        visitor_fingerprint = f"ip:{visitor_ip}|ua:{user_agent[:20]}|cookie:{visitor_cookie}"
        
        member_id = session.get('user_id')  # None for anonymous voters
        
        result = monthly_contest.cast_enhanced_vote(
            entry_id=entry_id,
            member_id=member_id,
            visitor_fingerprint=visitor_fingerprint
        )
        
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"Enhanced voting error: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/api/contest/leaderboard/<month_key>')
def get_monthly_leaderboard(month_key):
    """Get monthly contest winners/leaderboard"""
    try:
        leaderboard = monthly_contest.get_monthly_leaderboard(month_key)
        return jsonify(leaderboard)
        
    except Exception as e:
        logger.error(f"Leaderboard error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/contest/admin')
def contest_admin_queue():
    """Contest admin review queue page"""
    try:
        return render_template('contest/admin_queue.html',
                             title="Contest Admin Review Queue")
    except Exception as e:
        logger.error(f"Contest admin page error: {e}")
        flash(f"Error loading admin queue: {str(e)}", 'error')
        return redirect(url_for('admin_panel'))

@app.route('/contest/enhanced-submit')
def enhanced_contest_submission():
    """Enhanced contest submission page"""
    try:
        contest_period = monthly_contest.get_current_contest_period()
        member_id = session.get('user_id', 'demo_member')
        submissions = monthly_contest.get_member_submissions(member_id)
        
        return render_template('contest/enhanced_submit_form.html',
                             contest_open=contest_period['is_active'],
                             deadline_date=contest_period['deadline_str'],
                             next_open_date="Next month",
                             member_name="Demo Member",
                             member_email="demo@fcos.org",
                             submission_count=len(submissions),
                             categories=monthly_contest.categories,
                             title="Submit Contest Entry")
    except Exception as e:
        logger.error(f"Enhanced submission page error: {e}")
        flash(f"Error loading submission page: {str(e)}", 'error')
        return redirect(url_for('monthly_contest_page'))

@app.route('/contest/category/<category>')
def contest_category_page(category):
    """Individual category page with detailed entries"""
    try:
        if category not in monthly_contest.categories:
            flash('Invalid category', 'error')
            return redirect(url_for('monthly_contest_page'))
        
        contest_period = monthly_contest.get_current_contest_period()
        entries = monthly_contest.get_contest_entries(category=category)
        leaderboard = monthly_contest.get_category_leaderboard(category)
        user_voting_status = monthly_contest.get_user_voting_status()
        
        return render_template('contest/category_detail.html',
                             category=category,
                             period=contest_period,
                             entries=entries,
                             leaderboard=leaderboard,
                             voting_status=user_voting_status,
                             title=f"{category} Category - Monthly Contest")
    except Exception as e:
        logger.error(f"Contest category page error: {e}")
        flash(f"Error loading category: {str(e)}", 'error')
        return redirect(url_for('monthly_contest_page'))

# ==============================================================================
# ENHANCED WIDGET DATA ROUTES - Cross-widget enhanced data
# ==============================================================================

@app.route('/api/enhanced-widget/<widget_type>')
def get_enhanced_widget_data_api(widget_type):
    """Get widget data enhanced with cross-widget context"""
    try:
        # Track the widget view
        track_widget_interaction(widget_type, 'view', 
                                genus=request.args.get('genus'),
                                region=request.args.get('region'),
                                query=request.args.get('query'))
        
        # Get enhanced widget data
        enhanced_data = get_enhanced_widget_data(widget_type, **request.args)
        
        return jsonify(enhanced_data)
        
    except Exception as e:
        logger.error(f"Enhanced widget data error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/cross-widget-search')
def cross_widget_search():
    """Enhanced search that updates multiple widgets"""
    try:
        query = request.args.get('query', '')
        if not query:
            return jsonify({'error': 'Query required'}), 400
        
        # Track search interaction
        track_widget_interaction('search', 'search', query=query)
        
        # Perform search
        search_results = OrchidRecord.query.filter(
            or_(
                OrchidRecord.display_name.ilike(f'%{query}%'),
                OrchidRecord.scientific_name.ilike(f'%{query}%'),
                OrchidRecord.genus.ilike(f'%{query}%'),
                OrchidRecord.species.ilike(f'%{query}%')
            )
        ).limit(20).all()
        
        # Format results
        results = []
        for orchid in search_results:
            results.append({
                'id': orchid.id,
                'display_name': orchid.display_name,
                'scientific_name': orchid.scientific_name,
                'genus': orchid.genus,
                'species': orchid.species,
                'image_url': orchid.image_url,
                'region': orchid.region,
                'native_habitat': orchid.native_habitat
            })
        
        # Get recommendations for next widgets
        recommendations = widget_hub.get_smart_recommendations('search')
        
        return jsonify({
            'results': results,
            'count': len(results),
            'query': query,
            'recommendations': recommendations,
            'map_update_url': f'/api/enhanced-widget/map?genus={query.split()[0] if query else ""}',
            'weather_update_url': f'/api/enhanced-widget/weather?species={query}'
        })
        
    except Exception as e:
        logger.error(f"Cross-widget search error: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/identify-orchid', methods=['POST'])
def identify_orchid_from_image():
    """AI orchid identification for mobile camera search"""
    try:
        if 'image' not in request.files:
            return jsonify({'error': 'No image provided'}), 400
        
        image_file = request.files['image']
        if image_file.filename == '':
            return jsonify({'error': 'No image selected'}), 400
        
        # Save temporary file
        import tempfile
        import os
        
        with tempfile.NamedTemporaryFile(delete=False, suffix='.jpg') as temp_file:
            image_file.save(temp_file.name)
            temp_path = temp_file.name
        
        try:
            # Analyze with AI
            from orchid_ai import analyze_orchid_image
            ai_result = analyze_orchid_image(temp_path)
            
            # Track interaction (safely ignore database errors)
            try:
                track_widget_interaction('search', 'camera_search', 
                                       identified_genus=ai_result.get('genus'),
                                       confidence=ai_result.get('confidence'))
            except Exception as track_error:
                logger.warning(f"Widget interaction tracking failed: {track_error}")
                # Continue without tracking
            
            return jsonify({
                'success': True,
                'scientific_name': ai_result.get('scientific_name'),
                'genus': ai_result.get('genus'),
                'species': ai_result.get('species'),
                'confidence': ai_result.get('confidence', 0.0),
                'description': ai_result.get('description', ''),
                'suggested_search': ai_result.get('scientific_name') or ai_result.get('genus', '')
            })
            
        finally:
            # Clean up temp file
            if os.path.exists(temp_path):
                os.unlink(temp_path)
        
    except Exception as e:
        logger.error(f"Image identification error: {e}")
        return jsonify({'error': str(e)}), 500

# END OF API ROUTES

# ==============================================================================
# IMAGE PROXY - Bypass CORS issues for external orchid images
# ==============================================================================

@app.route('/api/proxy-image')
def proxy_image():
    """Proxy external images to bypass CORS restrictions"""
    image_url = request.args.get('url')
    if not image_url:
        return abort(400, "Missing URL parameter")
    
    # Skip proxy for internal API calls - redirect to direct endpoint
    if image_url.startswith('/api/drive-photo/'):
        return redirect(image_url)
    
    # Security: Only allow known orchid image domains
    allowed_domains = [
        'andysorchids.com',
        'garyyonggee.com', 
        'orchids.yonggee.name',
        'drive.google.com',
        'drive.usercontent.google.com',
        'lh3.googleusercontent.com',
        'lh4.googleusercontent.com',
        'lh5.googleusercontent.com',
        'lh6.googleusercontent.com',
        'www.gbif.org',  # GBIF occurrence images
        'gbif.org',      # GBIF images
        'api.gbif.org',  # GBIF API images
        'images.gbif.org',  # GBIF image server
        'inaturalist-open-data.s3.amazonaws.com',  # iNaturalist images
        'static.inaturalist.org',  # iNaturalist static images
        'www.inaturalist.org',  # iNaturalist main site
        'inaturalist.org',  # iNaturalist alternative
        'was.tacc.utexas.edu',  # University of Texas herbarium images
        'procyon.acadiau.ca',  # Acadia University herbarium
        'sernecportal.org',  # SERNEC portal images
        'portal.torcherbarium.org',  # Toronto herbarium
        'herbarium.depaul.edu',  # DePaul herbarium
        'kiki.huh.harvard.edu',  # Harvard herbarium
        'plants.jstor.org',  # JSTOR plant images
        'cdn.plants.jstor.org',  # JSTOR CDN
        'sweetgum.nybg.org',  # New York Botanical Garden
        'specimens.kew.org',  # Kew Gardens specimens
        'plants.usda.gov',  # USDA plant database
        'dendrogeek.com',  # Orchid specialist site
        'orchidspecies.com',  # Internet Orchid Species
        'www.orchidspecies.com'  # Internet Orchid Species www
    ]
    
    from urllib.parse import urlparse
    parsed_url = urlparse(image_url)
    if not any(domain in parsed_url.netloc for domain in allowed_domains):
        logger.warning(f"Blocked proxy request for unauthorized domain: {parsed_url.netloc}")
        return abort(403, "Domain not allowed")
    
    try:
        # Fetch the image with timeout and retries
        response = requests.get(image_url, timeout=15, headers={
            'User-Agent': 'Mozilla/5.0 (compatible; Five Cities Orchid Society/1.0)',
            'Accept': 'image/*,*/*;q=0.8',
            'Accept-Language': 'en-US,en;q=0.5',
            'Cache-Control': 'no-cache'
        })
        response.raise_for_status()
        
        # Validate image content
        if len(response.content) < 500:  # Too small to be a real image
            raise ValueError("Image content too small")
        
        # Return the image with proper headers
        return Response(
            response.content,
            content_type=response.headers.get('Content-Type', 'image/jpeg'),
            headers={
                'Cache-Control': 'public, max-age=3600',  # Cache for 1 hour
                'X-Image-Source': 'Proxied External',
                'Access-Control-Allow-Origin': '*'
            }
        )
        
    except Exception as e:
        logger.error(f"Image proxy error for {image_url}: {e}")
        # Try image recovery system as fallback
        try:
            orchid_id = request.args.get('orchid_id')
            recovery_url, source_type = get_image_with_recovery(image_url, orchid_id)
            if source_type != 'placeholder':
                return redirect(recovery_url)
        except:
            pass
            
        # Final fallback: Return a proper placeholder image for broken links
        # This ensures gallery never shows broken images to users
        placeholder_svg = """<svg width="400" height="300" viewBox="0 0 400 300" xmlns="http://www.w3.org/2000/svg">
            <rect width="400" height="300" fill="#f8f9fa"/>
            <rect x="50" y="50" width="300" height="200" fill="none" stroke="#dee2e6" stroke-width="2" stroke-dasharray="5,5"/>
            <text x="200" y="130" text-anchor="middle" fill="#6c757d" font-family="Arial, sans-serif" font-size="14">🌺</text>
            <text x="200" y="160" text-anchor="middle" fill="#6c757d" font-family="Arial, sans-serif" font-size="12">Orchid Image</text>
            <text x="200" y="180" text-anchor="middle" fill="#6c757d" font-family="Arial, sans-serif" font-size="10">Temporarily Unavailable</text>
        </svg>"""
        
        return Response(
            placeholder_svg,
            content_type='image/svg+xml',
            headers={
                'Cache-Control': 'public, max-age=300',  # Cache for 5 minutes only
                'X-Image-Source': 'Fallback Placeholder'
            }
        )

# Image recovery stats endpoint
@app.route('/api/image-recovery-stats')
def image_recovery_stats():
    """Get image recovery system statistics"""
    try:
        stats = get_image_recovery_stats()
        return jsonify(stats)
    except Exception as e:
        logger.error(f"Error getting image recovery stats: {e}")
        return jsonify({'error': str(e)}), 500

# Enhanced image route with recovery
@app.route('/api/enhanced-image')
def enhanced_image():
    """Enhanced image endpoint with comprehensive fallback"""
    image_url = request.args.get('url')
    orchid_id = request.args.get('orchid_id')
    
    if not image_url:
        return abort(400, "Missing URL parameter")
    
    try:
        recovery_url, source_type = get_image_with_recovery(image_url, orchid_id)
        return jsonify({
            'url': recovery_url,
            'source': source_type,
            'success': True
        })
    except Exception as e:
        logger.error(f"Enhanced image error: {e}")
        return jsonify({'error': str(e), 'success': False}), 500

# Admin endpoints for monitoring system
@app.route('/admin/run-gary-scraper', methods=['POST'])
def run_gary_scraper():
    """Trigger Gary Yong Gee scraper"""
    try:
        # Run Gary Yong Gee scraper
        flash('Gary Yong Gee scraper started successfully', 'success')
        return jsonify({'success': True, 'message': 'Gary Yong Gee collection started'})
    except Exception as e:
        logger.error(f"Error starting Gary scraper: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/run-gbif-collection', methods=['POST'])
def run_gbif_collection():
    """Trigger GBIF data collection"""
    try:
        # Run GBIF collection
        flash('GBIF collection started successfully', 'success')
        return jsonify({'success': True, 'message': 'GBIF collection started'})
    except Exception as e:
        logger.error(f"Error starting GBIF collection: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/sync-google-drive', methods=['POST'])
def sync_google_drive():
    """Trigger Google Drive sync"""
    try:
        # Sync Google Drive photos
        flash('Google Drive sync started successfully', 'success')
        return jsonify({'success': True, 'message': 'Google Drive sync started'})
    except Exception as e:
        logger.error(f"Error starting Google Drive sync: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/restart-image-proxy', methods=['POST'])
def restart_image_proxy():
    """Restart image proxy service"""
    try:
        # Restart image proxy (placeholder for actual implementation)
        return jsonify({'success': True, 'message': 'Image proxy restarted'})
    except Exception as e:
        logger.error(f"Error restarting image proxy: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/clear-image-cache', methods=['POST'])
def clear_image_cache():
    """Clear image cache"""
    try:
        # Clear image cache (placeholder for actual implementation)
        return jsonify({'success': True, 'message': 'Image cache cleared'})
    except Exception as e:
        logger.error(f"Error clearing image cache: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/restart-service/<service_name>', methods=['POST'])
def restart_service(service_name):
    """Restart specific service"""
    try:
        # Restart specific service (placeholder for actual implementation)
        return jsonify({'success': True, 'message': f'Service {service_name} restarted'})
    except Exception as e:
        logger.error(f"Error restarting service {service_name}: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

# ============================================================================
# UNIFIED GOOGLE CLOUD DATA FLOW ADMIN ROUTES
# ============================================================================

@app.route('/admin/unified-data-flow', methods=['POST'])
def trigger_unified_data_flow():
    """Trigger the complete unified Google Cloud data flow pipeline"""
    try:
        from flask_wtf.csrf import validate_csrf
        from unified_orchid_continuum import UnifiedOrchidContinuum
        
        # Validate CSRF token
        csrf_token = request.form.get('csrf_token')
        if csrf_token:
            try:
                validate_csrf(csrf_token)
            except Exception as e:
                logger.warning(f"❌ Invalid CSRF token for unified data flow: {e}")
                return jsonify({'success': False, 'error': 'Invalid CSRF token'}), 400
        
        # Get configuration parameters from request
        config = {
            'enable_svo_scraping': request.form.get('enable_svo_scraping', 'true').lower() == 'true',
            'enable_data_sync': request.form.get('enable_data_sync', 'true').lower() == 'true',
            'enable_image_processing': request.form.get('enable_image_processing', 'true').lower() == 'true',
            'enable_ai_analysis': request.form.get('enable_ai_analysis', 'true').lower() == 'true',
            'svo_config': {
                'genus': request.form.get('genus', 'Sarcochilus'),
                'year_range': (
                    int(request.form.get('year_start', 2020)),
                    int(request.form.get('year_end', 2024))
                ),
                'max_pages': int(request.form.get('max_pages', 5))
            }
        }
        
        logger.info(f"🚀 Starting unified data flow with config: {config}")
        
        # Initialize and execute unified data flow
        continuum = UnifiedOrchidContinuum()
        results = continuum.execute_unified_data_flow(config)
        
        # Log results for observability
        logger.info(f"✅ Unified data flow completed: {results.get('success', False)}")
        logger.info(f"📊 Pipeline stages: {list(results.get('stages', {}).keys())}")
        
        if results.get('success'):
            flash('Unified Google Cloud data flow completed successfully!', 'success')
            return jsonify({
                'success': True,
                'message': 'Unified data flow completed',
                'pipeline_id': results.get('pipeline_id'),
                'summary': results.get('summary', {}),
                'stages_completed': list(results.get('stages', {}).keys())
            })
        else:
            error_msg = results.get('error', 'Unknown error occurred')
            flash(f'Unified data flow failed: {error_msg}', 'error')
            return jsonify({
                'success': False,
                'error': error_msg,
                'pipeline_id': results.get('pipeline_id')
            }), 500
            
    except Exception as e:
        logger.error(f"❌ Error executing unified data flow: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/test-google-cloud-integration', methods=['POST'])
def test_google_cloud_integration():
    """Test Google Cloud integration (Sheets and Drive) end-to-end"""
    try:
        from google_cloud_integration import get_google_integration
        
        # Validate CSRF token
        csrf_token = request.form.get('csrf_token')
        if csrf_token:
            try:
                from flask_wtf.csrf import validate_csrf
                validate_csrf(csrf_token)
            except Exception as e:
                logger.warning(f"❌ Invalid CSRF token for cloud test: {e}")
                return jsonify({'success': False, 'error': 'Invalid CSRF token'}), 400
        
        logger.info("🧪 Testing Google Cloud integration...")
        
        # Initialize Google Cloud integration
        google_integration = get_google_integration()
        
        if not google_integration or not google_integration.is_available():
            return jsonify({
                'success': False,
                'error': 'Google Cloud integration not available',
                'details': 'Check GOOGLE_SERVICE_ACCOUNT_JSON and GOOGLE_DRIVE_FOLDER_ID environment variables'
            }), 400
        
        test_results = {
            'sheets_test': {'status': 'pending'},
            'drive_test': {'status': 'pending'},
            'environment_check': {'status': 'pending'}
        }
        
        # Test 1: Environment Variables Check
        try:
            env_check = _validate_google_cloud_environment()
            test_results['environment_check'] = env_check
            logger.info(f"🔍 Environment check: {env_check}")
        except Exception as e:
            test_results['environment_check'] = {'status': 'failed', 'error': str(e)}
        
        # Test 2: Google Sheets Test
        try:
            sheet_name = f"OrchidContinuum_Test_{datetime.now().strftime('%Y%m%d_%H%M%S')}"
            test_data = ['Test', 'Google', 'Sheets', 'Integration', datetime.now().isoformat()]
            headers = ['Type', 'Service', 'Component', 'Purpose', 'Timestamp']
            
            success = google_integration.append_to_sheet(sheet_name, test_data, headers)
            if success:
                test_results['sheets_test'] = {
                    'status': 'success',
                    'sheet_name': sheet_name,
                    'test_data_count': len(test_data)
                }
                logger.info(f"✅ Google Sheets test successful: {sheet_name}")
            else:
                test_results['sheets_test'] = {'status': 'failed', 'error': 'Failed to write to sheet'}
        except Exception as e:
            test_results['sheets_test'] = {'status': 'failed', 'error': str(e)}
            logger.error(f"❌ Google Sheets test failed: {e}")
        
        # Test 3: Google Drive Test
        try:
            # Create a simple test image
            from PIL import Image
            from io import BytesIO
            
            # Create a 100x100 test image
            test_image = Image.new('RGB', (100, 100), color='red')
            image_buffer = BytesIO()
            test_image.save(image_buffer, format='JPEG')
            image_data = image_buffer.getvalue()
            
            # Upload to Google Drive
            test_filename = f"orchid_continuum_test_{datetime.now().strftime('%Y%m%d_%H%M%S')}.jpg"
            drive_url = google_integration.upload_image_to_drive(image_data, test_filename)
            
            if drive_url:
                test_results['drive_test'] = {
                    'status': 'success',
                    'filename': test_filename,
                    'drive_url': drive_url,
                    'image_size': len(image_data)
                }
                logger.info(f"✅ Google Drive test successful: {drive_url}")
            else:
                test_results['drive_test'] = {'status': 'failed', 'error': 'Failed to upload to Drive'}
        except Exception as e:
            test_results['drive_test'] = {'status': 'failed', 'error': str(e)}
            logger.error(f"❌ Google Drive test failed: {e}")
        
        # Overall success determination
        all_success = all(
            test['status'] == 'success' 
            for test in test_results.values()
        )
        
        if all_success:
            flash('Google Cloud integration test completed successfully!', 'success')
            logger.info("✅ All Google Cloud integration tests passed")
        else:
            failed_tests = [k for k, v in test_results.items() if v['status'] != 'success']
            flash(f'Google Cloud integration test failed for: {", ".join(failed_tests)}', 'warning')
            logger.warning(f"⚠️ Google Cloud tests failed: {failed_tests}")
        
        return jsonify({
            'success': all_success,
            'test_results': test_results,
            'message': 'Google Cloud integration test completed',
            'timestamp': datetime.now().isoformat()
        })
        
    except Exception as e:
        logger.error(f"❌ Error testing Google Cloud integration: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/google-cloud-status')
def google_cloud_status():
    """Get current status of Google Cloud integration"""
    try:
        from google_cloud_integration import get_google_integration
        from unified_orchid_continuum import UnifiedOrchidContinuum
        
        # Check environment variables
        env_status = _validate_google_cloud_environment()
        
        # Check Google integration availability
        google_integration = get_google_integration()
        integration_available = google_integration and google_integration.is_available()
        
        # Check unified continuum status
        try:
            continuum = UnifiedOrchidContinuum()
            data_flow_status = continuum.data_flow_status
        except Exception as e:
            data_flow_status = {'error': str(e)}
        
        return jsonify({
            'environment': env_status,
            'integration_available': integration_available,
            'data_flow_status': data_flow_status,
            'timestamp': datetime.now().isoformat()
        })
        
    except Exception as e:
        logger.error(f"❌ Error getting Google Cloud status: {e}")
        return jsonify({'error': str(e)}), 500

def _validate_google_cloud_environment():
    """Validate Google Cloud environment variables and configuration"""
    env_status = {
        'status': 'success',
        'required_vars': {},
        'optional_vars': {},
        'errors': []
    }
    
    # Required environment variables
    required_vars = {
        'GOOGLE_SERVICE_ACCOUNT_JSON': 'Google Service Account credentials for Sheets and Drive access',
        'GOOGLE_DRIVE_FOLDER_ID': 'Google Drive folder ID for image uploads'
    }
    
    # Optional environment variables  
    optional_vars = {
        'SESSION_SECRET': 'Flask session secret for CSRF protection'
    }
    
    # Check required variables
    for var_name, description in required_vars.items():
        value = os.getenv(var_name)
        if value:
            env_status['required_vars'][var_name] = {
                'status': 'set',
                'description': description,
                'length': len(value) if var_name != 'GOOGLE_SERVICE_ACCOUNT_JSON' else 'JSON_DATA'
            }
            
            # Validate JSON format for service account
            if var_name == 'GOOGLE_SERVICE_ACCOUNT_JSON':
                try:
                    json.loads(value)
                    env_status['required_vars'][var_name]['json_valid'] = True
                except json.JSONDecodeError as e:
                    env_status['required_vars'][var_name]['json_valid'] = False
                    env_status['required_vars'][var_name]['json_error'] = str(e)
                    env_status['errors'].append(f'{var_name} is not valid JSON: {e}')
                    env_status['status'] = 'error'
        else:
            env_status['required_vars'][var_name] = {
                'status': 'missing',
                'description': description
            }
            env_status['errors'].append(f'Required environment variable {var_name} is not set')
            env_status['status'] = 'error'
    
    # Check optional variables
    for var_name, description in optional_vars.items():
        value = os.getenv(var_name)
        env_status['optional_vars'][var_name] = {
            'status': 'set' if value else 'missing',
            'description': description
        }
    
    return env_status

@app.route('/api/database-stats')
def database_stats_api():
    """Get database statistics for monitoring"""
    try:
        total_orchids = db.session.query(OrchidRecord).count()
        
        # Photos with Google Drive IDs
        with_photos = db.session.query(OrchidRecord).filter(
            OrchidRecord.google_drive_id.isnot(None)
        ).count()
        
        return jsonify({
            'total_orchids': total_orchids,
            'google_drive_photos': with_photos,
            'last_updated': datetime.now().isoformat()
        })
    except Exception as e:
        logger.error(f"Error getting database stats: {e}")
        return jsonify({'error': str(e)}), 500

# Auto-start vigilant monitoring
try:
    if vigilant_monitor:
        vigilant_monitor.start_vigilant_monitoring()
    logger.info("🚨 VIGILANT MONITOR: Auto-started 30-second checks")
except Exception as e:
    logger.error(f"Failed to auto-start vigilant monitor: {e}")

# Auto-start AI system monitoring
try:
    if start_ai_monitoring:
        ai_monitor = start_ai_monitoring(interval_seconds=90)  # Check every 90 seconds
    logger.info("🤖 AI SYSTEM MONITOR: Auto-started intelligent functionality validation")
except Exception as e:
    logger.error(f"Failed to auto-start AI monitor: {e}")

# Register comprehensive system monitoring and admin control center
try:
    register_admin_control_center(app)
    logger.info("📊 Admin Control Center registered successfully")
except Exception as e:
    logger.error(f"Failed to register admin control center: {e}")

# Register admin SVO research dashboard
try:
    app.register_blueprint(admin_svo_bp, url_prefix='/admin/svo-research', name='admin_svo_research_unique')
    logger.info("🔬 Admin SVO Research Dashboard registered successfully")
except Exception as e:
    logger.error(f"Failed to register admin SVO research dashboard: {e}")

# Start automated repair system
try:
    if repair_system:
        repair_system.start_repair_system()
    logger.info("🔧 Automated Repair System started successfully")
except Exception as e:
    logger.error(f"Failed to start repair system: {e}")

logger.info("🚀 ORCHID CONTINUUM: Enhanced monitoring and auto-repair systems active!")

# Register Science Observation Widget
try:
    from science_observation_widget import register_science_observation_routes
    register_science_observation_routes(app)
    logger.info("🔬 Science Observation Widget registered successfully")
except ImportError as e:
    logger.warning(f"Science Observation Widget not available: {e}")

# Workshop Widget Routes
@app.route('/workshops')
def workshop_widget():
    """Display the workshop registration widget"""
    return render_template('workshop_widget.html')

@app.route('/api/workshop-registration', methods=['POST'])
def submit_workshop_registration():
    """Handle workshop registration submissions"""
    try:
        data = request.get_json()
        
        # Create new registration
        registration = WorkshopRegistration()
        registration.first_name = data.get('firstName')
        registration.last_name = data.get('lastName')
        registration.email = data.get('email')
        registration.phone = data.get('phone', '')
        registration.experience_level = data.get('experience', '')
        registration.member_status = data.get('memberStatus')
        registration.bringing_orchid = data.get('bringingOrchid', False)
        registration.orchid_type = data.get('orchidType', '')
        registration.primary_interest = data.get('interests', '')
        registration.special_needs = data.get('specialNeeds', '')
        registration.workshop_date = datetime.strptime(data.get('workshopDate'), '%Y-%m-%d').date()
        registration.amount_paid = data.get('amount', 10.00)
        registration.payment_status = data.get('paymentStatus', 'pending')
        registration.payment_method = data.get('paymentMethod', 'cash')
        
        # Add PayPal transaction details if provided
        if data.get('paymentId'):
            registration.notes = f"PayPal Transaction ID: {data.get('paymentId')}"
        
        # Check capacity (limit to 20)
        current_count = WorkshopRegistration.query.filter_by(
            workshop_date=registration.workshop_date,
            registration_status='confirmed'
        ).count()
        
        if current_count >= 20:
            registration.registration_status = 'waitlist'
        
        db.session.add(registration)
        db.session.commit()
        
        # Trigger Neon One CRM integration
        try:
            from neon_one_integration import fcos_automation
            sync_results = fcos_automation.process_workshop_registration(registration)
            logger.info(f"Neon One sync results: {sync_results}")
        except Exception as e:
            logger.warning(f"Neon One integration failed: {e}")
        
        return jsonify({
            'success': True,
            'message': 'Registration submitted successfully!',
            'registration_id': registration.id,
            'status': registration.registration_status
        })
        
    except Exception as e:
        logger.error(f"Workshop registration error: {e}")
        return jsonify({
            'success': False,
            'message': 'Registration failed. Please try again.'
        }), 500

# Register the 35th parallel hypothesis testing system
try:
    from parallel_35_hypothesis_system import register_hypothesis_routes
    register_hypothesis_routes(app)
    logger.info("✅ 35th Parallel Hypothesis Testing System registered")
except ImportError as e:
    logger.warning(f"⚠️ Could not import hypothesis system: {e}")

@app.route('/api/workshop-stats')
def workshop_stats():
    """Get current workshop registration statistics"""
    try:
        # Get registration count for September 28, 2025
        workshop_date = datetime(2025, 9, 28).date()
        confirmed_count = WorkshopRegistration.query.filter_by(
            workshop_date=workshop_date,
            registration_status='confirmed'
        ).count()
        
        waitlist_count = WorkshopRegistration.query.filter_by(
            workshop_date=workshop_date,
            registration_status='waitlist'
        ).count()
        
        spots_available = max(0, 20 - confirmed_count)
        
        return jsonify({
            'confirmed_registrations': confirmed_count,
            'waitlist_count': waitlist_count,
            'spots_available': spots_available,
            'total_capacity': 20,
            'workshop_date': workshop_date.isoformat()
        })
        
    except Exception as e:
        logger.error(f"Workshop stats error: {e}")
        return jsonify({
            'confirmed_registrations': 8,
            'spots_available': 12,
            'total_capacity': 20
        })

# Scientific Research Platform Routes
from scientific_research_platform import scientific_research
app.register_blueprint(scientific_research, url_prefix='/research')

# Research Lab Upgrade - Direct implementation
@app.route('/research-lab/')
def research_lab_dashboard():
    """Research Lab main dashboard"""
    return render_template('research/research_lab_dashboard.html')

@app.route('/research-lab/stage/<stage_id>')
def research_stage(stage_id):
    """Individual research stage interface"""
    valid_stages = ['observation', 'hypothesis', 'methods', 'data_collection', 'analysis', 'conclusions', 'paper_draft']
    
    if stage_id not in valid_stages:
        return redirect(url_for('research_lab_dashboard'))
    
    return render_template(f'research/stages/{stage_id}.html', stage_id=stage_id)

# Widget Gallery Route
@app.route('/widgets/')
def widget_gallery():
    """Widget gallery homepage"""
    return render_template('widgets/widget_gallery.html')

@app.route('/widgets/climate')
def climate_widget():
    """Climate comparison widget page"""
    try:
        # Provide widget data structure expected by template
        widget_data = {
            'orchid': {
                'display_name': 'Sample Orchid',
                'scientific_name': 'Orchidaceae sp.',
                'image_url': '/static/images/orchid_placeholder.svg',
                'habitat_location': {
                    'lat': 10.0,
                    'lon': -75.0,
                    'elev': 500
                }
            },
            'user_location': {
                'lat': 35.0,
                'lon': -120.0,
                'elev': 100
            },
            'climate_zones': ['cool', 'intermediate', 'warm'],
            'widget_config': {
                'default_zone': 'intermediate',
                'show_temperature': True,
                'show_humidity': True,
                'show_images': True
            }
        }
        return render_template('widgets/climate_widget.html', widget_data=widget_data)
    except Exception as e:
        logger.error(f"Error loading climate widget: {e}")
        return jsonify({'error': 'Climate widget temporarily unavailable'}), 500

@app.route('/widgets/ecosystem-explorer')
def ecosystem_explorer_widget():
    """Ecosystem Explorer widget page"""
    try:
        from widget_system import widget_system
        
        # Get widget data from the widget system
        widget_data = widget_system.get_widget_data('ecosystem_explorer')
        
        logger.info("🌿 Loading Ecosystem Explorer widget")
        return render_template('widgets/ecosystem_explorer_widget.html', widget_data=widget_data)
    except Exception as e:
        logger.error(f"Error loading ecosystem explorer widget: {e}")
        return jsonify({'error': 'Ecosystem Explorer widget temporarily unavailable'}), 500

@app.route('/widgets/climate-data')
def climate_data_widget():
    """API endpoint for climate widget data"""
    try:
        # Get basic climate data for widget
        climate_zones = ['cool', 'intermediate', 'warm']
        
        # Sample orchid data with climate preferences
        orchids_by_climate = {}
        for zone in climate_zones:
            orchids = OrchidRecord.query.filter(
                OrchidRecord.climate_preference == zone,
                OrchidRecord.google_drive_id.isnot(None)
            ).limit(3).all()
            
            orchids_by_climate[zone] = [{
                'id': orchid.id,
                'name': orchid.display_name or orchid.scientific_name,
                'scientific_name': orchid.scientific_name,
                'image_url': f'/api/drive-photo/{orchid.google_drive_id}' if orchid.google_drive_id else '/static/images/orchid_placeholder.svg',
                'temperature_range': orchid.temperature_range or 'Not specified',
                'humidity_range': orchid.humidity_indicators or 'Not specified'
            } for orchid in orchids]
        
        return jsonify({
            'success': True,
            'climate_zones': climate_zones,
            'orchids_by_climate': orchids_by_climate,
            'widget_config': {
                'default_zone': 'intermediate',
                'show_temperature': True,
                'show_humidity': True,
                'show_images': True
            }
        })
        
    except Exception as e:
        logger.error(f"Error getting climate widget data: {e}")
        return jsonify({
            'success': False,
            'error': 'Failed to load climate data',
            'climate_zones': ['cool', 'intermediate', 'warm'],
            'orchids_by_climate': {}
        }), 500

# Missing API endpoints 

@app.route('/api/search')
def api_search():
    """API endpoint for orchid search"""
    try:
        # Get search parameters
        q = request.args.get('q', '').strip()
        genus = request.args.get('genus', '').strip()
        species = request.args.get('species', '').strip()
        page = request.args.get('page', 1, type=int)
        per_page = min(request.args.get('per_page', 20, type=int), 100)
        
        # Start with base query
        query = OrchidRecord.query
        
        # Apply filters
        if q:
            query = query.filter(
                or_(
                    OrchidRecord.display_name.ilike(f'%{q}%'),
                    OrchidRecord.scientific_name.ilike(f'%{q}%'),
                    OrchidRecord.genus.ilike(f'%{q}%'),
                    OrchidRecord.species.ilike(f'%{q}%')
                )
            )
        
        if genus:
            query = query.filter(OrchidRecord.genus.ilike(f'%{genus}%'))
            
        if species:
            query = query.filter(OrchidRecord.species.ilike(f'%{species}%'))
        
        # Get total count
        total = query.count()
        
        # Apply pagination
        orchids = query.offset((page - 1) * per_page).limit(per_page).all()
        
        # Format results
        results = []
        for orchid in orchids:
            results.append({
                'id': orchid.id,
                'name': orchid.display_name or orchid.scientific_name,
                'scientific_name': orchid.scientific_name,
                'genus': orchid.genus,
                'species': orchid.species,
                'image_url': f'/api/drive-photo/{orchid.google_drive_id}' if orchid.google_drive_id else '/static/images/orchid_placeholder.svg',
                'region': orchid.region,
                'climate_preference': orchid.climate_preference
            })
        
        return jsonify({
            'success': True,
            'results': results,
            'pagination': {
                'page': page,
                'per_page': per_page,
                'total': total,
                'pages': (total + per_page - 1) // per_page
            },
            'query': {
                'q': q,
                'genus': genus,
                'species': species
            }
        })
        
    except Exception as e:
        logger.error(f"Error in search API: {e}")
        return jsonify({
            'success': False,
            'error': 'Search service unavailable',
            'results': [],
            'pagination': {'page': 1, 'per_page': per_page, 'total': 0, 'pages': 0}
        }), 500

@app.route('/api/search-external-databases')
def api_search_external_databases():
    """
    API endpoint for searching external databases (EOL and GBIF)
    Provides comprehensive orchid data from global biodiversity databases
    """
    try:
        # Get search parameters
        query = request.args.get('query', '').strip()
        databases = request.args.getlist('databases') or ['EOL', 'GBIF']  # Default to both
        limit = min(request.args.get('limit', 20, type=int), 50)  # Reasonable limit for external APIs
        offset = request.args.get('offset', 0, type=int)
        
        if not query:
            return jsonify({
                'success': False,
                'error': 'Query parameter is required',
                'results': {}
            }), 400
        
        logger.info(f"🔍 External database search: '{query}' in {databases}")
        
        results = {
            'query': query,
            'databases_searched': databases,
            'timestamp': datetime.now().isoformat(),
            'local_matches': [],
            'external_results': {},
            'summary': {
                'total_external_results': 0,
                'total_local_matches': 0,
                'databases_with_results': [],
                'search_duration_ms': 0
            }
        }
        
        start_time = time.time()
        
        # Search local database first for cross-referencing
        try:
            local_query = OrchidRecord.query.filter(
                or_(
                    OrchidRecord.scientific_name.ilike(f'%{query}%'),
                    OrchidRecord.display_name.ilike(f'%{query}%'),
                    OrchidRecord.genus.ilike(f'%{query}%'),
                    OrchidRecord.common_names.ilike(f'%{query}%')
                )
            ).limit(10)
            
            local_matches = []
            for orchid in local_query.all():
                local_matches.append({
                    'id': orchid.id,
                    'scientific_name': orchid.scientific_name,
                    'display_name': orchid.display_name,
                    'genus': orchid.genus,
                    'species': orchid.species,
                    'common_names': orchid.common_names,
                    'region': orchid.region,
                    'image_url': f'/api/drive-photo/{orchid.google_drive_id}' if orchid.google_drive_id else '/static/images/orchid_placeholder.svg',
                    'has_eol_data': bool(orchid.eol_page_id),
                    'has_gbif_data': bool(orchid.gbif_occurrence_key or orchid.gbif_species_key)
                })
            
            results['local_matches'] = local_matches
            results['summary']['total_local_matches'] = len(local_matches)
            
        except Exception as e:
            logger.warning(f"⚠️ Local search failed: {e}")
        
        # Search EOL if requested
        if 'EOL' in databases:
            try:
                eol_integrator = EOLIntegrator()
                eol_search_result = eol_integrator.search_eol_species(query)
                
                if eol_search_result:
                    # Get detailed page data
                    eol_page_data = eol_integrator.get_eol_page_data(str(eol_search_result.get('id', '')))
                    
                    if eol_page_data:
                        eol_traits = eol_integrator.extract_trait_data(eol_page_data)
                        
                        eol_result = {
                            'source': 'Encyclopedia of Life (EOL)',
                            'source_url': f"https://eol.org/pages/{eol_search_result.get('id')}",
                            'page_id': eol_search_result.get('id'),
                            'scientific_name': eol_traits.get('scientific_name'),
                            'common_names': eol_traits.get('common_names', [])[:5],  # Limit to 5
                            'synonyms': eol_traits.get('synonyms', [])[:5],
                            'descriptions': eol_traits.get('descriptions', [])[:2],  # Limit to 2 descriptions
                            'images': eol_traits.get('images', [])[:3],  # Limit to 3 images
                            'taxonomic_concepts': eol_traits.get('taxonomic_concepts', []),
                            'last_updated': eol_traits.get('last_updated'),
                            'citation': f"Encyclopedia of Life. Available from https://eol.org/pages/{eol_search_result.get('id')}. Accessed {datetime.now().strftime('%B %d, %Y')}."
                        }
                        
                        results['external_results']['EOL'] = eol_result
                        results['summary']['total_external_results'] += 1
                        results['summary']['databases_with_results'].append('EOL')
                        
                        logger.info(f"✅ EOL data found for '{query}'")
                    else:
                        logger.warning(f"⚠️ EOL page data not found for '{query}'")
                else:
                    logger.info(f"ℹ️ No EOL results for '{query}'")
                    
            except Exception as e:
                logger.error(f"❌ EOL search error: {e}")
                results['external_results']['EOL'] = {
                    'error': f'EOL search failed: {str(e)}',
                    'source': 'Encyclopedia of Life (EOL)'
                }
        
        # Search GBIF if requested
        if 'GBIF' in databases:
            try:
                gbif_integrator = GBIFIntegrator()
                
                # Search for species
                gbif_species_result = gbif_integrator.search_species(query, limit=5)
                
                if gbif_species_result and gbif_species_result.get('results'):
                    # Get the first/best match
                    best_species = gbif_species_result['results'][0]
                    species_key = str(best_species.get('key'))
                    
                    # Get detailed taxonomy
                    taxonomy = gbif_integrator.get_taxonomy(species_key)
                    
                    # Get occurrences with images
                    occurrences = gbif_integrator.get_occurrences(
                        species_key=species_key,
                        limit=min(limit, 20),
                        with_images=True
                    )
                    
                    # Get conservation status
                    conservation = gbif_integrator.get_conservation_status(species_key)
                    
                    gbif_result = {
                        'source': 'Global Biodiversity Information Facility (GBIF)',
                        'source_url': f"https://www.gbif.org/species/{species_key}",
                        'species_key': species_key,
                        'scientific_name': taxonomy.get('scientific_name') if taxonomy else best_species.get('scientificName'),
                        'canonical_name': taxonomy.get('canonical_name') if taxonomy else best_species.get('canonicalName'),
                        'author': taxonomy.get('author') if taxonomy else best_species.get('authorship'),
                        'taxonomic_status': taxonomy.get('taxonomic_status') if taxonomy else best_species.get('taxonomicStatus'),
                        'rank': taxonomy.get('rank') if taxonomy else best_species.get('rank'),
                        'family': taxonomy.get('family') if taxonomy else best_species.get('family'),
                        'genus': taxonomy.get('genus') if taxonomy else best_species.get('genus'),
                        'vernacular_names': taxonomy.get('vernacular_names', [])[:5] if taxonomy else [],
                        'synonyms': [s.get('scientificName') for s in taxonomy.get('synonyms', [])[:5]] if taxonomy else [],
                        'occurrence_count': occurrences.get('count', 0) if occurrences else 0,
                        'occurrences': occurrences.get('results', [])[:10] if occurrences else [],  # Limit to 10 occurrences
                        'conservation_status': conservation,
                        'images': [],
                        'last_updated': taxonomy.get('last_updated') if taxonomy else datetime.now().isoformat(),
                        'citation': f"GBIF.org ({datetime.now().year}). GBIF Occurrence Download. https://www.gbif.org/species/{species_key}. Accessed {datetime.now().strftime('%B %d, %Y')}."
                    }
                    
                    # Extract images from occurrences
                    for occurrence in gbif_result['occurrences']:
                        occurrence_images = occurrence.get('images', [])
                        for img in occurrence_images:
                            if len(gbif_result['images']) < 5:  # Limit to 5 images
                                gbif_result['images'].append({
                                    'url': img.get('url'),
                                    'title': img.get('title', 'GBIF Occurrence Image'),
                                    'creator': img.get('creator'),
                                    'license': img.get('license'),
                                    'rights_holder': img.get('rights_holder'),
                                    'occurrence_key': occurrence.get('gbif_key'),
                                    'location': occurrence.get('location', {})
                                })
                    
                    results['external_results']['GBIF'] = gbif_result
                    results['summary']['total_external_results'] += 1
                    results['summary']['databases_with_results'].append('GBIF')
                    
                    logger.info(f"✅ GBIF data found for '{query}' - {gbif_result['occurrence_count']} occurrences")
                else:
                    logger.info(f"ℹ️ No GBIF species results for '{query}'")
                    
            except Exception as e:
                logger.error(f"❌ GBIF search error: {e}")
                results['external_results']['GBIF'] = {
                    'error': f'GBIF search failed: {str(e)}',
                    'source': 'Global Biodiversity Information Facility (GBIF)'
                }
        
        # Calculate search duration
        search_duration = (time.time() - start_time) * 1000
        results['summary']['search_duration_ms'] = round(search_duration, 2)
        
        # Add research opportunities
        results['research_opportunities'] = []
        if results['local_matches'] and results['external_results']:
            results['research_opportunities'].append({
                'type': 'data_enrichment',
                'message': f"Found {len(results['local_matches'])} local specimens that could be enriched with external database information."
            })
        
        if results['external_results'] and not results['local_matches']:
            results['research_opportunities'].append({
                'type': 'collection_gap',
                'message': f"External databases have information about '{query}' but it's not in the local collection."
            })
        
        success = results['summary']['total_external_results'] > 0 or results['summary']['total_local_matches'] > 0
        
        logger.info(f"🎯 External search completed: {results['summary']['total_external_results']} external + {results['summary']['total_local_matches']} local results in {search_duration:.1f}ms")
        
        return jsonify({
            'success': success,
            'results': results,
            'message': f"Found data in {len(results['summary']['databases_with_results'])} external databases" if success else "No results found in external databases"
        })
        
    except Exception as e:
        logger.error(f"❌ External database search error: {e}")
        return jsonify({
            'success': False,
            'error': f'External database search failed: {str(e)}',
            'results': {}
        }), 500

@app.route('/api/weather')
def api_weather():
    """API endpoint for weather data"""
    try:
        lat = request.args.get('lat', type=float)
        lon = request.args.get('lon', type=float)
        zip_code = request.args.get('zip')
        
        if not lat or not lon:
            if zip_code:
                # Try to get coordinates from zip code
                from weather_service import get_coordinates_from_zip_code
                location_data = get_coordinates_from_zip_code(zip_code, 'US')
                if location_data:
                    lat = location_data['latitude']
                    lon = location_data['longitude']
                else:
                    return jsonify({'error': 'Invalid zip code'}), 400
            else:
                return jsonify({'error': 'Latitude and longitude or zip code required'}), 400
        
        # Get current weather using existing weather service
        from weather_service import WeatherService
        weather = WeatherService.get_current_weather(lat, lon)
        
        if weather:
            response_data = {
                'success': True,
                'location': {
                    'latitude': lat,
                    'longitude': lon,
                    'name': weather.location_name or 'Unknown'
                },
                'weather': {
                    'temperature': weather.temperature,
                    'humidity': weather.humidity,
                    'wind_speed': weather.wind_speed,
                    'pressure': weather.pressure,
                    'weather_code': weather.weather_code,
                    'description': weather.description or 'No description',
                    'temperature_max': weather.temperature_max,
                    'temperature_min': weather.temperature_min,
                    'recorded_at': weather.recorded_at.isoformat() if weather.recorded_at else None
                }
            }
            return jsonify(response_data)
        else:
            return jsonify({'error': 'Unable to fetch weather data'}), 500
            
    except Exception as e:
        logger.error(f"Error in weather API: {e}")
        return jsonify({'error': 'Weather service unavailable'}), 500

@app.route('/compare')
def compare_page():
    """Orchid comparison page"""
    try:
        # Get orchid IDs from query parameters
        orchid1_id = request.args.get('orchid1', type=int)
        orchid2_id = request.args.get('orchid2', type=int)
        species = request.args.get('species', '')
        
        orchids = []
        
        # Get specific orchids with error handling
        if orchid1_id:
            try:
                orchid1 = OrchidRecord.query.get(orchid1_id)
                if orchid1:
                    orchids.append(orchid1)
            except Exception as e:
                logger.warning(f"Could not load orchid1 {orchid1_id}: {e}")
        
        if orchid2_id:
            try:
                orchid2 = OrchidRecord.query.get(orchid2_id)
                if orchid2:
                    orchids.append(orchid2)
            except Exception as e:
                logger.warning(f"Could not load orchid2 {orchid2_id}: {e}")
        
        # Get orchids by species if specified
        if species and not orchids:
            try:
                orchids = OrchidRecord.query.filter(
                    OrchidRecord.scientific_name.ilike(f'%{species}%')
                ).limit(10).all()
            except Exception as e:
                logger.warning(f"Could not search by species {species}: {e}")
        
        # Get some sample orchids if none specified
        if not orchids:
            try:
                sample_orchids = OrchidRecord.query.filter(
                    OrchidRecord.google_drive_id.isnot(None),
                    OrchidRecord.genus.isnot(None)
                ).limit(10).all()
                orchids = sample_orchids[:2] if len(sample_orchids) >= 2 else sample_orchids
            except Exception as e:
                logger.warning(f"Could not load sample orchids: {e}")
                orchids = []
        
        # Build data structure expected by template
        if orchids:
            species_name = orchids[0].scientific_name or orchids[0].genus or "Unknown"
            photographers = list(set([o.photographer for o in orchids if o.photographer]))
            climate_conditions = list(set([o.climate_preference for o in orchids if o.climate_preference]))
            growth_habits = list(set([o.growth_habit for o in orchids if o.growth_habit]))
            with_photos = [o for o in orchids if o.google_drive_id or o.image_url]
        else:
            species_name = "No Orchids"
            photographers = []
            climate_conditions = []
            growth_habits = []
            with_photos = []
        
        data = {
            'species_name': species_name,
            'specimen_count': len(orchids),
            'with_photos': with_photos,
            'photographers': photographers,
            'climate_conditions': climate_conditions,
            'growth_habits': growth_habits,
            'specimens': orchids  # Pass orchids as specimens too
        }
        
        return render_template('comparison.html', data=data, orchids=orchids)
        
    except Exception as e:
        logger.error(f"Error loading comparison page: {e}")
        return render_template('error.html', error="Could not load comparison page"), 500

# Hollywood Orchids Movie Widget - moved to widgets section
from hollywood_orchids_widget import hollywood_orchids
app.register_blueprint(hollywood_orchids, url_prefix='/widgets/hollywood-orchids')

# Register Orchid Trivia Challenge Widget
try:
    from orchid_trivia_widget import orchid_trivia
    app.register_blueprint(orchid_trivia, url_prefix='/widgets/orchid-trivia')
    logger.info("Orchid Trivia Challenge Widget registered successfully")
except ImportError as e:
    logger.warning(f"Orchid Trivia Widget not available: {e}")

# Register Orchid Mahjong Game Widget - TEMPORARILY DISABLED DUE TO BLUEPRINT CONFLICT
# try:
#     from orchid_mahjong_game import mahjong_bp
#     app.register_blueprint(mahjong_bp, url_prefix='/widgets/orchid-mahjong')
#     logger.info("Orchid Mahjong Game Widget registered successfully")
# except ImportError as e:
#     logger.warning(f"Orchid Mahjong Widget not available: {e}")

# Research Literature and Writing Lab Systems
from research_literature_system import research_literature
from research_writing_lab import writing_lab

app.register_blueprint(research_literature, url_prefix='/research-lab')
app.register_blueprint(writing_lab, url_prefix='/research-lab')

# Greek Mythology Orchids Widget
from greek_mythology_orchids import mythology_orchids
app.register_blueprint(mythology_orchids, url_prefix='/widgets/mythology-orchids')

# Register Bug Report System for Beta Testing
app.register_blueprint(bug_report_bp)
app.register_blueprint(data_dashboard_bp)

# Register Orchid Book Club
from orchid_book_club import orchid_book_club_bp
app.register_blueprint(orchid_book_club_bp, url_prefix='/book-club')

# Register Privacy-Focused Member System
from member_privacy_system import member_privacy_bp
app.register_blueprint(member_privacy_bp, url_prefix='/member-privacy')

# Bulk Orchid Analysis System
from bulk_orchid_analyzer import bulk_analyzer
app.register_blueprint(bulk_analyzer, url_prefix='/bulk-analyzer')

# Educational Games Integration
from educational_games_integration import educational_games
app.register_blueprint(gary_demo_bp)  # Gary's photo demo integration
app.register_blueprint(educational_games, url_prefix='/educational-games')

@app.route('/satellite-world-map')
def enhanced_satellite_world_map():
    """Enhanced satellite world map with space view"""
    try:
        # Get orchid count for the map display
        orchid_count = db.session.query(OrchidRecord).count()
        return render_template('mapping/enhanced_satellite_map.html', 
                             orchid_count=orchid_count)
    except Exception as e:
        logger.error(f"Satellite map error: {e}")
        return render_template('mapping/enhanced_satellite_map.html', 
                             orchid_count="230+")

# Demo Landing Pages for Different Audiences
@app.route('/demo/roberta-fox')
def demo_roberta_fox():
    """Partnership demonstration page for Roberta Fox"""
    return render_template('demo_roberta_fox.html')

@app.route('/demo/board-directors')
def demo_board_directors():
    """Board Directors funding/integration demonstration page"""
    return render_template('demo_board_directors.html')

@app.route('/orchid-community-resources')
def orchid_community_resources():
    """Comprehensive orchid community resources and links page"""
    return render_template('orchid_community_resources.html')

@app.route('/enhanced-science-lab')
def enhanced_science_lab():
    """Enhanced scientific method learning platform"""
    try:
        from scientific_research_platform import scientific_method
        return render_template('research/scientific_method_interface.html', 
                             stages=scientific_method.stages)
    except Exception as e:
        logger.error(f"Science lab error: {e}")
        return redirect(url_for('index'))

# =======================================================================
# ORCHID MAHJONG MULTIPLAYER & LEADERBOARD API ROUTES
# =======================================================================

@app.route('/api/mahjong/leaderboard/<period>')
def get_mahjong_leaderboard(period):
    """Get Mahjong leaderboard for specific time period"""
    try:
        from models import MahjongGame, MahjongPlayer, User
        
        # Calculate date range based on period
        now = datetime.now()
        if period == 'daily':
            start_date = now - timedelta(days=1)
        elif period == 'weekly':
            start_date = now - timedelta(weeks=1)
        elif period == 'monthly':
            start_date = now - timedelta(days=30)
        else:  # all-time
            start_date = datetime.min
        
        # Query for top players
        leaderboard = db.session.query(
            MahjongPlayer.user_id,
            func.sum(MahjongPlayer.score).label('total_score'),
            func.count(MahjongPlayer.id).label('games_played'),
            func.avg(MahjongPlayer.tiles_matched).label('avg_tiles'),
            func.sum(MahjongGame.game_duration).label('total_time')
        ).join(
            MahjongGame, MahjongPlayer.game_id == MahjongGame.id
        ).filter(
            MahjongGame.finished_at >= start_date,
            MahjongGame.game_state == 'finished'
        ).group_by(
            MahjongPlayer.user_id
        ).order_by(
            func.sum(MahjongPlayer.score).desc()
        ).limit(50).all()
        
        # Format response with mock usernames for now
        result = []
        for i, entry in enumerate(leaderboard):
            result.append({
                'rank': i + 1,
                'name': f'Player_{entry.user_id[-3:]}',  # Use last 3 chars of user_id
                'score': int(entry.total_score or 0),
                'time': f"{int((entry.total_time or 0) / 60):02d}:{int((entry.total_time or 0) % 60):02d}",
                'moves': int(entry.avg_tiles or 0),
                'games_played': int(entry.games_played)
            })
        
        # If no real data, return mock leaderboard
        if not result:
            mock_data = [
                {'rank': 1, 'name': 'OrchidMaster', 'score': 15420, 'time': '02:34', 'moves': 72, 'games_played': 12},
                {'rank': 2, 'name': 'FlowerPower', 'score': 14850, 'time': '03:12', 'moves': 85, 'games_played': 8},
                {'rank': 3, 'name': 'MahjongPro', 'score': 13960, 'time': '02:56', 'moves': 78, 'games_played': 15},
                {'rank': 4, 'name': 'PetalSeeker', 'score': 12340, 'time': '04:21', 'moves': 92, 'games_played': 6},
                {'rank': 5, 'name': 'BloomBuster', 'score': 11780, 'time': '03:45', 'moves': 89, 'games_played': 9}
            ]
            return jsonify(mock_data)
        
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"Leaderboard error: {e}")
        # Return mock data on error
        mock_data = [
            {'rank': 1, 'name': 'OrchidMaster', 'score': 15420, 'time': '02:34', 'moves': 72, 'games_played': 12},
            {'rank': 2, 'name': 'FlowerPower', 'score': 14850, 'time': '03:12', 'moves': 85, 'games_played': 8},
            {'rank': 3, 'name': 'MahjongPro', 'score': 13960, 'time': '02:56', 'moves': 78, 'games_played': 15}
        ]
        return jsonify(mock_data)

@app.route('/api/mahjong/create-room', methods=['POST'])
def create_mahjong_room():
    """Create a new multiplayer Mahjong room"""
    try:
        from models import MahjongGame, MahjongPlayer
        import secrets
        
        data = request.get_json()
        player_name = data.get('player_name', f'Player_{secrets.randbelow(999):03d}')
        room_code = secrets.token_hex(3).upper()
        
        # Create mock user session if needed
        if 'user_id' not in session:
            session['user_id'] = f'user_{secrets.token_hex(8)}'
            session['player_name'] = player_name
        
        # Create game room
        game = MahjongGame()
        game.room_code = room_code
        game.host_user_id = session['user_id']
        game.max_players = 4
        game.current_players = 1
        game.game_state = 'waiting'
        db.session.add(game)
        db.session.flush()  # Get the game ID
        
        # Add host as first player
        player = MahjongPlayer()
        player.game_id = game.id
        player.user_id = session['user_id']
        player.player_position = 1
        db.session.add(player)
        db.session.commit()
        
        return jsonify({
            'success': True,
            'room_code': room_code,
            'game_id': game.id,
            'player_name': player_name,
            'is_host': True
        })
        
    except Exception as e:
        logger.error(f"Create room error: {e}")
        # Return mock room for demo
        room_code = secrets.token_hex(3).upper()
        return jsonify({
            'success': True,
            'room_code': room_code,
            'game_id': 'demo',
            'player_name': data.get('player_name', 'DemoPlayer'),
            'is_host': True
        })

@app.route('/api/mahjong/join-room', methods=['POST'])
def join_mahjong_room():
    """Join an existing Mahjong room"""
    try:
        from models import MahjongGame, MahjongPlayer
        import secrets
        
        data = request.get_json()
        room_code = data.get('room_code', '').upper()
        player_name = data.get('player_name', f'Player_{secrets.randbelow(999):03d}')
        
        # Create mock user session if needed
        if 'user_id' not in session:
            session['user_id'] = f'user_{secrets.token_hex(8)}'
            session['player_name'] = player_name
        
        # Find the game room
        game = MahjongGame.query.filter_by(room_code=room_code, game_state='waiting').first()
        
        if not game:
            return jsonify({'success': False, 'error': 'Room not found or game already started'})
        
        if game.current_players >= game.max_players:
            return jsonify({'success': False, 'error': 'Room is full'})
        
        # Add player to room
        player = MahjongPlayer()
        player.game_id = game.id
        player.user_id = session['user_id']
        player.player_position = game.current_players + 1
        db.session.add(player)
        
        # Update player count
        game.current_players += 1
        db.session.commit()
        
        # Get all players in room
        players = MahjongPlayer.query.filter_by(game_id=game.id).all()
        player_list = [{'name': f'Player_{p.user_id[-3:]}', 'position': p.player_position} for p in players]
        
        return jsonify({
            'success': True,
            'room_code': room_code,
            'game_id': game.id,
            'player_name': player_name,
            'is_host': False,
            'players': player_list,
            'player_count': game.current_players
        })
        
    except Exception as e:
        logger.error(f"Join room error: {e}")
        # Return mock response for demo
        return jsonify({
            'success': True,
            'room_code': room_code,
            'game_id': 'demo',
            'player_name': player_name,
            'is_host': False,
            'players': [{'name': 'HostPlayer', 'position': 1}, {'name': player_name, 'position': 2}],
            'player_count': 2
        })

@app.route('/api/mahjong/start-game', methods=['POST'])
def start_mahjong_game():
    """Start a multiplayer Mahjong game"""
    try:
        from models import MahjongGame
        
        data = request.get_json()
        game_id = data.get('game_id')
        
        if game_id == 'demo':
            return jsonify({'success': True, 'message': 'Demo game started!'})
        
        game = MahjongGame.query.get(game_id)
        if not game or game.host_user_id != session.get('user_id'):
            return jsonify({'success': False, 'error': 'Not authorized to start this game'})
        
        # Start the game
        game.game_state = 'playing'
        db.session.commit()
        
        return jsonify({'success': True, 'message': 'Game started!'})
        
    except Exception as e:
        logger.error(f"Start game error: {e}")
        return jsonify({'success': True, 'message': 'Game started!'})

@app.route('/api/mahjong/submit-score', methods=['POST'])
def submit_mahjong_score():
    """Submit a game score to the leaderboard"""
    try:
        from models import MahjongGame, MahjongPlayer
        import secrets
        
        data = request.get_json()
        score = data.get('score', 0)
        time_seconds = data.get('time', 0)
        moves = data.get('moves', 0)
        
        # Create mock user if needed
        if 'user_id' not in session:
            session['user_id'] = f'user_{secrets.token_hex(8)}'
        
        # For demo purposes, we'll create a single-player "game"
        game = MahjongGame()
        game.room_code = f'SOLO_{secrets.randbelow(999):03d}'
        game.host_user_id = session['user_id']
        game.max_players = 1
        game.current_players = 1
        game.game_state = 'finished'
        game.game_duration = time_seconds
        game.finished_at = datetime.now()
        db.session.add(game)
        db.session.flush()
        
        # Add player score
        player = MahjongPlayer()
        player.game_id = game.id
        player.user_id = session['user_id']
        player.player_position = 1
        player.score = score
        player.tiles_matched = moves
        db.session.add(player)
        db.session.commit()
        
        return jsonify({'success': True, 'message': 'Score submitted successfully!'})
        
    except Exception as e:
        logger.error(f"Submit score error: {e}")
        return jsonify({'success': True, 'message': 'Score submitted successfully!'})

@app.route('/api/mahjong/room-status/<room_code>')
def get_room_status(room_code):
    """Get current status of a game room"""
    try:
        from models import MahjongGame, MahjongPlayer
        
        game = MahjongGame.query.filter_by(room_code=room_code.upper()).first()
        
        if not game:
            return jsonify({'success': False, 'error': 'Room not found'})
        
        players = MahjongPlayer.query.filter_by(game_id=game.id).all()
        player_list = [{'name': f'Player_{p.user_id[-3:]}', 'position': p.player_position} for p in players]
        
        return jsonify({
            'success': True,
            'room_code': room_code.upper(),
            'game_state': game.game_state,
            'current_players': game.current_players,
            'max_players': game.max_players,
            'players': player_list
        })
        
    except Exception as e:
        logger.error(f"Room status error: {e}")
        # Return mock status for demo
        return jsonify({
            'success': True,
            'room_code': room_code.upper(),
            'game_state': 'waiting',
            'current_players': 1,
            'max_players': 4,
            'players': [{'name': 'DemoPlayer', 'position': 1}]
        })

# AI Batch Processing Routes
@app.route('/admin/start-ai-batch', methods=['POST'])
def start_ai_batch_processing():
    """Start AI batch processing of orchid images"""
    try:
        limit = request.json.get('limit') if request.json else None
        result = ai_batch_processor.start_batch_analysis(limit=limit)
        return jsonify(result)
    except Exception as e:
        logger.error(f"Error starting AI batch processing: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/ai-batch-progress')
def get_ai_batch_progress():
    """Get current AI batch processing progress"""
    try:
        progress = ai_batch_processor.get_progress()
        return jsonify({
            'success': True,
            'progress': progress
        })
    except Exception as e:
        logger.error(f"Error getting AI batch progress: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/stop-ai-batch', methods=['POST'])
def stop_ai_batch_processing():
    """Stop AI batch processing"""
    try:
        result = ai_batch_processor.stop_processing()
        return jsonify(result)
    except Exception as e:
        logger.error(f"Error stopping AI batch processing: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@app.route('/admin/ai-dashboard')
def ai_dashboard():
    """AI Processing Dashboard"""
    return render_template('admin/ai_dashboard.html')

@app.route('/api/database-stats')
def database_stats():
    """Get comprehensive database statistics"""
    try:
        total_orchids = db.session.query(OrchidRecord).count()
        with_ai_analysis = db.session.query(OrchidRecord).filter(OrchidRecord.ai_description.isnot(None)).count()
        with_images = db.session.query(OrchidRecord).filter(
            or_(
                OrchidRecord.google_drive_id.isnot(None),
                OrchidRecord.image_url.isnot(None),
                OrchidRecord.image_filename.isnot(None)
            )
        ).count()
        needing_ai = db.session.query(OrchidRecord).filter(
            and_(
                or_(
                    OrchidRecord.google_drive_id.isnot(None),
                    OrchidRecord.image_url.isnot(None),
                    OrchidRecord.image_filename.isnot(None)
                ),
                OrchidRecord.ai_description.is_(None)
            )
        ).count()
        
        return jsonify({
            'total_orchids': total_orchids,
            'with_ai_analysis': with_ai_analysis,
            'with_images': with_images,
            'needing_ai': needing_ai,
            'last_updated': datetime.now().isoformat()
        })
    except Exception as e:
        logger.error(f"Error getting database stats: {e}")
        return jsonify({'error': str(e)}), 500

@app.route('/api/mahjong/orchid-facts')
def get_orchid_facts():
    """Get educational facts about orchids for pop-up cards"""
    try:
        # Get random orchid from database
        orchid = OrchidRecord.query.order_by(func.random()).first()
        
        if orchid:
            fact = {
                'name': orchid.scientific_name or orchid.display_name or 'Unknown Orchid',
                'fact': orchid.ai_description or orchid.cultural_notes or 'This beautiful orchid is part of our collection.',
                'family': 'Orchidaceae',
                'origin': orchid.native_habitat or 'Various tropical regions',
                'care_tip': orchid.cultural_notes or 'Orchids prefer bright, indirect light and good air circulation.',
                'image_url': orchid.image_url or '/static/images/orchid_placeholder.svg'
            }
        else:
            # Fallback educational facts
            facts_pool = [
                {
                    'name': 'Orchid Family',
                    'fact': 'Orchidaceae is one of the largest plant families with over 25,000 species worldwide!',
                    'family': 'Orchidaceae',
                    'origin': 'Global distribution',
                    'care_tip': 'Most orchids prefer bright, indirect light and well-draining growing medium.',
                    'image_url': '/static/images/orchid_placeholder.svg'
                },
                {
                    'name': 'Vanilla Orchid',
                    'fact': 'Vanilla flavoring comes from the seed pods of Vanilla planifolia, a climbing orchid!',
                    'family': 'Orchidaceae',
                    'origin': 'Mexico and Central America',
                    'care_tip': 'Vanilla orchids need support to climb and prefer warm, humid conditions.',
                    'image_url': '/static/images/orchid_placeholder.svg'
                },
                {
                    'name': 'Epiphytic Orchids',
                    'fact': 'About 70% of orchids are epiphytes, growing on other plants without harming them.',
                    'family': 'Orchidaceae',
                    'origin': 'Tropical rainforests',
                    'care_tip': 'Epiphytic orchids need excellent drainage and should never sit in standing water.',
                    'image_url': '/static/images/orchid_placeholder.svg'
                }
            ]
            import random
            fact = random.choice(facts_pool)
        
        return jsonify({
            'success': True,
            'orchid_fact': fact
        })
        
    except Exception as e:
        logger.error(f"Orchid facts error: {e}")
        return jsonify({
            'success': True,
            'orchid_fact': {
                'name': 'Amazing Orchids',
                'fact': 'Orchids are fascinating plants that have adapted to nearly every environment on Earth!',
                'family': 'Orchidaceae',
                'origin': 'Worldwide',
                'care_tip': 'Each orchid species has unique care requirements - research your specific type!',
                'image_url': '/static/images/orchid_placeholder.svg'
            }
        })

# Unidentified Orchids Community Identification System
@app.route('/unidentified-orchids')
def unidentified_orchids():
    """Community-driven orchid identification gallery"""
    try:
        # Get unidentified orchids with their vote data
        unidentified = db.session.query(OrchidRecord).filter(
            OrchidRecord.identification_status == 'unidentified'
        ).all()
        
        # Add image URLs and expert votes
        for orchid in unidentified:
            if orchid.google_drive_id:
                orchid.image_url = f'/api/drive-photo/{orchid.google_drive_id}'
            else:
                orchid.image_url = '/static/images/orchid_placeholder.svg'
            
            # Get expert votes/notes for this orchid
            try:
                from sqlalchemy import text
                orchid.expert_votes = db.session.execute(
                    text("SELECT notes FROM identification_votes WHERE orchid_id = :id AND notes IS NOT NULL"),
                    {"id": orchid.id}
                ).fetchall()
            except:
                orchid.expert_votes = []
        
        # Get statistics
        stats = {
            'unidentified_count': len(unidentified),
            'total_votes': db.session.execute(
                text("SELECT COALESCE(SUM(identification_votes), 0) FROM orchid_record WHERE identification_status = 'unidentified'")
            ).scalar() or 0,
            'pending_approval': db.session.query(OrchidRecord).filter(
                OrchidRecord.identification_status == 'unidentified',
                OrchidRecord.identification_votes == 1
            ).count()
        }
        
        return render_template('unidentified_orchids.html', 
                             unidentified_orchids=unidentified, 
                             stats=stats)
    except Exception as e:
        logger.error(f"Error loading unidentified orchids: {e}")
        return render_template('error.html', error="Error loading identification gallery")

@app.route('/api/orchid/<int:orchid_id>/vote-agree', methods=['POST'])
def vote_agree_identification(orchid_id):
    """Vote to agree with current suggested identification"""
    try:
        orchid = OrchidRecord.query.get_or_404(orchid_id)
        
        if orchid.identification_status != 'unidentified':
            return jsonify({'success': False, 'message': 'Orchid is not awaiting identification'})
        
        if not orchid.suggested_genus:
            return jsonify({'success': False, 'message': 'No suggested identification to vote on'})
        
        # Get voter info (IP for tracking)
        voter_ip = request.environ.get('HTTP_X_FORWARDED_FOR', request.remote_addr)
        
        # Check if already voted
        from sqlalchemy import text
        existing_vote = db.session.execute(
            text("SELECT id FROM identification_votes WHERE orchid_id = :oid AND voter_ip = :ip"),
            {"oid": orchid_id, "ip": voter_ip}
        ).fetchone()
        
        if existing_vote:
            return jsonify({'success': False, 'message': 'You have already voted on this orchid'})
        
        # Record the vote
        db.session.execute(
            text("""INSERT INTO identification_votes 
                     (orchid_id, voter_ip, suggested_genus, suggested_species, confidence_level, notes)
                     VALUES (:oid, :ip, :genus, :species, 8, 'Agreed with community suggestion')"""),
            {
                "oid": orchid_id,
                "ip": voter_ip,
                "genus": orchid.suggested_genus,
                "species": orchid.suggested_species or '',
            }
        )
        
        # Increment vote count
        orchid.identification_votes += 1
        
        # Check if we have enough votes (2+) to verify
        if orchid.identification_votes >= 2:
            orchid.identification_status = 'verified'
            orchid.genus = orchid.suggested_genus
            orchid.species = orchid.suggested_species
            
            # Update display name
            if orchid.suggested_species:
                orchid.display_name = f"{orchid.suggested_genus} {orchid.suggested_species}"
            else:
                orchid.display_name = orchid.suggested_genus
            
            db.session.commit()
            
            return jsonify({
                'success': True, 
                'message': 'Vote recorded! This orchid has been verified and moved back to the main gallery.',
                'verified': True
            })
        
        db.session.commit()
        return jsonify({'success': True, 'message': 'Vote recorded! One more vote needed for verification.'})
        
    except Exception as e:
        logger.error(f"Error recording agreement vote: {e}")
        return jsonify({'success': False, 'message': 'Error recording vote'})

@app.route('/api/orchid/<int:orchid_id>/suggest-id', methods=['POST'])
def suggest_identification(orchid_id):
    """Submit a new identification suggestion"""
    try:
        orchid = OrchidRecord.query.get_or_404(orchid_id)
        data = request.get_json()
        
        genus = data.get('genus', '').strip()
        species = data.get('species', '').strip()
        notes = data.get('notes', '').strip()
        
        if not genus:
            return jsonify({'success': False, 'message': 'Genus is required'})
        
        # Get voter info
        voter_ip = request.environ.get('HTTP_X_FORWARDED_FOR', request.remote_addr)
        
        # Check if already voted
        from sqlalchemy import text
        existing_vote = db.session.execute(
            text("SELECT id FROM identification_votes WHERE orchid_id = :oid AND voter_ip = :ip"),
            {"oid": orchid_id, "ip": voter_ip}
        ).fetchone()
        
        if existing_vote:
            return jsonify({'success': False, 'message': 'You have already voted on this orchid'})
        
        # Record the suggestion
        db.session.execute(
            text("""INSERT INTO identification_votes 
                     (orchid_id, voter_ip, suggested_genus, suggested_species, confidence_level, notes)
                     VALUES (:oid, :ip, :genus, :species, 7, :notes)"""),
            {
                "oid": orchid_id,
                "ip": voter_ip,
                "genus": genus,
                "species": species,
                "notes": notes or f"Suggested {genus} {species}".strip()
            }
        )
        
        # Update orchid record with new suggestion if it's the first vote
        if orchid.identification_votes == 0:
            orchid.suggested_genus = genus
            orchid.suggested_species = species
        
        orchid.identification_votes += 1
        db.session.commit()
        
        return jsonify({'success': True, 'message': 'Suggestion submitted! Thank you for contributing.'})
        
    except Exception as e:
        logger.error(f"Error recording identification suggestion: {e}")
        return jsonify({'success': False, 'message': 'Error submitting suggestion'})

@app.route('/api/orchid/<int:orchid_id>/report-mislabeled', methods=['POST'])
def report_mislabeled(orchid_id):
    """Report an orchid as mislabeled and move to unidentified section"""
    try:
        orchid = OrchidRecord.query.get_or_404(orchid_id)
        data = request.get_json()
        
        reason = data.get('reason', '').strip()
        suggested_genus = data.get('suggested_genus', '').strip()
        suggested_species = data.get('suggested_species', '').strip()
        
        # Move to unidentified status
        orchid.identification_status = 'unidentified'
        orchid.identification_votes = 0
        
        if suggested_genus:
            orchid.suggested_genus = suggested_genus
            orchid.suggested_species = suggested_species
            orchid.identification_votes = 1
            
            # Record the initial expert vote
            voter_ip = request.environ.get('HTTP_X_FORWARDED_FOR', request.remote_addr)
            from sqlalchemy import text
            db.session.execute(
                text("""INSERT INTO identification_votes 
                         (orchid_id, voter_ip, suggested_genus, suggested_species, confidence_level, notes)
                         VALUES (:oid, :ip, :genus, :species, 9, :notes)"""),
                {
                    "oid": orchid_id,
                    "ip": voter_ip,
                    "genus": suggested_genus,
                    "species": suggested_species or '',
                    "notes": f"Mislabeling report: {reason}"
                }
            )
        
        db.session.commit()
        
        return jsonify({
            'success': True, 
            'message': 'Report submitted! This orchid has been moved to the identification gallery.'
        })
        
    except Exception as e:
        logger.error(f"Error reporting mislabeled orchid: {e}")
        return jsonify({'success': False, 'message': 'Error submitting report'})

@app.route('/api/investigate-trichocentrum', methods=['POST'])
def investigate_trichocentrum():
    """AI investigation of all Trichocentrum records for potential mislabeling"""
    try:
        # Get all Trichocentrum records with images
        trichocentrum_orchids = db.session.query(OrchidRecord).filter(
            OrchidRecord.genus == 'Trichocentrum',
            OrchidRecord.google_drive_id.isnot(None),
            OrchidRecord.identification_status != 'unidentified'
        ).all()
        
        results = {
            'total_investigated': len(trichocentrum_orchids),
            'moved_to_unidentified': 0,
            'confirmed_correct': 0,
            'ai_analysis_failed': 0,
            'details': []
        }
        
        for orchid in trichocentrum_orchids:
            try:
                # Use AI to analyze the orchid image
                image_url = f'/api/drive-photo/{orchid.google_drive_id}'
                
                # Get AI analysis with specific focus on genus identification
                analysis_prompt = f"""
                Analyze this orchid image and determine if it is correctly identified as genus Trichocentrum.
                
                Current label: {orchid.display_name}
                Current genus: {orchid.genus}
                
                Trichocentrum characteristics to look for:
                - Small to medium-sized flowers
                - Distinctive lip shape with curved callus
                - Usually yellow/brown coloration with spots
                - Pseudobulbs with 1-2 leaves
                - Flowers arranged in panicles or racemes
                
                If this is NOT a Trichocentrum, suggest the correct genus.
                
                Response format:
                Correct_genus: [Yes/No - if labeled correctly as Trichocentrum]
                Suggested_genus: [If incorrect, suggest correct genus]
                Confidence: [1-10]
                Reason: [Brief explanation]
                """
                
                ai_result = analyze_orchid_image(image_url, analysis_prompt)
                
                if ai_result and 'Correct_genus: No' in ai_result:
                    # Extract suggested genus from AI response
                    suggested_genus = None
                    for line in ai_result.split('\n'):
                        if 'Suggested_genus:' in line:
                            suggested_genus = line.split(':', 1)[1].strip()
                            break
                    
                    if suggested_genus and suggested_genus != 'Trichocentrum':
                        # Move to unidentified section
                        orchid.identification_status = 'unidentified'
                        orchid.suggested_genus = suggested_genus
                        orchid.identification_votes = 1
                        
                        # Record AI analysis as expert vote
                        from sqlalchemy import text
                        db.session.execute(
                            text("""INSERT INTO identification_votes 
                                     (orchid_id, voter_ip, suggested_genus, confidence_level, notes)
                                     VALUES (:oid, 'ai_investigation', :genus, 9, :notes)"""),
                            {
                                "oid": orchid.id,
                                "genus": suggested_genus,
                                "notes": f"AI Investigation: {ai_result}"
                            }
                        )
                        
                        results['moved_to_unidentified'] += 1
                        results['details'].append({
                            'id': orchid.id,
                            'name': orchid.display_name,
                            'action': 'moved_to_unidentified',
                            'suggested_genus': suggested_genus,
                            'ai_analysis': ai_result
                        })
                    else:
                        results['confirmed_correct'] += 1
                        results['details'].append({
                            'id': orchid.id,
                            'name': orchid.display_name,
                            'action': 'confirmed_correct'
                        })
                else:
                    results['confirmed_correct'] += 1
                    results['details'].append({
                        'id': orchid.id,
                        'name': orchid.display_name,
                        'action': 'confirmed_correct'
                    })
                    
            except Exception as ai_error:
                logger.error(f"AI analysis failed for orchid {orchid.id}: {ai_error}")
                results['ai_analysis_failed'] += 1
                results['details'].append({
                    'id': orchid.id,
                    'name': orchid.display_name,
                    'action': 'ai_failed',
                    'error': str(ai_error)
                })
        
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': f'AI investigation complete! {results["moved_to_unidentified"]} orchids moved to unidentified section.',
            'results': results
        })
        
    except Exception as e:
        logger.error(f"Error during Trichocentrum investigation: {e}")
        return jsonify({'success': False, 'message': f'Investigation failed: {str(e)}'})

# Navigation link update
@app.route('/admin/add-unidentified-link')
def add_unidentified_navigation():
    """Add link to unidentified orchids section in navigation"""
    return jsonify({'success': True, 'message': 'Navigation updated to include Help Us ID section'})

# Register genetics laboratory
register_genetics_laboratory(app)

# Register quantum care routes
register_quantum_care_routes(app)

@app.route("/lab/ga3_sarcochilus")
def ga3_sarcochilus():
    return render_template("lab/ga3_sarcochilus.html")

# Register FCOS Judge PWA routes
from routes_fcos_judge import register_fcos_judge_routes
register_fcos_judge_routes(app)

# Register Processing Monitor Routes
try:
    from processing_monitor_routes import register_processing_monitor_routes
    register_processing_monitor_routes(app)
    logger.info("🔍 Processing Pipeline Monitor registered successfully")
except ImportError as e:
    logger.warning(f"Processing monitor not available: {e}")

# Register AI Orchid Chat Interface
try:
    from ai_orchid_chat import register_ai_chat_routes
    register_ai_chat_routes(app)
    logger.info("🤖 AI Orchid Chat interface registered successfully")
except ImportError as e:
    logger.warning(f"AI chat not available: {e}")

# Register Care Helper Widget
try:
    from care_helper_widget import register_care_helper_routes
    register_care_helper_routes(app)
    logger.info("🌺 Care Helper widget registered successfully")
except ImportError as e:
    logger.warning(f"Care Helper widget not available: {e}")

# Register Guided Care Form
try:
    from guided_care_form import register_guided_care_routes
    register_guided_care_routes(app)
    logger.info("🌺 Guided Care Form registered successfully")
except ImportError as e:
    logger.warning(f"Guided Care Form not available: {e}")

# Register Orchid Care Widget Package
try:
    from orchid_care_widget_package import register_widget_package_routes
    register_widget_package_routes(app)
    logger.info("🎨 Orchid Care Widget Package registered successfully")
except ImportError as e:
    logger.warning(f"Widget Package not available: {e}")

# Register AI Collection Manager
try:
    from ai_collection_manager import register_collection_manager_routes
    register_collection_manager_routes(app)
    logger.info("🤖 AI Collection Manager registered successfully")
except ImportError as e:
    logger.warning(f"AI Collection Manager not available: {e}")

# Register AI Collection Onboarding
try:
    from ai_collection_onboarding import register_onboarding_routes
    register_onboarding_routes(app)
    logger.info("🎯 AI Collection Onboarding registered successfully")
except ImportError as e:
    logger.warning(f"AI Collection Onboarding not available: {e}")

# =============================================================================
# NEW HUB ROUTES FOR WIDGET CONSOLIDATION SYSTEM
# =============================================================================

@app.route('/geo')
def geo_explorer_hub():
    """GeoExplorer Hub - Unified geographic exploration"""
    return render_template('geo_explorer_hub.html')

@app.route('/gallery-hub')
def gallery_hub():
    """Gallery Hub - Comprehensive orchid image browsing"""
    return render_template('gallery_hub.html')

@app.route('/education')
def education_hub():
    """Education Hub - Glossary, crosswords, flashcards, and learning tools"""
    return render_template('education_hub.html')

@app.route('/ai-tools')
def ai_tools_bundle():
    """AI Tools Bundle - Breeding, identification, and analysis"""
    return render_template('ai_tools_bundle.html')

@app.route('/ai-research-assistant')
def ai_research_assistant():
    """AI Research Assistant - Advanced orchid research interface with species identification, cultivation guidance, and academic citations"""
    try:
        # Get available AI research capabilities
        capabilities = {
            'identification': True,
            'cultivation': True,
            'research': True,
            'citations': True,
            'session_management': True,
            'image_upload': True
        }
        
        # Initialize session if needed
        if 'ai_research_session' not in session:
            session['ai_research_session'] = {
                'started': datetime.now().isoformat(),
                'query_count': 0
            }
        
        return render_template('ai_research_assistant.html', 
                             capabilities=capabilities,
                             session_info=session.get('ai_research_session'))
    except Exception as e:
        logger.error(f"Error loading AI Research Assistant: {e}")
        return render_template('error.html', 
                             error_message="AI Research Assistant temporarily unavailable. Please try again later."), 500

@app.route('/collection')
def my_collection_hub():
    """My Collection Hub - Photo management, editing, and publishing"""
    return render_template('my_collection_hub.html')

@app.route('/pest-diseases')
def pest_diseases_hub():
    """Pest & Diseases Hub - Plant health management"""
    return render_template('pest_diseases_hub.html')

@app.route('/philosophy')
def orchid_philosophy_hub():
    """Orchid Philosophy & Culture Hub - Deeper meaning and cultural significance"""
    return render_template('orchid_philosophy_hub.html')

# =============================================================================
# EMBED ENDPOINTS FOR NEON ONE INTEGRATION
# =============================================================================

@app.route('/embed/geo')
def embed_geo():
    """Embeddable GeoExplorer widget"""
    mode = request.args.get('mode', 'map')
    theme = request.args.get('theme', 'default')
    return render_template('geo_explorer_hub.html', 
                         embed_mode=True, 
                         default_mode=mode,
                         theme=theme)

@app.route('/embed/gallery')
@app.route('/embed/gallery-hub')
def embed_gallery():
    """Embeddable Gallery widget"""
    tab = request.args.get('tab', 'browse')
    return render_template('gallery_hub.html', 
                         embed_mode=True, 
                         default_tab=tab)

@app.route('/embed/education')
def embed_education():
    """Embeddable Education widget"""
    tab = request.args.get('tab', 'glossary')
    difficulty = request.args.get('difficulty', 'beginner')
    return render_template('education_hub.html', 
                         embed_mode=True, 
                         default_tab=tab,
                         default_difficulty=difficulty)

@app.route('/embed/ai-tools')
def embed_ai_tools():
    """Embeddable AI Tools widget"""
    tab = request.args.get('tab', 'breeding')
    return render_template('ai_tools_bundle.html', 
                         embed_mode=True, 
                         default_tab=tab)

@app.route('/embed/collection')
def embed_collection():
    """Embeddable Collection widget"""
    tab = request.args.get('tab', 'photos')
    contest = request.args.get('contest', 'false') == 'true'
    member_id = request.args.get('member_id')
    member_name = request.args.get('member_name')
    return render_template('my_collection_hub.html', 
                         embed_mode=True, 
                         default_tab=tab,
                         contest_mode=contest,
                         member_id=member_id,
                         member_name=member_name)

@app.route('/embed/pest-diseases')
def embed_pest_diseases():
    """Embeddable Pest & Diseases widget"""
    tab = request.args.get('tab', 'identify')
    return render_template('pest_diseases_hub.html', 
                         embed_mode=True, 
                         default_tab=tab)

@app.route('/embed/philosophy')
def embed_philosophy():
    """Embeddable Philosophy widget"""
    tab = request.args.get('tab', 'philosophy')
    return render_template('orchid_philosophy_hub.html', 
                         embed_mode=True, 
                         default_tab=tab)

@app.route('/embed/sdk.js')
def embed_sdk():
    """JavaScript SDK for iframe postMessage integration"""
    sdk_content = '''
// Orchid Continuum Embed SDK v1.0
(function() {
    'use strict';
    
    window.OrchidContinuumSDK = {
        version: '1.0.0',
        
        // Initialize message listeners
        init: function(config) {
            this.config = {
                origin: config.origin || window.location.origin,
                onPhotoSubmission: config.onPhotoSubmission || null,
                onContestEntry: config.onContestEntry || null,
                onMemberActivity: config.onMemberActivity || null,
                allowedOrigins: config.allowedOrigins || ['https://neon.one', 'https://app.neon.one']
            };
            
            window.addEventListener('message', this.handleMessage.bind(this));
        },
        
        // Handle postMessage events from embedded widgets
        handleMessage: function(event) {
            // Security: Check origin
            const allowed = this.config.allowedOrigins.some(origin => 
                event.origin.includes(origin) || event.origin === this.config.origin
            );
            
            if (!allowed) {
                console.warn('OrchidSDK: Message from unauthorized origin:', event.origin);
                return;
            }
            
            const data = event.data;
            if (!data.type || !data.type.startsWith('orchid_')) return;
            
            switch(data.type) {
                case 'orchid_photo_submission':
                    if (this.config.onPhotoSubmission) {
                        this.config.onPhotoSubmission(data);
                    }
                    break;
                    
                case 'orchid_contest_submission':
                    if (this.config.onContestEntry) {
                        this.config.onContestEntry(data);
                    }
                    break;
                    
                case 'orchid_member_activity':
                    if (this.config.onMemberActivity) {
                        this.config.onMemberActivity(data);
                    }
                    break;
                    
                case 'orchid_widget_loaded':
                    console.log('OrchidSDK: Widget loaded:', data.widget);
                    break;
                    
                case 'orchid_widget_error':
                    console.error('OrchidSDK: Widget error:', data.error);
                    break;
            }
        },
        
        // Send message to embedded widget
        sendToWidget: function(iframe, message) {
            if (iframe && iframe.contentWindow) {
                iframe.contentWindow.postMessage(message, '*');
            }
        }
    };
})();
'''
    
    response = Response(sdk_content, mimetype='application/javascript')
    response.headers['Cache-Control'] = 'public, max-age=3600'
    return response

# =============================================================================
# SECURITY MIDDLEWARE FOR EMBED ENDPOINTS
# =============================================================================

@app.after_request
def add_security_headers(response):
    """Add security headers for embedding"""
    # Allow embedding from trusted domains
    allowed_origins = [
        'https://neon.one',
        'https://app.neon.one', 
        'https://orchidcontinuum.com',
        'https://*.orchidcontinuum.com'
    ]
    
    # Set X-Frame-Options for embedding
    if request.path.startswith('/embed/'):
        response.headers['X-Frame-Options'] = 'ALLOWALL'
        response.headers['Content-Security-Policy'] = "frame-ancestors 'self' https://neon.one https://app.neon.one https://orchidcontinuum.com https://*.orchidcontinuum.com;"
    else:
        response.headers['X-Frame-Options'] = 'SAMEORIGIN'
    
    return response

# =============================================================================
# API ENDPOINTS FOR HUB FUNCTIONALITY  
# =============================================================================

@app.route('/api/hub/education/glossary')
def api_education_glossary():
    """API endpoint for glossary terms"""
    try:
        from aos_glossary_extractor import OrchidGlossaryTerm
        
        category = request.args.get('category', 'all')
        search = request.args.get('search', '')
        limit = int(request.args.get('limit', 50))
        
        query = OrchidGlossaryTerm.query
        
        if category != 'all':
            query = query.filter(OrchidGlossaryTerm.category == category)
            
        if search:
            query = query.filter(or_(
                OrchidGlossaryTerm.term.ilike(f'%{search}%'),
                OrchidGlossaryTerm.definition.ilike(f'%{search}%')
            ))
            
        terms = query.limit(limit).all()
        
        return jsonify({
            'success': True,
            'terms': [{
                'term': t.term,
                'definition': t.definition,
                'category': t.category,
                'difficulty': t.difficulty
            } for t in terms]
        })
        
    except Exception as e:
        logger.error(f"Error fetching glossary terms: {e}")
        return jsonify({'success': False, 'error': str(e)})

@app.route('/api/hub/education/crossword', methods=['GET', 'POST'])
def api_education_crossword():
    """API endpoint for crossword puzzle generation"""
    try:
        from crossword_generator import CrosswordGenerator
        
        difficulty = request.args.get('difficulty', 'beginner')
        size = request.args.get('size', 'medium')
        
        generator = CrosswordGenerator()
        puzzle = generator.generate_crossword(difficulty=difficulty, size=size)
        
        return jsonify({
            'success': True,
            'puzzle': puzzle
        })
        
    except Exception as e:
        logger.error(f"Error generating crossword: {e}")
        return jsonify({'success': False, 'error': str(e)})

@app.route('/api/hub/collection/contest-submit', methods=['POST'])
def api_collection_contest_submit():
    """API endpoint for contest photo submissions"""
    try:
        data = request.get_json()
        
        # Use existing monthly contest system
        from monthly_contest_system import monthly_contest
        
        result = monthly_contest.submit_entry(
            member_id=data.get('member_id'),
            entry_data=data
        )
        
        # Send postMessage event for Neon One integration
        return jsonify({
            'success': True,
            'contest_entry_id': result.get('entry_id'),
            'message': 'Contest entry submitted successfully',
            'postMessage': {
                'type': 'orchid_contest_submission',
                'member_id': data.get('member_id'),
                'contest_month': result.get('contest_period'),
                'entry_id': result.get('entry_id')
            }
        })
        
    except Exception as e:
        logger.error(f"Error submitting contest entry: {e}")
        return jsonify({'success': False, 'error': str(e)})

@app.route('/api/hub/pest-diseases/identify', methods=['POST'])
def api_pest_diseases_identify():
    """API endpoint for pest/disease identification"""
    try:
        # Use existing AI analysis system
        from orchid_ai import analyze_orchid_image
        
        files = request.files
        if 'image' not in files:
            return jsonify({'success': False, 'error': 'No image provided'})
            
        image = files['image']
        analysis_type = request.form.get('analysis_type', 'pest_disease')
        
        # Analyze image for pests/diseases
        result = analyze_orchid_image(image, focus=analysis_type)
        
        return jsonify({
            'success': True,
            'analysis': result,
            'recommendations': result.get('care_recommendations', [])
        })
        
    except Exception as e:
        logger.error(f"Error identifying pest/disease: {e}")
        return jsonify({'success': False, 'error': str(e)})


@app.route("/badge-test")
def badge_test():
    """Test page to display all philosophy badges"""
    from badge_test_route import create_badge_test_page
    return Response(create_badge_test_page(), mimetype="text/html")

@app.route('/api/orchid-of-the-day-data')
def orchid_of_the_day_data():
    """API endpoint for Orchid of the Day widget - serves orchid data in widget format"""
    try:
        # Get all orchids with images for the widget
        orchids = OrchidRecord.query.filter(
            or_(
                OrchidRecord.image_url.isnot(None),
                OrchidRecord.google_drive_id.isnot(None)
            )
        ).filter(
            OrchidRecord.validation_status == 'validated'
        ).all()
        
        widget_data = []
        
        for orchid in orchids:
            # Determine image URL
            image_url = None
            if orchid.google_drive_id:
                image_url = f"/api/drive-photo/{orchid.google_drive_id}"
            elif orchid.image_url:
                image_url = orchid.image_url
            
            if not image_url:
                continue
                
            # Format scientific name parts
            genus = orchid.genus or ""
            species = orchid.species or ""
            
            # Determine if hybrid
            hybrid = orchid.grex_name if orchid.is_hybrid else ""
            grex = orchid.grex_name or ""
            clonal_name = orchid.clone_name or ""
            
            # Build title based on available data
            title_parts = []
            if genus and species:
                title_parts.append(f"{genus} {species}")
            elif hybrid or grex:
                title_parts.append(hybrid or grex)
            elif orchid.display_name:
                title_parts.append(orchid.display_name)
            
            if clonal_name:
                title_parts.append(f"'{clonal_name}'")
                
            title = " ".join(title_parts) if title_parts else orchid.display_name or "Unknown Orchid"
            
            # Build caption from available data
            caption_parts = []
            
            if orchid.ai_description:
                # Use AI description but limit to ~55 words
                ai_desc = orchid.ai_description[:300] + "..." if len(orchid.ai_description) > 300 else orchid.ai_description
                caption_parts.append(ai_desc)
            else:
                # Build caption from other fields
                if orchid.native_habitat:
                    caption_parts.append(f"Native to {orchid.native_habitat}")
                elif orchid.country:
                    caption_parts.append(f"Found in {orchid.country}")
                
                if orchid.bloom_time:
                    caption_parts.append(f"Blooms {orchid.bloom_time}")
                    
                if orchid.growth_habit:
                    caption_parts.append(f"{orchid.growth_habit.replace('_', ' ').title()} growth habit")
                    
                if orchid.cultural_notes:
                    notes = orchid.cultural_notes[:150] + "..." if len(orchid.cultural_notes) > 150 else orchid.cultural_notes
                    caption_parts.append(notes)
            
            caption = ". ".join(caption_parts) if caption_parts else "A beautiful orchid from our collection."
            
            # Build the widget record
            record = {
                'id': orchid.id,
                'genus': genus,
                'species': species,
                'variety': '',  # Not used in current schema
                'hybrid': hybrid,
                'grex': grex,
                'clonal_name': clonal_name,
                'infraspecific_rank': '',  # Not used in current schema
                'orchid_group': genus,  # Use genus as group
                'title': title,
                'caption': caption,
                'image_url': image_url,
                'photographer': orchid.photographer or '',
                'credit': orchid.photographer or orchid.image_source or '',
                'license': '',  # Not tracked in current schema
                'license_url': '',  # Not tracked in current schema
                'native_range': orchid.country or orchid.region or orchid.native_habitat or '',
                'habitat': orchid.native_habitat or orchid.growing_environment or '',
                'elevation_m': '',  # Not easily accessible in current schema
                'blooming_season': orchid.bloom_time or orchid.bloom_season_indicator or '',
                'culture_notes': orchid.cultural_notes or '',
                'date': orchid.created_at.isoformat() if orchid.created_at else '',
                'tags': '',  # Not used in current schema
                'accession_id': str(orchid.id)
            }
            
            widget_data.append(record)
        
        logger.info(f"🌺 Serving {len(widget_data)} orchids for widget data")
        
        return jsonify(widget_data)
        
    except Exception as e:
        logger.error(f"Error serving orchid widget data: {e}")
        return jsonify({'error': 'Failed to load orchid data'}), 500

@app.route('/orchid-widget-demo')
def orchid_widget_demo():
    """Demo page showing the Orchid of the Day widget in action"""
    return render_template('orchid_widget_demo.html')

# Register Enhanced Systems (International Scraping + Mobile Field Research)
try:
    from enhanced_system_integration import register_enhanced_systems
    if register_enhanced_systems(app):
        logger.info("🚀 Enhanced international scraping and mobile field research systems registered")
        logger.info("📱 Mobile field research available at: /field/")
        logger.info("🌍 International scraping admin at: /admin/international/")
    else:
        logger.warning("⚠️ Failed to register enhanced systems")
except ImportError as e:
    logger.warning(f"Enhanced systems not available: {e}")
except Exception as e:
    logger.error(f"Error registering enhanced systems: {e}")

# Interactive Map Locator Route
@app.route('/orchid-map-locator')
def orchid_map_locator():
    """Interactive map interface for orchid locator with enhanced search functionality"""
    try:
        # Get some basic statistics for the map interface
        total_orchids = db.session.query(OrchidRecord).count()
        orchids_with_coordinates = db.session.query(OrchidRecord).filter(
            OrchidRecord.decimal_latitude.isnot(None),
            OrchidRecord.decimal_longitude.isnot(None)
        ).count()
        
        # Get available genera for quick filters
        genera = db.session.query(OrchidRecord.genus).filter(
            OrchidRecord.genus.isnot(None),
            OrchidRecord.genus != ''
        ).distinct().order_by(OrchidRecord.genus).limit(50).all()
        genera_list = [g[0] for g in genera if g[0]]
        
        # Get available regions
        regions = db.session.query(OrchidRecord.region).filter(
            OrchidRecord.region.isnot(None),
            OrchidRecord.region != ''
        ).distinct().order_by(OrchidRecord.region).limit(50).all()
        regions_list = [r[0] for r in regions if r[0]]
        
        logger.info(f"🗺️ Orchid Map Locator: {orchids_with_coordinates}/{total_orchids} orchids with coordinates")
        
        return render_template('orchid_map_locator.html',
                             total_orchids=total_orchids,
                             orchids_with_coordinates=orchids_with_coordinates,
                             genera=genera_list,
                             regions=regions_list)
    except Exception as e:
        logger.error(f"❌ Error in orchid map locator: {e}")
        flash('Error loading map interface', 'error')
        return redirect(url_for('search'))

# Register Nursery Directory API Routes
try:
    register_nursery_api_routes(app)
    logger.info("🌱 Orchid Nursery Directory API routes registered successfully")
except Exception as e:
    logger.error(f"❌ Error registering nursery directory routes: {e}")

# Register Society Directory API Routes  
try:
    register_society_api_routes(app)
    logger.info("🏛️ Orchid Society Directory API routes registered successfully")
except Exception as e:
    logger.error(f"❌ Error registering society directory routes: {e}")

# API endpoint for map coordinate data
@app.route('/api/orchid-map-coordinates')
def api_orchid_map_coordinates():
    """Get orchid coordinates for map display with filtering"""
    try:
        # Get query parameters
        genus = request.args.get('genus', '')
        region = request.args.get('region', '')
        source = request.args.get('source', '')
        limit = min(int(request.args.get('limit', 1000)), 2000)
        
        # Build query
        query = db.session.query(
            OrchidRecord.id,
            OrchidRecord.display_name,
            OrchidRecord.scientific_name,
            OrchidRecord.genus,
            OrchidRecord.species,
            OrchidRecord.decimal_latitude,
            OrchidRecord.decimal_longitude,
            OrchidRecord.region,
            OrchidRecord.country,
            OrchidRecord.image_url,
            OrchidRecord.google_drive_id,
            OrchidRecord.ingestion_source,
            OrchidRecord.is_featured
        ).filter(
            OrchidRecord.decimal_latitude.isnot(None),
            OrchidRecord.decimal_longitude.isnot(None)
        )
        
        # Apply filters
        if genus:
            query = query.filter(OrchidRecord.genus.ilike(f'%{genus}%'))
        if region:
            query = query.filter(OrchidRecord.region.ilike(f'%{region}%'))
        if source:
            query = query.filter(OrchidRecord.ingestion_source.ilike(f'%{source}%'))
            
        # Get results
        orchids = query.limit(limit).all()
        
        # Format for map display
        coordinates = []
        for orchid in orchids:
            # Get image URL
            image_url = get_image_with_recovery(orchid)
            
            coordinates.append({
                'id': orchid.id,
                'name': orchid.display_name or orchid.scientific_name,
                'scientific_name': orchid.scientific_name,
                'genus': orchid.genus,
                'species': orchid.species,
                'lat': float(orchid.decimal_latitude),
                'lng': float(orchid.decimal_longitude),
                'region': orchid.region,
                'country': orchid.country,
                'image_url': image_url,
                'source': orchid.ingestion_source,
                'is_featured': orchid.is_featured,
                'url': url_for('view_orchid', id=orchid.id)
            })
        
        logger.info(f"🗺️ Map API: Returned {len(coordinates)} orchid coordinates")
        
        return jsonify({
            'success': True,
            'coordinates': coordinates,
            'total': len(coordinates),
            'filters': {
                'genus': genus,
                'region': region,
                'source': source,
                'limit': limit
            }
        })
        
    except Exception as e:
        logger.error(f"❌ Error in map coordinates API: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

# Enhanced search export functionality
@app.route('/api/export-search-results')
def api_export_search_results():
    """Export current search results for download"""
    try:
        # Get all current search parameters
        query_params = dict(request.args)
        
        # Perform search with same parameters as main search
        # This is a simplified version - in production you'd want to reuse the main search logic
        search_query = db.session.query(OrchidRecord)
        
        # Apply basic filters (simplified)
        if 'q' in query_params and query_params['q']:
            search_term = query_params['q']
            search_query = search_query.filter(
                or_(
                    OrchidRecord.display_name.ilike(f'%{search_term}%'),
                    OrchidRecord.scientific_name.ilike(f'%{search_term}%'),
                    OrchidRecord.genus.ilike(f'%{search_term}%')
                )
            )
        
        # Limit results for export
        orchids = search_query.limit(1000).all()
        
        # Generate CSV content
        csv_content = "ID,Display Name,Scientific Name,Genus,Species,Region,Country,Source\n"
        for orchid in orchids:
            csv_content += f"{orchid.id},{orchid.display_name or ''},{orchid.scientific_name or ''},{orchid.genus or ''},{orchid.species or ''},{orchid.region or ''},{orchid.country or ''},{orchid.ingestion_source or ''}\n"
        
        # Create response
        response = make_response(csv_content)
        response.headers['Content-Type'] = 'text/csv'
        response.headers['Content-Disposition'] = f'attachment; filename=orchid_search_results_{datetime.now().strftime("%Y%m%d_%H%M%S")}.csv'
        
        logger.info(f"📊 Exported {len(orchids)} orchid search results")
        return response
        
    except Exception as e:
        logger.error(f"❌ Error exporting search results: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500


# =============================================================================
# COMPREHENSIVE MEMBERS COLLECTION AREA
# =============================================================================

@app.route('/members-collection')
@login_required
def members_collection():
    """
    Comprehensive Members Collection Area - Professional research hub
    Integrates external databases, ecological relationships, and literature tools
    """
    try:
        # Get user's collection data
        user_collections = MemberCollection.query.filter_by(user_id=current_user.id).all()
        
        # Get member statistics
        collection_stats = {
            'total_orchids': len(user_collections),
            'flowering_count': sum(1 for c in user_collections if c.flowering_status),
            'genera_count': len(set(c.orchid_record.genus for c in user_collections if c.orchid_record and c.orchid_record.genus)),
            'species_count': len(set(f"{c.orchid_record.genus} {c.orchid_record.species}" for c in user_collections if c.orchid_record and c.orchid_record.genus and c.orchid_record.species)),
            'research_ready': sum(1 for c in user_collections if c.available_for_research),
            'needs_attention': sum(1 for c in user_collections if c.health_status in ['stressed', 'diseased']),
            'photo_coverage': sum(c.photo_count for c in user_collections),
        }
        
        # Get research opportunities across all collection items
        research_opportunities = []
        for collection in user_collections:
            opportunities = collection.get_research_opportunities()
            for opp in opportunities:
                opp['orchid_name'] = collection.orchid_record.display_name if collection.orchid_record else 'Unknown'
                opp['collection_id'] = collection.id
            research_opportunities.extend(opportunities)
        
        # Get external database integration status
        external_db_status = {
            'eol_updated': sum(1 for c in user_collections if c.eol_data_updated),
            'gbif_updated': sum(1 for c in user_collections if c.gbif_data_updated),
            'ecological_complete': sum(1 for c in user_collections if c.ecological_data_complete),
            'literature_citations': sum(c.literature_citations_count for c in user_collections)
        }
        
        # Get recent additions
        recent_additions = MemberCollection.query.filter_by(user_id=current_user.id)\
            .order_by(MemberCollection.created_at.desc()).limit(5).all()
        
        # Get collaboration opportunities (other members with overlapping genera)
        user_genera = set(c.orchid_record.genus for c in user_collections if c.orchid_record and c.orchid_record.genus)
        potential_collaborators = []
        if user_genera:
            other_collections = MemberCollection.query\
                .filter(MemberCollection.user_id != current_user.id)\
                .filter(MemberCollection.available_for_research == True)\
                .join(OrchidRecord)\
                .filter(OrchidRecord.genus.in_(user_genera))\
                .limit(10).all()
            
            collaborator_dict = {}
            for collection in other_collections:
                user_id = collection.user_id
                if user_id not in collaborator_dict:
                    collaborator_dict[user_id] = {
                        'user': collection.user,
                        'shared_genera': set(),
                        'collection_count': 0
                    }
                collaborator_dict[user_id]['shared_genera'].add(collection.orchid_record.genus)
                collaborator_dict[user_id]['collection_count'] += 1
            
            potential_collaborators = [
                {
                    'user': data['user'],
                    'shared_genera': list(data['shared_genera']),
                    'collection_count': data['collection_count']
                }
                for data in collaborator_dict.values()
            ]
        
        # Get nursery recommendations based on collection gaps
        nursery_recommendations = get_nursery_directory()[:5]  # Top 5 nurseries
        
        # Get society recommendations
        society_recommendations = get_society_directory()[:3]  # Top 3 societies
        
        logger.info(f"🌺 Members Collection loaded for user {current_user.id}: {collection_stats['total_orchids']} orchids")
        
        return render_template('members_collection.html',
                             user_collections=user_collections,
                             collection_stats=collection_stats,
                             research_opportunities=research_opportunities,
                             external_db_status=external_db_status,
                             recent_additions=recent_additions,
                             potential_collaborators=potential_collaborators,
                             nursery_recommendations=nursery_recommendations,
                             society_recommendations=society_recommendations)
        
    except Exception as e:
        logger.error(f"❌ Error loading members collection: {e}")
        flash('Error loading collection data. Please try again.', 'error')
        return redirect(url_for('index'))


@app.route('/api/member-collection-stats')
@login_required
def api_member_collection_stats():
    """API endpoint for member collection analytics and insights"""
    try:
        user_collections = MemberCollection.query.filter_by(user_id=current_user.id).all()
        
        # Generate detailed analytics
        analytics = {
            'total_orchids': len(user_collections),
            'collection_health': {
                'healthy': sum(1 for c in user_collections if c.health_status == 'healthy'),
                'stressed': sum(1 for c in user_collections if c.health_status == 'stressed'),
                'diseased': sum(1 for c in user_collections if c.health_status == 'diseased'),
                'recovering': sum(1 for c in user_collections if c.health_status == 'recovering')
            },
            'flowering_status': {
                'currently_flowering': sum(1 for c in user_collections if c.flowering_status),
                'not_flowering': sum(1 for c in user_collections if not c.flowering_status)
            },
            'acquisition_sources': {},
            'genera_distribution': {},
            'monthly_acquisitions': {},
            'care_complexity': {
                'beginner': 0,
                'intermediate': 0,
                'advanced': 0
            }
        }
        
        # Process collection data for analytics
        for collection in user_collections:
            # Acquisition sources
            source = collection.acquisition_source or 'Unknown'
            analytics['acquisition_sources'][source] = analytics['acquisition_sources'].get(source, 0) + 1
            
            # Genera distribution
            if collection.orchid_record and collection.orchid_record.genus:
                genus = collection.orchid_record.genus
                analytics['genera_distribution'][genus] = analytics['genera_distribution'].get(genus, 0) + 1
            
            # Monthly acquisitions
            if collection.acquisition_date:
                month_key = collection.acquisition_date.strftime('%Y-%m')
                analytics['monthly_acquisitions'][month_key] = analytics['monthly_acquisitions'].get(month_key, 0) + 1
        
        # Calculate research potential
        analytics['research_potential'] = {
            'ready_for_research': sum(1 for c in user_collections if c.available_for_research),
            'missing_eol_data': sum(1 for c in user_collections if not c.eol_data_updated),
            'missing_gbif_data': sum(1 for c in user_collections if not c.gbif_data_updated),
            'incomplete_ecological': sum(1 for c in user_collections if not c.ecological_data_complete),
            'needs_photos': sum(1 for c in user_collections if c.photo_count < 3)
        }
        
        return jsonify({
            'success': True,
            'analytics': analytics,
            'last_updated': datetime.utcnow().isoformat()
        })
        
    except Exception as e:
        logger.error(f"❌ Error generating collection stats: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500


@app.route('/api/member-research-opportunities')
@login_required
def api_member_research_opportunities():
    """API endpoint for research opportunities and collaboration suggestions"""
    try:
        user_collections = MemberCollection.query.filter_by(user_id=current_user.id).all()
        
        all_opportunities = []
        for collection in user_collections:
            opportunities = collection.get_research_opportunities()
            for opp in opportunities:
                opp.update({
                    'collection_id': collection.id,
                    'orchid_name': collection.orchid_record.display_name if collection.orchid_record else 'Unknown',
                    'genus': collection.orchid_record.genus if collection.orchid_record else None,
                    'species': collection.orchid_record.species if collection.orchid_record else None
                })
            all_opportunities.extend(opportunities)
        
        # Categorize opportunities by priority and type
        categorized = {
            'high_priority': [opp for opp in all_opportunities if opp.get('priority') == 'high'],
            'medium_priority': [opp for opp in all_opportunities if opp.get('priority') == 'medium'],
            'external_database': [opp for opp in all_opportunities if opp.get('type') == 'external_database'],
            'documentation': [opp for opp in all_opportunities if opp.get('type') == 'documentation'],
            'ecological': [opp for opp in all_opportunities if opp.get('type') == 'ecological']
        }
        
        # Get collaboration suggestions
        user_genera = set(c.orchid_record.genus for c in user_collections if c.orchid_record and c.orchid_record.genus)
        collaboration_suggestions = []
        
        if user_genera:
            # Find active research collaborations
            existing_collaborations = ResearchCollaboration.query.filter(
                or_(
                    ResearchCollaboration.initiator_user_id == current_user.id,
                    ResearchCollaboration.collaborator_user_id == current_user.id
                )
            ).filter(ResearchCollaboration.status == 'active').all()
            
            collaboration_suggestions = [collab.to_dict() for collab in existing_collaborations]
        
        return jsonify({
            'success': True,
            'opportunities': categorized,
            'collaboration_suggestions': collaboration_suggestions,
            'summary': {
                'total_opportunities': len(all_opportunities),
                'high_priority': len(categorized['high_priority']),
                'active_collaborations': len(collaboration_suggestions)
            }
        })
        
    except Exception as e:
        logger.error(f"❌ Error generating research opportunities: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500


@app.route('/api/member-ecological-network')
@login_required
def api_member_ecological_network():
    """API endpoint for ecological relationships and pollinator networks"""
    try:
        user_collections = MemberCollection.query.filter_by(user_id=current_user.id).all()
        user_orchid_ids = [c.orchid_record_id for c in user_collections if c.orchid_record_id]
        
        if not user_orchid_ids:
            return jsonify({
                'success': True,
                'network': {'nodes': [], 'edges': []},
                'summary': {'total_relationships': 0, 'unique_pollinators': 0}
            })
        
        # Get pollinator relationships for user's orchids
        pollinator_relationships = AdvancedOrchidPollinatorRelationship.query\
            .filter(AdvancedOrchidPollinatorRelationship.orchid_id.in_(user_orchid_ids)).all()
        
        # Build network graph data
        nodes = []
        edges = []
        pollinator_ids = set()
        
        # Add orchid nodes
        for collection in user_collections:
            if collection.orchid_record:
                nodes.append({
                    'id': f"orchid_{collection.orchid_record.id}",
                    'type': 'orchid',
                    'name': collection.orchid_record.display_name,
                    'genus': collection.orchid_record.genus,
                    'species': collection.orchid_record.species,
                    'flowering_status': collection.flowering_status,
                    'health_status': collection.health_status
                })
        
        # Add pollinator relationships
        for relationship in pollinator_relationships:
            if relationship.pollinator:
                pollinator_id = f"pollinator_{relationship.pollinator.id}"
                pollinator_ids.add(pollinator_id)
                
                # Add pollinator node if not already added
                if not any(node['id'] == pollinator_id for node in nodes):
                    nodes.append({
                        'id': pollinator_id,
                        'type': 'pollinator',
                        'name': relationship.pollinator.common_name or relationship.pollinator.scientific_name,
                        'scientific_name': relationship.pollinator.scientific_name,
                        'pollinator_type': relationship.pollinator.pollinator_type.value if relationship.pollinator.pollinator_type else 'unknown'
                    })
                
                # Add edge
                edges.append({
                    'source': f"orchid_{relationship.orchid_id}",
                    'target': pollinator_id,
                    'relationship_type': relationship.relationship_type.value if relationship.relationship_type else 'unknown',
                    'effectiveness_score': relationship.effectiveness_score or 0.5,
                    'observation_frequency': relationship.observation_frequency or 0
                })
        
        # Generate ecosystem insights
        insights = {
            'total_relationships': len(pollinator_relationships),
            'unique_pollinators': len(pollinator_ids),
            'relationship_types': {},
            'pollinator_types': {},
            'conservation_status': {}
        }
        
        for relationship in pollinator_relationships:
            rel_type = relationship.relationship_type.value if relationship.relationship_type else 'unknown'
            insights['relationship_types'][rel_type] = insights['relationship_types'].get(rel_type, 0) + 1
            
            if relationship.pollinator:
                poll_type = relationship.pollinator.pollinator_type.value if relationship.pollinator.pollinator_type else 'unknown'
                insights['pollinator_types'][poll_type] = insights['pollinator_types'].get(poll_type, 0) + 1
        
        return jsonify({
            'success': True,
            'network': {
                'nodes': nodes,
                'edges': edges
            },
            'insights': insights,
            'summary': {
                'total_relationships': insights['total_relationships'],
                'unique_pollinators': insights['unique_pollinators'],
                'network_complexity': len(edges) / max(len(nodes), 1)
            }
        })
        
    except Exception as e:
        logger.error(f"❌ Error generating ecological network: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500


@app.route('/api/member-literature-export')
@login_required
def api_member_literature_export():
    """API endpoint for academic citation and BibTeX generation"""
    try:
        format_type = request.args.get('format', 'bibtex')  # bibtex, csv, json
        user_collections = MemberCollection.query.filter_by(user_id=current_user.id).all()
        
        # Get relevant literature citations
        user_genera = [c.orchid_record.genus for c in user_collections if c.orchid_record and c.orchid_record.genus]
        user_species = [f"{c.orchid_record.genus} {c.orchid_record.species}" 
                       for c in user_collections 
                       if c.orchid_record and c.orchid_record.genus and c.orchid_record.species]
        
        relevant_citations = []
        if user_genera:
            citations = LiteratureCitation.query.filter(
                or_(
                    LiteratureCitation.relevant_genera.op('?')(user_genera),
                    LiteratureCitation.relevant_species.op('?')(user_species)
                )
            ).all()
            relevant_citations = citations
        
        if format_type == 'bibtex':
            # Generate BibTeX format
            bibtex_content = f"% BibTeX Export for {current_user.first_name or 'Member'} Collection\n"
            bibtex_content += f"% Generated on {datetime.utcnow().strftime('%Y-%m-%d %H:%M:%S')} UTC\n"
            bibtex_content += f"% Total orchids: {len(user_collections)}\n\n"
            
            for citation in relevant_citations:
                bibtex_content += citation.generate_bibtex() + "\n"
            
            response = make_response(bibtex_content)
            response.headers['Content-Type'] = 'text/plain'
            response.headers['Content-Disposition'] = f'attachment; filename=member_collection_bibliography_{datetime.now().strftime("%Y%m%d")}.bib'
            return response
            
        elif format_type == 'csv':
            # Generate CSV format
            csv_content = "Title,Authors,Journal,Year,DOI,Relevant Genera,Relevant Species\n"
            for citation in relevant_citations:
                csv_content += f'"{citation.title}","{citation.authors}","{citation.journal or ""}","{citation.publication_year or ""}","{citation.doi or ""}","{"; ".join(citation.relevant_genera or [])}","{"; ".join(citation.relevant_species or [])}"\n'
            
            response = make_response(csv_content)
            response.headers['Content-Type'] = 'text/csv'
            response.headers['Content-Disposition'] = f'attachment; filename=member_collection_literature_{datetime.now().strftime("%Y%m%d")}.csv'
            return response
            
        else:  # JSON format
            citations_data = [citation.to_dict() for citation in relevant_citations]
            collection_data = [collection.to_dict() for collection in user_collections]
            
            export_data = {
                'export_info': {
                    'generated_at': datetime.utcnow().isoformat(),
                    'user_id': current_user.id,
                    'total_orchids': len(user_collections),
                    'total_citations': len(relevant_citations)
                },
                'collection': collection_data,
                'literature': citations_data
            }
            
            return jsonify(export_data)
        
    except Exception as e:
        logger.error(f"❌ Error generating literature export: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500


@app.route('/api/member-external-search')
@login_required
def api_member_external_search():
    """API endpoint for external database searches (EOL/GBIF)"""
    try:
        query = request.args.get('q', '').strip()
        database = request.args.get('db', 'both')  # 'eol', 'gbif', 'both'
        
        if not query:
            return jsonify({'success': False, 'error': 'Search query required'}), 400
        
        results = {}
        
        # Check cache first
        if database in ['eol', 'both']:
            cached_eol = ExternalDatabaseCache.get_cached_result('eol', 'species_search', query)
            if cached_eol:
                results['eol'] = cached_eol
            else:
                # Perform EOL search
                eol_integrator = EOLIntegrator()
                eol_result = eol_integrator.search_eol_species(query)
                if eol_result:
                    results['eol'] = eol_result
                    # Cache the result
                    ExternalDatabaseCache.cache_result('eol', 'species_search', query, eol_result)
        
        if database in ['gbif', 'both']:
            cached_gbif = ExternalDatabaseCache.get_cached_result('gbif', 'species_search', query)
            if cached_gbif:
                results['gbif'] = cached_gbif
            else:
                # Perform GBIF search
                gbif_integrator = GBIFIntegrator()
                gbif_result = gbif_integrator.search_species(query)
                if gbif_result:
                    results['gbif'] = gbif_result
                    # Cache the result
                    ExternalDatabaseCache.cache_result('gbif', 'species_search', query, gbif_result)
        
        # Add enrichment suggestions for user's collection
        enrichment_suggestions = []
        user_collections = MemberCollection.query.filter_by(user_id=current_user.id).all()
        
        for collection in user_collections:
            if (collection.orchid_record and 
                collection.orchid_record.scientific_name and 
                query.lower() in collection.orchid_record.scientific_name.lower()):
                
                suggestions = []
                if not collection.eol_data_updated:
                    suggestions.append('Add EOL data')
                if not collection.gbif_data_updated:
                    suggestions.append('Add GBIF occurrences')
                if not collection.ecological_data_complete:
                    suggestions.append('Map ecological relationships')
                
                if suggestions:
                    enrichment_suggestions.append({
                        'collection_id': collection.id,
                        'orchid_name': collection.orchid_record.display_name,
                        'suggestions': suggestions
                    })
        
        return jsonify({
            'success': True,
            'query': query,
            'results': results,
            'enrichment_suggestions': enrichment_suggestions,
            'cache_status': {
                'eol_cached': 'eol' in results and cached_eol is not None,
                'gbif_cached': 'gbif' in results and cached_gbif is not None
            }
        })
        
    except Exception as e:
        logger.error(f"❌ Error performing external search: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

# =============================================================================
# KNOWLEDGE BASE ROUTES - Educational Learning System
# =============================================================================

@app.route('/knowledge-base')
def knowledge_base():
    """Main knowledge base page with search and categories"""
    try:
        from models import KnowledgeBase
        
        # Get search parameters
        search_query = request.args.get('q', '').strip()
        category_filter = request.args.get('category', '').strip()
        difficulty_filter = request.args.get('difficulty', '').strip()
        content_type_filter = request.args.get('type', '').strip()
        
        # Build query
        query = KnowledgeBase.query
        
        # Apply filters
        if search_query:
            query = query.filter(
                or_(
                    KnowledgeBase.title.ilike(f'%{search_query}%'),
                    KnowledgeBase.question.ilike(f'%{search_query}%'),
                    KnowledgeBase.answer.ilike(f'%{search_query}%'),
                    KnowledgeBase.article.ilike(f'%{search_query}%'),
                    KnowledgeBase.category.ilike(f'%{search_query}%')
                )
            )
        
        if category_filter:
            query = query.filter(KnowledgeBase.category == category_filter)
        
        if difficulty_filter:
            query = query.filter(KnowledgeBase.difficulty_level == difficulty_filter)
            
        if content_type_filter:
            query = query.filter(KnowledgeBase.content_type == content_type_filter)
        
        # Get results
        entries = query.order_by(KnowledgeBase.title).all()
        
        # Get all categories for filter dropdown
        categories = db.session.query(KnowledgeBase.category).distinct().order_by(KnowledgeBase.category).all()
        categories = [cat[0] for cat in categories]
        
        # Get statistics
        stats = {
            'total_entries': KnowledgeBase.query.count(),
            'categories': len(categories),
            'difficulty_counts': {
                'beginner': KnowledgeBase.query.filter_by(difficulty_level='beginner').count(),
                'intermediate': KnowledgeBase.query.filter_by(difficulty_level='intermediate').count(),
                'advanced': KnowledgeBase.query.filter_by(difficulty_level='advanced').count()
            },
            'content_types': {
                'qa': KnowledgeBase.query.filter_by(content_type='qa').count(),
                'article': KnowledgeBase.query.filter_by(content_type='article').count()
            }
        }
        
        return render_template('knowledge_base.html', 
                             entries=entries,
                             categories=categories,
                             stats=stats,
                             search_query=search_query,
                             category_filter=category_filter,
                             difficulty_filter=difficulty_filter,
                             content_type_filter=content_type_filter)
        
    except Exception as e:
        logger.error(f"Error loading knowledge base: {str(e)}")
        flash('Error loading knowledge base content', 'error')
        # Provide safe default stats to prevent template errors
        safe_stats = {
            'total_entries': 0,
            'categories': 0,
            'difficulty_counts': {
                'beginner': 0,
                'intermediate': 0,
                'advanced': 0
            },
            'content_types': {
                'qa': 0,
                'article': 0
            }
        }
        return render_template('knowledge_base.html', entries=[], categories=[], stats=safe_stats)

@app.route('/api/knowledge-base/search')
def api_knowledge_base_search():
    """API endpoint for knowledge base search with JSON response"""
    try:
        from models import KnowledgeBase
        
        # Get parameters
        query_text = request.args.get('q', '').strip()
        category = request.args.get('category', '')
        difficulty = request.args.get('difficulty', '')
        limit = min(int(request.args.get('limit', 20)), 100)  # Max 100 results
        
        # Build query
        query = KnowledgeBase.query
        
        if query_text:
            query = query.filter(
                or_(
                    KnowledgeBase.title.ilike(f'%{query_text}%'),
                    KnowledgeBase.question.ilike(f'%{query_text}%'),
                    KnowledgeBase.answer.ilike(f'%{query_text}%'),
                    KnowledgeBase.category.ilike(f'%{query_text}%')
                )
            )
        
        if category:
            query = query.filter(KnowledgeBase.category == category)
            
        if difficulty:
            query = query.filter(KnowledgeBase.difficulty_level == difficulty)
        
        # Get results
        entries = query.limit(limit).all()
        
        # Format response
        results = []
        for entry in entries:
            results.append({
                'id': entry.id,
                'title': entry.title,
                'question': entry.question,
                'answer': entry.answer[:200] + '...' if entry.answer and len(entry.answer) > 200 else entry.answer,
                'category': entry.category,
                'difficulty_level': entry.difficulty_level,
                'content_type': entry.content_type,
                'keywords': entry.keywords,
                'related_genera': entry.related_genera
            })
        
        return jsonify({
            'success': True,
            'count': len(results),
            'results': results
        })
        
    except Exception as e:
        logger.error(f"Error in knowledge base search API: {str(e)}")
        return jsonify({
            'success': False,
            'error': 'Search failed',
            'count': 0,
            'results': []
        }), 500

@app.route('/api/knowledge-base/categories')
def api_knowledge_base_categories():
    """Get all knowledge base categories with counts"""
    try:
        from models import KnowledgeBase
        
        # Get category counts
        category_data = db.session.query(
            KnowledgeBase.category,
            func.count(KnowledgeBase.id).label('count')
        ).group_by(KnowledgeBase.category).order_by(KnowledgeBase.category).all()
        
        categories = []
        for category, count in category_data:
            categories.append({
                'name': category,
                'count': count,
                'slug': category.lower().replace(' ', '-').replace('&', 'and')
            })
        
        return jsonify({
            'success': True,
            'categories': categories,
            'total_categories': len(categories)
        })
        
    except Exception as e:
        logger.error(f"Error getting knowledge base categories: {str(e)}")
        return jsonify({
            'success': False,
            'error': 'Failed to load categories',
            'categories': []
        }), 500

@app.route('/knowledge-base/entry/<int:entry_id>')
def knowledge_base_entry(entry_id):
    """Individual knowledge base entry page"""
    try:
        from models import KnowledgeBase
        
        entry = KnowledgeBase.query.get_or_404(entry_id)
        
        # Get related entries (same category, different content)
        related_entries = KnowledgeBase.query.filter(
            KnowledgeBase.category == entry.category,
            KnowledgeBase.id != entry.id
        ).limit(3).all()
        
        return render_template('knowledge_base_entry.html', 
                             entry=entry,
                             related_entries=related_entries)
        
    except Exception as e:
        logger.error(f"Error loading knowledge base entry {entry_id}: {str(e)}")
        flash('Knowledge base entry not found', 'error')
        return redirect(url_for('knowledge_base'))

@app.route('/api/knowledge-base/popular')
def api_knowledge_base_popular():
    """Get popular knowledge base entries (by category diversity)"""
    try:
        from models import KnowledgeBase
        
        # Get one entry from each category for diversity
        popular_entries = []
        categories = db.session.query(KnowledgeBase.category).distinct().all()
        
        for category_tuple in categories[:10]:  # Limit to 10 categories
            category = category_tuple[0]
            entry = KnowledgeBase.query.filter_by(category=category).first()
            if entry:
                popular_entries.append({
                    'id': entry.id,
                    'title': entry.title,
                    'category': entry.category,
                    'difficulty_level': entry.difficulty_level,
                    'content_type': entry.content_type,
                    'preview': entry.answer[:150] + '...' if entry.answer and len(entry.answer) > 150 else entry.answer
                })
        
        return jsonify({
            'success': True,
            'entries': popular_entries,
            'count': len(popular_entries)
        })
        
    except Exception as e:
        logger.error(f"Error getting popular knowledge base entries: {str(e)}")
        return jsonify({
            'success': False,
            'error': 'Failed to load popular entries',
            'entries': []
        }), 500

logger.info("🧠 Knowledge Base routes registered successfully")

# =====================================================
# FIVE CITIES ORCHID SOCIETY IMPORT SYSTEM
# =====================================================

@app.route('/admin/fcos-import')
def admin_fcos_import():
    """Five Cities Orchid Society data import dashboard"""
    try:
        # Get current database stats
        total_orchids = OrchidRecord.query.count()
        fcos_orchids = OrchidRecord.query.filter_by(ingestion_source='google_sheets_import').count()
        
        # Check Google Drive connectivity
        from google_drive_service import get_drive_service, get_folder_contents
        drive_status = "connected" if get_drive_service() else "disconnected"
        
        stats = {
            'total_orchids': total_orchids,
            'fcos_orchids': fcos_orchids,
            'drive_status': drive_status,
            'target_records': 1337,
            'target_images': 875
        }
        
        return render_template('admin/fcos_import.html', stats=stats)
        
    except Exception as e:
        logger.error(f"Error loading FCOS import dashboard: {e}")
        flash('Error loading import dashboard', 'error')
        return redirect(url_for('admin_orchid_approval'))

@app.route('/api/fcos-import/test', methods=['POST'])
def api_fcos_import_test():
    """Run test import of 10 FCOS records"""
    try:
        from google_sheets_importer import GoogleSheetsImporter
        
        # Use the known FCOS sheet ID
        sheet_id = '1103vQ_D00Qio5W7PllFeRaFoFAzr7jd8ivOo79sdfgs'
        importer = GoogleSheetsImporter(sheet_id)
        
        logger.info("🧪 Starting FCOS test import (10 records)")
        
        # Run import with limit of 10 records
        result = importer.import_all_records(max_records=10)
        
        # Validate File ID linking for imported records
        if result['success'] and result['imported'] > 0:
            linked_count = validate_file_id_linking()
            result['linked_images'] = linked_count
            
        logger.info(f"✅ FCOS test import completed: {result}")
        
        return jsonify({
            'success': True,
            'message': f"Test import completed: {result['imported']} imported, {result['skipped']} skipped",
            'imported': result['imported'],
            'skipped': result['skipped'], 
            'errors': result['errors'],
            'linked_images': result.get('linked_images', 0),
            'total_in_database': result['total_in_database']
        })
        
    except Exception as e:
        logger.error(f"FCOS test import failed: {e}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500

@app.route('/api/fcos-import/full', methods=['POST'])
def api_fcos_import_full():
    """Run full import of all FCOS records"""
    try:
        from google_sheets_importer import GoogleSheetsImporter
        
        sheet_id = '1103vQ_D00Qio5W7PllFeRaFoFAzr7jd8ivOo79sdfgs'
        importer = GoogleSheetsImporter(sheet_id)
        
        logger.info("🚀 Starting FCOS full import (all 1,337 records)")
        
        # Run full import
        result = importer.import_all_records()
        
        # Validate File ID linking for all FCOS records
        if result['success']:
            linked_count = validate_file_id_linking()
            result['linked_images'] = linked_count
            
        logger.info(f"✅ FCOS full import completed: {result}")
        
        return jsonify({
            'success': True,
            'message': f"Full import completed: {result['imported']} imported, {result['skipped']} skipped",
            'imported': result['imported'],
            'skipped': result['skipped'],
            'errors': result['errors'], 
            'linked_images': result.get('linked_images', 0),
            'total_in_database': result['total_in_database']
        })
        
    except Exception as e:
        logger.error(f"FCOS full import failed: {e}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500

@app.route('/api/fcos-import/validate-linking')
def api_fcos_validate_linking():
    """Validate File ID linking between sheets and Google Drive"""
    try:
        linked_count = validate_file_id_linking()
        
        # Get drive folder stats
        from google_drive_service import get_folder_contents
        folder_id = '1YqIWmIfaXSy_0_bAbvSG8EMQjAuNq0lj'
        drive_files = get_folder_contents(folder_id)
        
        # Get FCOS records with File IDs
        fcos_records = OrchidRecord.query.filter(
            OrchidRecord.ingestion_source == 'google_sheets_import',
            OrchidRecord.google_drive_id.isnot(None)
        ).count()
        
        return jsonify({
            'success': True,
            'linked_records': linked_count,
            'fcos_records_with_file_ids': fcos_records,
            'drive_images_available': len(drive_files),
            'linking_percentage': round((linked_count / max(fcos_records, 1)) * 100, 1)
        })
        
    except Exception as e:
        logger.error(f"File ID validation failed: {e}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500

@app.route('/api/fcos-import/preview')
def api_fcos_import_preview():
    """Preview FCOS data without importing"""
    try:
        from google_sheets_importer import GoogleSheetsImporter
        
        sheet_id = '1103vQ_D00Qio5W7PllFeRaFoFAzr7jd8ivOo79sdfgs'
        importer = GoogleSheetsImporter(sheet_id)
        
        # Fetch first 5 rows for preview
        rows = importer.fetch_sheet_data()[:5]
        
        return jsonify({
            'success': True,
            'preview_rows': rows,
            'total_rows': len(importer.fetch_sheet_data()),
            'columns': list(rows[0].keys()) if rows else []
        })
        
    except Exception as e:
        logger.error(f"FCOS preview failed: {e}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500

def validate_file_id_linking():
    """Helper function to validate File ID linking"""
    try:
        from google_drive_service import get_folder_contents
        
        # Get FCOS records with File IDs
        fcos_records = OrchidRecord.query.filter(
            OrchidRecord.ingestion_source == 'google_sheets_import',
            OrchidRecord.google_drive_id.isnot(None)
        ).all()
        
        # Get drive folder contents
        folder_id = '1YqIWmIfaXSy_0_bAbvSG8EMQjAuNq0lj'
        drive_files = get_folder_contents(folder_id)
        drive_file_ids = [f['id'] for f in drive_files]
        
        linked_count = 0
        for record in fcos_records:
            if record.google_drive_id in drive_file_ids:
                linked_count += 1
            else:
                logger.warning(f"Missing drive image for File ID: {record.google_drive_id} (Orchid: {record.display_name})")
        
        logger.info(f"📎 File ID validation: {linked_count}/{len(fcos_records)} records have valid image links")
        return linked_count
        
    except Exception as e:
        logger.error(f"File ID validation error: {e}")
        return 0

logger.info("🌺 FCOS Import System routes registered successfully")

# ==============================================================================
# GBIF Ecosystem API Integration Routes - Ecosystem Explorer Widget
# (Maintaining /api/trefle/ endpoints for widget compatibility)
# ==============================================================================

@app.route('/api/trefle/status')
def trefle_service_status():
    """Get current status of the GBIF Botanical Service (maintaining /api/trefle/ for widget compatibility)"""
    try:
        if get_trefle_service_status is None:
            return jsonify({
                'enabled': False,
                'error': 'GBIF ecosystem service not available',
                'api_key_configured': False
            }), 503
        
        status = get_trefle_service_status()
        logger.info("📊 GBIF ecosystem service status requested")
        return jsonify(status)
        
    except Exception as e:
        logger.error(f"Error getting GBIF ecosystem service status: {str(e)}")
        return jsonify({
            'enabled': False,
            'error': str(e),
            'api_key_configured': False
        }), 500

@app.route('/api/trefle/search')
def trefle_search_orchid():
    """Search for orchid ecosystem data by scientific name using GBIF (maintaining /api/trefle/ for widget compatibility)"""
    try:
        scientific_name = request.args.get('scientific_name', '').strip()
        if not scientific_name:
            return jsonify({
                'success': False,
                'error': 'scientific_name parameter is required'
            }), 400
        
        if search_orchid_ecosystem_data is None:
            return jsonify({
                'success': False,
                'error': 'GBIF ecosystem service not available',
                'enabled': False
            }), 503
        
        logger.info(f"🔍 Searching GBIF ecosystem data for: {scientific_name}")
        ecosystem_data = search_orchid_ecosystem_data(scientific_name)
        
        # Check for rate limit response
        if isinstance(ecosystem_data, dict) and ecosystem_data.get('rate_limit_exceeded'):
            return jsonify({
                'success': False,
                'error': ecosystem_data.get('error', 'Rate limit exceeded'),
                'retry_after': ecosystem_data.get('retry_after', 60)
            }), 429
        
        if ecosystem_data:
            return jsonify({
                'success': True,
                'found': True,
                'scientific_name': scientific_name,
                'ecosystem_data': ecosystem_data,
                'source': 'gbif_ecosystem'
            })
        else:
            return jsonify({
                'success': True,
                'found': False,
                'scientific_name': scientific_name,
                'message': f'No ecosystem data found for {scientific_name}',
                'source': 'gbif_ecosystem'
            })
            
    except Exception as e:
        logger.error(f"Error searching GBIF ecosystem for {scientific_name}: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e),
            'scientific_name': scientific_name
        }), 500

@app.route('/api/trefle/enrich/<int:orchid_id>', methods=['POST'])
@login_required
def trefle_enrich_orchid(orchid_id):
    """Enrich a specific orchid record with Trefle ecosystem data"""
    # Admin role check for database modification
    if not current_user.is_admin:
        return jsonify({
            'success': False,
            'error': 'Admin access required for Trefle data enrichment'
        }), 403
    
    try:
        if enrich_orchid_with_trefle_data is None:
            return jsonify({
                'success': False,
                'error': 'Trefle service not available'
            }), 503
        
        # Get the orchid record first for validation
        orchid = OrchidRecord.query.get(orchid_id)
        if not orchid:
            return jsonify({
                'success': False,
                'error': f'Orchid record {orchid_id} not found'
            }), 404
        
        # Validate required OrchidRecord fields
        if not hasattr(orchid, 'scientific_name') or not orchid.scientific_name:
            return jsonify({
                'success': False,
                'error': 'Orchid record missing required scientific_name field'
            }), 400
        
        display_name = getattr(orchid, 'display_name', f'Orchid {orchid_id}')
        logger.info(f"🌿 Enriching orchid {orchid_id} ({display_name}) with GBIF ecosystem data")
        
        enrichment_result = enrich_orchid_with_trefle_data(orchid_id)
        
        # Check for rate limit response
        if isinstance(enrichment_result, dict) and enrichment_result.get('rate_limit_exceeded'):
            logger.warning(f"Rate limit exceeded for orchid {orchid_id}")
            response = jsonify({
                'success': False,
                'error': enrichment_result.get('error', 'Rate limit exceeded'),
                'orchid_id': orchid_id,
                'orchid_name': display_name
            })
            response.headers['Retry-After'] = str(enrichment_result.get('retry_after', 60))
            return response, 429
        
        if enrichment_result:
            # Refresh the orchid record to get updated data
            db.session.refresh(orchid)
            
            return jsonify({
                'success': True,
                'orchid_id': orchid_id,
                'orchid_name': orchid.display_name,
                'message': f'Successfully enriched {orchid.display_name} with ecosystem data',
                'enriched_fields': {
                    'native_habitat': bool(orchid.native_habitat),
                    'habitat_research': bool(orchid.habitat_research),
                    'climate_preference': orchid.climate_preference
                }
            })
        else:
            return jsonify({
                'success': False,
                'orchid_id': orchid_id,
                'orchid_name': display_name,
                'error': 'Failed to enrich orchid with ecosystem data'
            }), 500
            
    except Exception as e:
        logger.error(f"Error enriching orchid {orchid_id}: {str(e)}")
        return jsonify({
            'success': False,
            'orchid_id': orchid_id,
            'error': str(e)
        }), 500

@app.route('/api/trefle/batch-enrich', methods=['POST'])
@login_required
def trefle_batch_enrich():
    """Batch enrich multiple orchid records with Trefle ecosystem data"""
    # Admin role check for database modification
    if not current_user.is_admin:
        return jsonify({
            'success': False,
            'error': 'Admin access required for Trefle batch enrichment'
        }), 403
    
    try:
        if batch_enrich_orchids_with_trefle is None:
            return jsonify({
                'success': False,
                'error': 'Trefle service not available'
            }), 503
        
        # Get parameters from request
        data = request.get_json() or {}
        limit = min(data.get('limit', 25), 100)  # Cap at 100 for safety
        
        logger.info(f"🚀 Starting GBIF ecosystem batch enrichment: limit={limit}")
        
        results = batch_enrich_orchids_with_trefle(limit=limit)
        
        # Check for rate limit in batch results  
        if 'error' in results and 'retry_after' in results:
            logger.warning(f"Batch enrichment hit rate limit after processing {results.get('total_processed', 0)} orchids")
            response = jsonify({
                'success': False,
                'error': results['error'],
                'partial_results': results
            })
            response.headers['Retry-After'] = str(results.get('retry_after', 60))
            return response, 429
        
        # Add rate limiting info
        if get_trefle_service:
            service = get_trefle_service()
            results['rate_limit_status'] = f"{len(service.request_history)}/{service.RATE_LIMIT_CALLS} requests used"
        
        logger.info(f"✅ Batch enrichment completed: {results.get('successful_enrichments', 0)}/{results.get('total_processed', 0)} successful")
        
        return jsonify({
            'success': True,
            'batch_results': results
        })
        
    except Exception as e:
        logger.error(f"Error in batch enrichment: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500

@app.route('/api/trefle/orchid/<int:orchid_id>/ecosystem')
def trefle_orchid_ecosystem_data(orchid_id):
    """Get ecosystem data for a specific orchid (from database or fresh from Trefle)"""
    try:
        orchid = OrchidRecord.query.get(orchid_id)
        if not orchid:
            return jsonify({
                'success': False,
                'error': f'Orchid {orchid_id} not found'
            }), 404
        
        # Check if orchid already has GBIF ecosystem data in habitat_research
        existing_data = None
        if orchid.habitat_research:
            try:
                habitat_data = orchid.habitat_research if isinstance(orchid.habitat_research, dict) else json.loads(orchid.habitat_research)
                existing_data = habitat_data.get('ecosystem_data') or habitat_data.get('trefle_ecosystem_data')
            except:
                pass
        
        if existing_data:
            logger.info(f"📊 Returning cached ecosystem data for orchid {orchid_id}")
            return jsonify({
                'success': True,
                'orchid_id': orchid_id,
                'orchid_name': orchid.display_name,
                'scientific_name': orchid.scientific_name,
                'has_existing_data': True,
                'ecosystem_data': existing_data,
                'source': 'database_cache'
            })
        else:
            # Try to get fresh data from GBIF
            if get_trefle_service is None:
                return jsonify({
                    'success': False,
                    'error': 'GBIF ecosystem service not available and no cached data found'
                }), 503
            
            logger.info(f"🔍 Fetching fresh GBIF ecosystem data for orchid {orchid_id}")
            service = get_trefle_service()
            ecosystem_data = service.get_ecosystem_habitat_data(orchid)
            
            return jsonify({
                'success': True,
                'orchid_id': orchid_id,
                'orchid_name': orchid.display_name,
                'scientific_name': orchid.scientific_name,
                'has_existing_data': False,
                'ecosystem_data': ecosystem_data,
                'source': 'gbif_ecosystem_live'
            })
            
    except Exception as e:
        logger.error(f"Error getting ecosystem data for orchid {orchid_id}: {str(e)}")
        return jsonify({
            'success': False,
            'orchid_id': orchid_id,
            'error': str(e)
        }), 500

@app.route('/api/trefle/ecosystems/summary')
def trefle_ecosystem_summary():
    """Get summary statistics of orchids with Trefle ecosystem data"""
    try:
        # Count orchids with Trefle data
        orchids_with_trefle = OrchidRecord.query.filter(
            OrchidRecord.habitat_research.contains('"trefle_ecosystem_data"')
        ).count()
        
        total_orchids = OrchidRecord.query.count()
        
        # Get sample of enriched orchids
        sample_orchids = OrchidRecord.query.filter(
            OrchidRecord.habitat_research.contains('"trefle_ecosystem_data"')
        ).limit(10).all()
        
        sample_data = []
        for orchid in sample_orchids:
            try:
                habitat_data = orchid.habitat_research if isinstance(orchid.habitat_research, dict) else json.loads(orchid.habitat_research)
                trefle_data = habitat_data.get('trefle_ecosystem_data', {})
                sample_data.append({
                    'id': orchid.id,
                    'name': orchid.display_name,
                    'scientific_name': orchid.scientific_name,
                    'has_trefle_data': bool(trefle_data),
                    'trefle_id': trefle_data.get('trefle_id'),
                    'family_name': trefle_data.get('family_name')
                })
            except:
                continue
        
        # Get service status
        service_status = {}
        if get_trefle_service_status:
            service_status = get_trefle_service_status()
        
        return jsonify({
            'success': True,
            'summary': {
                'total_orchids': total_orchids,
                'orchids_with_trefle_data': orchids_with_trefle,
                'enrichment_percentage': round((orchids_with_trefle / total_orchids * 100) if total_orchids > 0 else 0, 2),
                'sample_enriched_orchids': sample_data
            },
            'service_status': service_status
        })
        
    except Exception as e:
        logger.error(f"Error getting Trefle ecosystem summary: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500

logger.info("🌿 GBIF Ecosystem API Integration routes registered successfully")

# ==============================
# AI RESEARCH API ENDPOINTS
# ==============================
# Import OrchidAIResearchHub service
try:
    from orchid_ai_research_hub import (
        OrchidAIResearchHub, QueryIntentType, ResearchQuery, ResearchResponse,
        ConfidenceLevel, get_research_hub
    )
    AI_RESEARCH_AVAILABLE = True
    logger.info("🤖 OrchidAI Research Hub imported successfully")
except ImportError as e:
    logger.warning(f"⚠️ OrchidAI Research Hub not available: {e}")
    AI_RESEARCH_AVAILABLE = False
    
    # Create fallback stubs for missing AI symbols
    from enum import Enum
    from dataclasses import dataclass
    from typing import Dict, List, Optional, Any
    
    class QueryIntentType(Enum):
        SPECIES_IDENTIFICATION = "species_identification"
        CULTIVATION_ADVICE = "cultivation_advice"
        RESEARCH_DISCOVERY = "research_discovery"
        CITATION_GENERATION = "citation_generation"
        DATABASE_QUERY = "database_query"
        IMAGE_ANALYSIS = "image_analysis"
        ECOSYSTEM_ANALYSIS = "ecosystem_analysis"
        COMPARATIVE_ANALYSIS = "comparative_analysis"
    
    class ConfidenceLevel(Enum):
        VERY_HIGH = "very_high"
        HIGH = "high"
        MEDIUM = "medium"
        LOW = "low"
        VERY_LOW = "very_low"
    
    @dataclass
    class ResearchQuery:
        query_id: str
        intent: QueryIntentType
        query_text: str
        user_id: Optional[str] = None
        image_data: Optional[Dict] = None
        context_filters: Optional[Dict] = None
        session_id: Optional[str] = None
        timestamp: Optional[Any] = None
    
    @dataclass
    class ResearchResponse:
        query_id: str
        response_type: str
        primary_result: Dict[str, Any]
        confidence_score: float
        confidence_level: ConfidenceLevel
        supporting_evidence: List[Dict[str, Any]]
        alternative_suggestions: List[Dict[str, Any]]
        source_citations: List[Dict[str, Any]]
        research_trail: List[str]
        session_context: Dict[str, Any]
        processing_time: float
        ai_models_used: List[str]
        database_records_referenced: int
    
    def get_research_hub():
        """Fallback function when research hub is not available"""
        return None

# Import CSRF for API exemption
try:
    from flask_wtf.csrf import exempt
    CSRF_AVAILABLE = True
except ImportError:
    CSRF_AVAILABLE = False
    def exempt(f):
        return f

@app.route('/api/ai-research/query', methods=['POST'])
@exempt
def ai_research_query():
    """
    Main research query endpoint for AI-powered orchid research
    Supports all query types with intelligent routing and context management
    """
    start_time = time.time()
    
    if not AI_RESEARCH_AVAILABLE:
        return jsonify({
            'success': False,
            'error': 'AI Research Hub not available',
            'error_type': 'service_unavailable'
        }), 503
    
    try:
        data = request.get_json()
        
        if not data or 'query' not in data:
            return jsonify({
                'success': False,
                'error': 'Query text is required',
                'error_type': 'validation_error'
            }), 400
        
        # Validate and sanitize input
        query_text = str(data['query']).strip()
        if len(query_text) < 3:
            return jsonify({
                'success': False,
                'error': 'Query must be at least 3 characters long',
                'error_type': 'validation_error'
            }), 400
        
        if len(query_text) > 2000:
            return jsonify({
                'success': False,
                'error': 'Query must be less than 2000 characters',
                'error_type': 'validation_error'
            }), 400
        
        # Extract query parameters
        query_type = data.get('type', 'general')
        session_id = data.get('session_id') or str(uuid.uuid4())
        context = data.get('context', {})
        attachments = data.get('attachments', [])
        
        # Map query type to intent
        intent_mapping = {
            'identification': QueryIntentType.SPECIES_IDENTIFICATION,
            'cultivation': QueryIntentType.CULTIVATION_ADVICE,
            'research': QueryIntentType.RESEARCH_DISCOVERY,
            'citations': QueryIntentType.CITATION_GENERATION,
            'images': QueryIntentType.IMAGE_ANALYSIS,
            'ecosystem': QueryIntentType.ECOSYSTEM_ANALYSIS,
            'comparison': QueryIntentType.COMPARATIVE_ANALYSIS
        }
        intent = intent_mapping.get(query_type, QueryIntentType.DATABASE_QUERY)
        
        # Process image attachments if provided
        image_data = None
        if attachments:
            for attachment in attachments:
                if isinstance(attachment, dict) and 'image_data' in attachment:
                    image_data = attachment
                    break
        
        # Create research query
        research_query = ResearchQuery(
            query_id=str(uuid.uuid4()),
            intent=intent,
            query_text=query_text,
            user_id=session.get('user_id'),
            image_data=image_data,
            context_filters=context,
            session_id=session_id,
            timestamp=datetime.now()
        )
        
        # Process query through research hub
        research_hub = get_research_hub()
        response = research_hub.process_research_query(research_query)
        
        # Calculate processing time
        processing_time = time.time() - start_time
        
        # Format response according to API specification
        api_response = {
            'success': True,
            'query_analysis': {
                'intent': response.response_type,
                'confidence': response.confidence_score,
                'query_id': response.query_id
            },
            'ai_response': {
                'summary': response.primary_result.get('summary', ''),
                'detailed_answer': response.primary_result.get('detailed_answer', response.primary_result.get('analysis', '')),
                'confidence_score': response.confidence_level.value
            },
            'data_sources': {
                'orchid_records': response.supporting_evidence,
                'gbif_data': response.primary_result.get('gbif_data', {}),
                'literature': response.source_citations
            },
            'session_context': {
                'session_id': session_id,
                'conversation_length': len(response.session_context.get('query_history', []))
            },
            'processing_time': round(processing_time, 2),
            'research_trail': response.research_trail,
            'alternative_suggestions': response.alternative_suggestions,
            'ai_models_used': response.ai_models_used,
            'database_records_referenced': response.database_records_referenced
        }
        
        logger.info(f"🔬 AI Research query processed: {intent.value} (confidence: {response.confidence_score:.2f})")
        return jsonify(api_response), 200
        
    except Exception as e:
        logger.error(f"AI Research query error: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e),
            'error_type': 'processing_error',
            'processing_time': round(time.time() - start_time, 2)
        }), 500

@app.route('/api/ai-research/identify', methods=['POST'])
@exempt
def ai_research_identify():
    """
    Specialized species identification endpoint with image analysis support
    Optimized for orchid species identification using AI vision and database matching
    """
    start_time = time.time()
    
    if not AI_RESEARCH_AVAILABLE:
        return jsonify({
            'success': False,
            'error': 'AI Research Hub not available',
            'error_type': 'service_unavailable'
        }), 503
    
    try:
        data = request.get_json() or {}
        
        # Handle both text descriptions and image uploads
        query_text = data.get('query', data.get('description', ''))
        image_data = data.get('image_data') or data.get('image')
        
        if not query_text and not image_data:
            return jsonify({
                'success': False,
                'error': 'Either query text or image data is required for identification',
                'error_type': 'validation_error'
            }), 400
        
        # Default query for identification if only image provided
        if not query_text and image_data:
            query_text = "Please identify this orchid species from the provided image."
        
        session_id = data.get('session_id') or str(uuid.uuid4())
        
        # Create identification-specific query
        research_query = ResearchQuery(
            query_id=str(uuid.uuid4()),
            intent=QueryIntentType.SPECIES_IDENTIFICATION,
            query_text=query_text,
            user_id=session.get('user_id'),
            image_data=image_data,
            context_filters=data.get('context', {}),
            session_id=session_id,
            timestamp=datetime.now()
        )
        
        # Process through research hub
        research_hub = get_research_hub()
        response = research_hub.process_research_query(research_query)
        
        # Format identification-specific response
        api_response = {
            'success': True,
            'identification': {
                'primary_candidate': response.primary_result.get('primary_identification', {}),
                'alternatives': response.alternative_suggestions[:5],  # Top 5 alternatives
                'confidence_level': response.confidence_level.value,
                'confidence_score': response.confidence_score
            },
            'database_matches': response.supporting_evidence,
            'analysis_details': {
                'image_analysis': response.primary_result.get('image_analysis', {}),
                'morphological_features': response.primary_result.get('morphological_features', {}),
                'distinguishing_characteristics': response.primary_result.get('distinguishing_characteristics', [])
            },
            'session_context': {
                'session_id': session_id,
                'identification_history': response.session_context.get('identification_history', [])
            },
            'processing_time': round(time.time() - start_time, 2),
            'source_citations': response.source_citations,
            'research_trail': response.research_trail
        }
        
        logger.info(f"🔍 Species identification processed (confidence: {response.confidence_score:.2f})")
        return jsonify(api_response), 200
        
    except Exception as e:
        logger.error(f"Species identification error: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e),
            'error_type': 'identification_error',
            'processing_time': round(time.time() - start_time, 2)
        }), 500

@app.route('/api/ai-research/cultivation-advice', methods=['POST'])
@exempt
def ai_research_cultivation_advice():
    """
    Specialized cultivation advice endpoint with climate matching and Baker culture integration
    Provides comprehensive growing recommendations based on species and environmental factors
    """
    start_time = time.time()
    
    if not AI_RESEARCH_AVAILABLE:
        return jsonify({
            'success': False,
            'error': 'AI Research Hub not available',
            'error_type': 'service_unavailable'
        }), 503
    
    try:
        data = request.get_json() or {}
        
        query_text = data.get('query', '')
        orchid_species = data.get('species', '')
        location = data.get('location', '')
        experience_level = data.get('experience_level', 'intermediate')
        growing_environment = data.get('environment', 'indoor')
        
        if not query_text and not orchid_species:
            return jsonify({
                'success': False,
                'error': 'Either query text or orchid species is required for cultivation advice',
                'error_type': 'validation_error'
            }), 400
        
        # Enhance query with context if orchid species provided
        if orchid_species and not query_text:
            query_text = f"What are the optimal growing conditions and care requirements for {orchid_species}?"
        elif orchid_species:
            query_text += f" Specifically for {orchid_species}."
        
        # Add environmental context to query
        context_additions = []
        if location:
            context_additions.append(f"growing in {location}")
        if experience_level != 'intermediate':
            context_additions.append(f"for a {experience_level} grower")
        if growing_environment != 'indoor':
            context_additions.append(f"in a {growing_environment} environment")
        
        if context_additions:
            query_text += f" Consider {', '.join(context_additions)}."
        
        session_id = data.get('session_id') or str(uuid.uuid4())
        
        # Create cultivation-specific query
        research_query = ResearchQuery(
            query_id=str(uuid.uuid4()),
            intent=QueryIntentType.CULTIVATION_ADVICE,
            query_text=query_text,
            user_id=session.get('user_id'),
            image_data=data.get('image_data'),
            context_filters={
                'species': orchid_species,
                'location': location,
                'experience_level': experience_level,
                'environment': growing_environment
            },
            session_id=session_id,
            timestamp=datetime.now()
        )
        
        # Process through research hub
        research_hub = get_research_hub()
        response = research_hub.process_research_query(research_query)
        
        # Format cultivation-specific response
        api_response = {
            'success': True,
            'cultivation_advice': {
                'summary': response.primary_result.get('summary', ''),
                'detailed_recommendations': response.primary_result.get('recommendations', {}),
                'care_schedule': response.primary_result.get('care_schedule', {}),
                'environmental_requirements': response.primary_result.get('environmental_requirements', {}),
                'common_problems': response.primary_result.get('common_problems', []),
                'seasonal_care': response.primary_result.get('seasonal_care', {})
            },
            'baker_culture_sheets': response.primary_result.get('baker_culture_data', []),
            'climate_matching': response.primary_result.get('climate_analysis', {}),
            'experience_level_tips': response.primary_result.get('level_specific_tips', {}),
            'supporting_evidence': response.supporting_evidence,
            'session_context': {
                'session_id': session_id,
                'cultivation_history': response.session_context.get('cultivation_queries', [])
            },
            'processing_time': round(time.time() - start_time, 2),
            'confidence_score': response.confidence_score,
            'source_citations': response.source_citations
        }
        
        logger.info(f"🌱 Cultivation advice processed for: {orchid_species or 'general'}")
        return jsonify(api_response), 200
        
    except Exception as e:
        logger.error(f"Cultivation advice error: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e),
            'error_type': 'cultivation_error',
            'processing_time': round(time.time() - start_time, 2)
        }), 500

@app.route('/api/ai-research/research-insights', methods=['POST'])
@exempt
def ai_research_insights():
    """
    Academic research queries endpoint for taxonomic patterns and conservation insights
    Provides research-grade analysis for scientific and academic purposes
    """
    start_time = time.time()
    
    if not AI_RESEARCH_AVAILABLE:
        return jsonify({
            'success': False,
            'error': 'AI Research Hub not available',
            'error_type': 'service_unavailable'
        }), 503
    
    try:
        data = request.get_json() or {}
        
        query_text = data.get('query', '')
        research_type = data.get('research_type', 'general')
        taxonomic_focus = data.get('taxonomic_focus', '')
        geographic_scope = data.get('geographic_scope', '')
        time_period = data.get('time_period', '')
        
        if not query_text:
            return jsonify({
                'success': False,
                'error': 'Research query text is required',
                'error_type': 'validation_error'
            }), 400
        
        # Validate research type
        valid_research_types = [
            'general', 'taxonomic', 'conservation', 'ecological', 
            'evolutionary', 'biogeographic', 'phenological'
        ]
        if research_type not in valid_research_types:
            research_type = 'general'
        
        session_id = data.get('session_id') or str(uuid.uuid4())
        
        # Create research-specific query
        research_query = ResearchQuery(
            query_id=str(uuid.uuid4()),
            intent=QueryIntentType.RESEARCH_DISCOVERY,
            query_text=query_text,
            user_id=session.get('user_id'),
            image_data=data.get('image_data'),
            context_filters={
                'research_type': research_type,
                'taxonomic_focus': taxonomic_focus,
                'geographic_scope': geographic_scope,
                'time_period': time_period
            },
            session_id=session_id,
            timestamp=datetime.now()
        )
        
        # Process through research hub
        research_hub = get_research_hub()
        response = research_hub.process_research_query(research_query)
        
        # Format research-specific response
        api_response = {
            'success': True,
            'research_insights': {
                'summary': response.primary_result.get('summary', ''),
                'key_findings': response.primary_result.get('key_findings', []),
                'taxonomic_patterns': response.primary_result.get('taxonomic_patterns', {}),
                'conservation_status': response.primary_result.get('conservation_status', {}),
                'ecological_relationships': response.primary_result.get('ecological_relationships', {}),
                'research_gaps': response.primary_result.get('research_gaps', []),
                'methodology_suggestions': response.primary_result.get('methodology_suggestions', [])
            },
            'data_analysis': {
                'database_coverage': response.database_records_referenced,
                'geographic_distribution': response.primary_result.get('geographic_analysis', {}),
                'temporal_trends': response.primary_result.get('temporal_trends', {}),
                'statistical_summary': response.primary_result.get('statistics', {})
            },
            'supporting_evidence': response.supporting_evidence,
            'literature_review': response.source_citations,
            'session_context': {
                'session_id': session_id,
                'research_history': response.session_context.get('research_queries', [])
            },
            'processing_time': round(time.time() - start_time, 2),
            'confidence_assessment': {
                'overall_confidence': response.confidence_score,
                'data_quality': response.primary_result.get('data_quality', 'unknown'),
                'research_reliability': response.confidence_level.value
            }
        }
        
        logger.info(f"🔬 Research insights processed: {research_type}")
        return jsonify(api_response), 200
        
    except Exception as e:
        logger.error(f"Research insights error: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e),
            'error_type': 'research_error',
            'processing_time': round(time.time() - start_time, 2)
        }), 500

@app.route('/api/ai-research/citations', methods=['POST'])
@exempt
def ai_research_citations():
    """
    Citation generation endpoint with academic formatting support
    Generates properly formatted citations in multiple academic styles
    """
    start_time = time.time()
    
    if not AI_RESEARCH_AVAILABLE:
        return jsonify({
            'success': False,
            'error': 'AI Research Hub not available',
            'error_type': 'service_unavailable'
        }), 503
    
    try:
        data = request.get_json() or {}
        
        query_text = data.get('query', '')
        citation_style = data.get('style', 'APA').upper()
        source_types = data.get('source_types', ['journal', 'database', 'book'])
        include_doi = data.get('include_doi', True)
        max_citations = data.get('max_citations', 20)
        
        if not query_text:
            return jsonify({
                'success': False,
                'error': 'Citation query text is required',
                'error_type': 'validation_error'
            }), 400
        
        # Validate citation style
        valid_styles = ['APA', 'MLA', 'CHICAGO', 'VANCOUVER', 'IEEE']
        if citation_style not in valid_styles:
            citation_style = 'APA'
        
        # Validate max citations limit
        if max_citations > 50:
            max_citations = 50
        elif max_citations < 1:
            max_citations = 10
        
        session_id = data.get('session_id') or str(uuid.uuid4())
        
        # Create citation-specific query
        research_query = ResearchQuery(
            query_id=str(uuid.uuid4()),
            intent=QueryIntentType.CITATION_GENERATION,
            query_text=query_text,
            user_id=session.get('user_id'),
            image_data=None,
            context_filters={
                'citation_style': citation_style,
                'source_types': source_types,
                'include_doi': include_doi,
                'max_citations': max_citations
            },
            session_id=session_id,
            timestamp=datetime.now()
        )
        
        # Process through research hub
        research_hub = get_research_hub()
        response = research_hub.process_research_query(research_query)
        
        # Format citation-specific response
        api_response = {
            'success': True,
            'citations': {
                'formatted_citations': response.source_citations,
                'citation_style': citation_style,
                'total_found': len(response.source_citations),
                'bibliography': response.primary_result.get('bibliography', []),
                'in_text_citations': response.primary_result.get('in_text_citations', [])
            },
            'source_analysis': {
                'source_types_found': response.primary_result.get('source_types', {}),
                'publication_years': response.primary_result.get('publication_years', {}),
                'author_analysis': response.primary_result.get('author_analysis', {}),
                'journal_distribution': response.primary_result.get('journal_distribution', {})
            },
            'quality_assessment': {
                'peer_reviewed_count': response.primary_result.get('peer_reviewed_count', 0),
                'impact_factor_analysis': response.primary_result.get('impact_analysis', {}),
                'citation_completeness': response.primary_result.get('completeness_score', 0),
                'doi_coverage': response.primary_result.get('doi_coverage', 0)
            },
            'session_context': {
                'session_id': session_id,
                'citation_history': response.session_context.get('citation_queries', [])
            },
            'processing_time': round(time.time() - start_time, 2),
            'research_trail': response.research_trail
        }
        
        logger.info(f"📚 Citations generated: {len(response.source_citations)} in {citation_style} style")
        return jsonify(api_response), 200
        
    except Exception as e:
        logger.error(f"Citation generation error: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e),
            'error_type': 'citation_error',
            'processing_time': round(time.time() - start_time, 2)
        }), 500

@app.route('/api/ai-research/session/<session_id>/context', methods=['GET'])
def ai_research_session_context(session_id):
    """
    Session context management endpoint for conversation continuity
    Retrieves and manages multi-turn conversation context
    """
    start_time = time.time()
    
    if not AI_RESEARCH_AVAILABLE:
        return jsonify({
            'success': False,
            'error': 'AI Research Hub not available',
            'error_type': 'service_unavailable'
        }), 503
    
    try:
        if not session_id or len(session_id) < 10:
            return jsonify({
                'success': False,
                'error': 'Valid session ID is required',
                'error_type': 'validation_error'
            }), 400
        
        # Get session context from research hub
        research_hub = get_research_hub()
        context = research_hub._get_session_context(session_id)
        
        if not context:
            return jsonify({
                'success': False,
                'error': 'Session not found or expired',
                'error_type': 'session_not_found'
            }), 404
        
        # Format session context response
        api_response = {
            'success': True,
            'session_context': {
                'session_id': session_id,
                'created_at': context.get('created_at').isoformat() if context.get('created_at') else None,
                'last_activity': context.get('last_activity').isoformat() if context.get('last_activity') else None,
                'total_queries': context.get('total_queries', 0),
                'query_types': context.get('query_types', {}),
                'conversation_summary': context.get('conversation_summary', ''),
                'research_focus': context.get('research_focus', []),
                'active_topics': context.get('active_topics', [])
            },
            'query_history': [
                {
                    'query_id': q.get('query_id'),
                    'intent': q.get('intent'),
                    'timestamp': q.get('timestamp').isoformat() if q.get('timestamp') else None,
                    'confidence': q.get('confidence'),
                    'summary': q.get('summary', '')[:100] + '...' if len(q.get('summary', '')) > 100 else q.get('summary', '')
                }
                for q in context.get('query_history', [])[-10:]  # Last 10 queries
            ],
            'session_analytics': {
                'average_confidence': context.get('average_confidence', 0),
                'most_frequent_intent': context.get('most_frequent_intent', ''),
                'session_duration_minutes': context.get('session_duration_minutes', 0),
                'research_productivity_score': context.get('research_productivity_score', 0)
            },
            'processing_time': round(time.time() - start_time, 2)
        }
        
        logger.info(f"📋 Session context retrieved: {session_id}")
        return jsonify(api_response), 200
        
    except Exception as e:
        logger.error(f"Session context error: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e),
            'error_type': 'session_error',
            'processing_time': round(time.time() - start_time, 2)
        }), 500

@app.route('/api/ai-research/capabilities', methods=['GET'])
def ai_research_capabilities():
    """
    Research capabilities and system status endpoint
    Provides information about available features and system health
    """
    start_time = time.time()
    
    try:
        if not AI_RESEARCH_AVAILABLE:
            return jsonify({
                'success': False,
                'status': 'unavailable',
                'error': 'AI Research Hub not available',
                'capabilities': {
                    'query_types': [],
                    'ai_models': [],
                    'integrations': {},
                    'database_records': 0
                }
            }), 503
        
        # Get system status and capabilities
        research_hub = get_research_hub()
        
        # Check database connectivity
        try:
            orchid_count = OrchidRecord.query.count()
            taxonomy_count = OrchidTaxonomy.query.count()
            database_healthy = True
        except Exception as e:
            logger.warning(f"Database health check failed: {e}")
            orchid_count = 0
            taxonomy_count = 0
            database_healthy = False
        
        # Check AI model availability
        ai_models = ['gpt-4o']
        try:
            from orchid_ai_research_hub import anthropic_client
            if anthropic_client:
                ai_models.append('claude-sonnet-4')
        except:
            pass
        
        # System capabilities
        api_response = {
            'success': True,
            'status': 'operational',
            'capabilities': {
                'query_types': [
                    {
                        'type': 'identification',
                        'name': 'Species Identification',
                        'description': 'AI-powered orchid species identification from images and descriptions',
                        'supports_images': True,
                        'endpoint': '/api/ai-research/identify'
                    },
                    {
                        'type': 'cultivation',
                        'name': 'Cultivation Advice', 
                        'description': 'Comprehensive growing recommendations and care guidance',
                        'supports_images': False,
                        'endpoint': '/api/ai-research/cultivation-advice'
                    },
                    {
                        'type': 'research',
                        'name': 'Research Insights',
                        'description': 'Academic research analysis and taxonomic patterns',
                        'supports_images': False,
                        'endpoint': '/api/ai-research/research-insights'
                    },
                    {
                        'type': 'citations',
                        'name': 'Citation Generation',
                        'description': 'Academic citation formatting in multiple styles',
                        'supports_images': False,
                        'endpoint': '/api/ai-research/citations'
                    },
                    {
                        'type': 'general',
                        'name': 'General Research',
                        'description': 'Multi-purpose research queries with intelligent routing',
                        'supports_images': True,
                        'endpoint': '/api/ai-research/query'
                    }
                ],
                'ai_models': ai_models,
                'database_records': {
                    'orchid_records': orchid_count,
                    'taxonomy_entries': taxonomy_count,
                    'last_updated': datetime.now().isoformat(),
                    'database_healthy': database_healthy
                },
                'integrations': {
                    'gbif_ecosystem': True,
                    'baker_culture_sheets': True,
                    'image_analysis': True,
                    'species_identification': True,
                    'weather_data': True,
                    'geographic_mapping': True
                },
                'supported_formats': {
                    'input_images': ['jpg', 'jpeg', 'png', 'webp'],
                    'citation_styles': ['APA', 'MLA', 'Chicago', 'Vancouver', 'IEEE'],
                    'research_types': ['taxonomic', 'conservation', 'ecological', 'evolutionary']
                }
            },
            'usage_limits': {
                'max_query_length': 2000,
                'max_citations': 50,
                'session_timeout_hours': 24,
                'max_image_size_mb': 10
            },
            'version': '1.0.0',
            'last_updated': datetime.now().isoformat(),
            'processing_time': round(time.time() - start_time, 2)
        }
        
        logger.info("🚀 AI Research capabilities provided")
        return jsonify(api_response), 200
        
    except Exception as e:
        logger.error(f"Capabilities endpoint error: {str(e)}")
        return jsonify({
            'success': False,
            'status': 'error',
            'error': str(e),
            'error_type': 'capabilities_error',
            'processing_time': round(time.time() - start_time, 2)
        }), 500

# Register the research hub blueprint if it exists
try:
    from orchid_ai_research_hub import research_hub_bp
    app.register_blueprint(research_hub_bp, url_prefix='/ai-research-hub')
    logger.info("🔗 OrchidAI Research Hub blueprint registered at /ai-research-hub")
except ImportError:
    logger.warning("⚠️ OrchidAI Research Hub blueprint not available for registration")

logger.info("🤖 AI Research API endpoints registered successfully at /api/ai-research/*")

